#define module_name DRLACP
#define module_ident "V3.2"
/*
 *	D R L A C P
 *
 *  	Based on the DECNET Rlogin ACP program originally written in MACRO-32 
 *	by Anthony C. McCracken, Northern Arizona University, and on LOGGER.C,
 *	a simple program to demostrate using a Pseudo Terminal, written by 
 *	Forrest A. Kenney, Digital Equipment Corporation.
 *
 * This program is the non-transparent partner of the REMOTE TERMINAL program.
 * It uses DECNET non-transparent techniques to complete the logical link to
 * to its remote partner. If that is successful, it will then create a pseudo
 * terminal and establish an interactive login. This is like SET HOST and
 * RTPAD.
 *
 * V3.2	- HUnter Goatley, goathunter@goatley.com	10-JUL-2023 13:32
 *	For Alpha, I64, and X86, use LIB$xxxQGI() routines instead of the
 *	builtins previously used for Alpha and I64.
 *
 * V3.1 - Hunter Goatley, goathunter@WKUVX1.WKU.EDU	16-JUN-1994 09:21
 *	o Ported to run under OpenVMS AXP too (now DEC C compliant).  Fixed
 *	  a few minor bugs.
 *
 * V3.0 - John Delgado, j_delgado@farber.harvard.edu	25-MAY-1994
 *	- Rewrote it in the C language
 *
 * V2.1 - Hunter Goatley <goathunter@WKUVX1.BITNET>  	24-MAY-1993 21:36
 *	 Changed LIB$SPAWN call to $CREPRC so that a real, honest-to-goodness
 *	 interactive process is created.
 *
 * V2.0 - Major design change.  The program is now Event driven instead of
 *	 interupt driven.  All work is now done in the main process and ASTs
 *	 are used only to set event and status flags.  The main I/O is now
 *	 done almost entirely with QIOs instead of QIOWs.  The main program now
 *	 loops around handling different events.  The beginning of the loop is
 *	 is a $WAITFR that waits on an event flag.  The event flag is set by
 *	 an AST routine declared by one of 4 QIOs or a LIB$SPAWN.  The AST
 *	 routine also sets an appropriate status bit in a status longword.
 *	 The loop then immediately clears the event flag and then queries each
 *	 bit in the status longword and calls the appropriate handler for each
 *	 set bit.  These handlers clear the bit and issue appropriate QIOs.
 *
 * V1.2 - Added a CHMK  #0  instruction to the beginning of each AST routine to
 *	 immediately dismiss the AST.  This allows other AST to interupt an
 *	 in progress AST.  This helps prevent data overruns that are caused by
 *	 a QIOW executing at AST level preventing other ASTs from doing their
 *	 jobs.
 *
 * V1.1 - Added code to accept terminal characteristics and propogate these
 *	 to the TWAnnnn:.  10-JAN-1992	 ACM
 *
 * V1.0 - Original version written. 11-NOV-91   ACM
 *
 */

/* Include various necessary description files */

#include	"local.h"

#include	<accdef.h>
#include	<descrip.h>
#include	<dvidef.h>
#include	<iodef.h>
#include	<libdef.h>
#include	<prcdef.h>
#include	<prvdef.h>
#include	<rms.h>
#include	<ssdef.h>
#include	<stdio.h>
#include	<ttdef.h>
#include	<tt2def.h>
#include	<lnmdef.h>
#include	<starlet.h>
#include	<lib$routines.h>
#include	"ptddef.h"

#define QI_NORMAL	(0)		/* OK, more than 1 entry on queue    */
#define QI_SECINTFAI	(1)		/* Secondary interlock failed (SNH)  */
#define QI_QUEONEENT	(2)		/* OK, is/was only entry on queue    */
#define QI_QUEWASEMP	(3)		/* Queue was empty                   */

#define DRLACP_ACCOUNTING "DRLACP_ACCOUNTING"


/* Forward routine references */

void initialization (void);
void create_pseudo_terminal (struct mem_region *);
void setup_net (void);
int allocate_io_buffer (struct io_buff **);
void bell_ast (void);
void free_io_buffer (struct io_buff *);
void ft_echo_ast (struct io_buff *);
void ft_read_ast (struct io_buff *);
void net_read_ast (void);
void set_line_ast (void);
void subprocess_exit (void);
void net_output_ast (struct io_buff *);
void xoff_ast (void);
void xon_ast (void);


/* External routine references */

extern void open_log_file (char *);
extern void log_printf ();
extern void close_log_file (void);


/* Global Variables*/

static unsigned short int ft_chan;
static unsigned short int term_mbx_chan;
static unsigned short int net_chan;
static int read_stopped = FALSE;

static struct dev_char starting_chars;
static struct sense_iosb starting_iosb;
static struct io_buff *net_r_buff;

#if defined(__DECC) || defined (__DECCXX)
#pragma __nostandard	/* _align is non-standard */
#else
#pragma nostandard
#endif

static struct q_head _align (QUADWORD) buffer_queue = {0, 0};

#if defined(__DECC) || defined (__DECCXX)
#pragma __standard
#else
#pragma standard
#endif

static unsigned char term_mbx_buff[ACC$K_TERMLEN];	/* Mailbox message */


/*
 * 	i n i t i a l i z a t i o n
 *
 * This routine sets the terminal characteristics, creates the pseudo 
 * terminal, starts up the subprocess, and accepts up the network connection.
 * If any of these steps fails, it signals the failure.
 */
void
initialization (void)
{
  static $DESCRIPTOR (net_name, "_NET:");
  static $DESCRIPTOR (netbox, "NT2TBOX");
  static $DESCRIPTOR (lnam, "SYS$NET");
  static $DESCRIPTOR (ltab, "LNM$PROCESS_TABLE");

  int loop;
  int status;
  unsigned short nmchan;
  struct mem_region ret_addr;

  static char ncb[256];
  static $DESCRIPTOR (ncbdsc, ncb);
  int ncb_retlen;

  struct { struct item field[1]; int terminator; } trn_itm_list;

  trn_itm_list.field[0].buflen	= sizeof(ncb);
  trn_itm_list.field[0].code	= LNM$_STRING;
  trn_itm_list.field[0].bufadr	= ncb;
  trn_itm_list.field[0].retlen	= &ncb_retlen;
  trn_itm_list.terminator	= 0;

  chk (sys$expreg (IO_BUFFERS, &ret_addr, 0, 0));
  net_r_buff = (struct io_buff *) ((char *) ret_addr.end - PAGE);
  for (loop = 0; loop < IO_BUFFERS - 1; loop++) {
    free_io_buffer ((struct io_buff *) ((char *) ret_addr.start + loop * PAGE_SIZE));
  }

/*
 * This part translates the logical name SYS$NET to retrieve the
 * Network Connect Block. It then assigns channels to network device and
 * Attempts to complete the logical link (IO$_ACCESS). It reads the 
 * terminal characteristics being sent over by DRLOGIN; it then calls
 * to create the pseudo terminal and to create detach interactive process
 */

  chk (sys$trnlnm (0, &ltab, &lnam, 0, &trn_itm_list));
  ncbdsc.dsc$w_length = ncb_retlen;

  chk (sys$crembx (0, &nmchan, 0, 0, 0, 0, &netbox, 0));

  chk (sys$assign (&net_name, &net_chan, 0, &netbox, 0));
  chk (sys$qiow (0, net_chan, IO$_ACCESS, 0, 0, 0,
		 0, &ncbdsc, 0, 0, 0, 0));
  status = sys$qiow (0, net_chan, IO$_READVBLK, &starting_iosb, 0, 0,
		     &starting_chars, sizeof (starting_chars), 0, 0, 0, 0);
  if (status & SS$_NORMAL)
    status = starting_iosb.status;
  chk (status);

  create_pseudo_terminal (&ret_addr);
  setup_net ();
}

/*
 *	e x i t _ h a n d l e r
 */
exit_handler (unsigned long *condition)
{
  static char *intmsg = "TERM";

  ptd$cancel (ft_chan);
  sys$cancel (net_chan);
  ptd$delete (ft_chan);
  sys$qiow (0, net_chan, IO$_WRITEVBLK | IO$M_INTERRUPT, 0, 0, 0,
	    intmsg, 4, 0, 0, 0, 0);
  sys$qiow (0, net_chan, IO$_DEACCESS | IO$M_SYNCH, 0, 0, 0,
	    0, 0, 0, 0, 0, 0);
  sys$dassgn (net_chan);
  return SS$_NORMAL;
}


/*
 *	s e t u p _ n e t
 *
 * Create subprocess and issues the initial read to the network
 */
void
setup_net (void)
{
  char devnam_buff[30];
  struct dsc$descriptor_s devnam;

  $DESCRIPTOR (loginout, "SYS$SYSTEM:LOGINOUT.EXE");
  int devnam_retlen;
  static int term_mbx;		/* The mailbox unit # */
  int status;

  INIT_ITMLST (tmbx_item_list, 1) = {
    { sizeof (int), DVI$_UNIT, (char *) &term_mbx, 0, },
    0,
  };

  struct { struct item field[1]; int terminator; } dev_item_list;

  devnam.dsc$w_length	= 0;
  devnam.dsc$b_dtype	= DSC$K_DTYPE_T;
  devnam.dsc$b_class	= DSC$K_CLASS_S;
  devnam.dsc$a_pointer	= devnam_buff;

  dev_item_list.field[0].buflen	= sizeof(devnam_buff);
  dev_item_list.field[0].code	= DVI$_DEVNAM;
  dev_item_list.field[0].bufadr	= devnam_buff;
  dev_item_list.field[0].retlen	= &devnam_retlen;
  dev_item_list.terminator	= 0;

  chk (sys$getdviw (0, ft_chan, 0, &dev_item_list, 0, 0, 0, 0));

  devnam.dsc$w_length = devnam_retlen;

/*
 * Create a new interactive process.  A termination mailbox is created so 
 * that DRLACP will know when the interactive process goes away.
 */

  chk (sys$crembx (0, &term_mbx_chan, ACC$K_TERMLEN, 0, 0, 0, 0, 0));
  chk (sys$getdviw (0, term_mbx_chan, 0, &tmbx_item_list, 0, 0, 0, 0));
  chk (sys$qio (0, term_mbx_chan, IO$_READVBLK, 0, subprocess_exit, 0,
		&term_mbx_buff, ACC$K_TERMLEN, 0, 0, 0, 0));
  chk (sys$creprc (0, &loginout, &devnam, &devnam, 0, 0, 0, 0, 4, 0,
		term_mbx, PRC$M_DETACH | PRC$M_INTER | PRC$M_NOPASSWORD, 0));
  chk (sys$qio (0, net_chan, IO$_READVBLK,
		&net_r_buff->status, net_read_ast, 0,
		&net_r_buff->data[0], 1, 0, 0, 0, 0));
}


/*
 *	c r e a t e _ p s e u d o _ t e r m i n a l
 *
 * Create the pseudo terminal and issue initial read to the device
 */
void
create_pseudo_terminal (struct mem_region * ret_addr)
{
  int status;

  struct io_buff *read_buffer;

  chk (ptd$create (&ft_chan, 0, &starting_chars, sizeof (starting_chars),
		   0, 0, 0, ret_addr));
  chk (ptd$set_event_notification (ft_chan, bell_ast, 0, 0, PTD$C_SEND_BELL));
  chk (ptd$set_event_notification (ft_chan, xoff_ast, 0, 0, PTD$C_SEND_XOFF));
  chk (ptd$set_event_notification (ft_chan, xon_ast, 0, 0, PTD$C_SEND_XON));

  if ((status = allocate_io_buffer (&read_buffer)) != LIB$_QUEWASEMP) {
    status = ptd$read (0, ft_chan, ft_read_ast, read_buffer,
		       &read_buffer->io_status, CHAR_BUF_SIZE);
  }
}


/*
 *	n e t _ r e a d _ a s t
 */
void
net_read_ast (void)
{
  int status;
  int char_cnt;

  struct io_buff *echo_buff;

  chk (net_r_buff->status);
  if ((status = allocate_io_buffer (&echo_buff)) != LIB$_QUEWASEMP) {
    chk (ptd$write (ft_chan, ft_echo_ast, echo_buff,
		    &net_r_buff->io_status, net_r_buff->byte_cnt,
		    &echo_buff->io_status, CHAR_BUF_SIZE));
  } else {
    chk (ptd$write (ft_chan, 0, echo_buff, &net_r_buff->io_status,
		    net_r_buff->byte_cnt, 0, 0));
  }
  if ((net_r_buff->io_status & SS$_NORMAL) ||
      (net_r_buff->io_status == SS$_DATAOVERUN)) {
    chk (sys$qio (0, net_chan, IO$_READVBLK,
		  &net_r_buff->status, net_read_ast, 0,
		  &net_r_buff->data[0], 1, 0, 0, 0, 0));
  }
}


/* 
 *	n e t _ o u t p u t _ a s t
 */
void
net_output_ast (struct io_buff *buff_addr)
{
  chk (buff_addr->status);
  free_io_buffer (buff_addr);
}


/*
 *	f t _ r e a d _ a s t
 *
 * This routine is called when a pseudo terminal read request completes.
 * It will write the buffer to the network and attempt to start another read
 * from the pseudo terminal.
 */
void
ft_read_ast (struct io_buff *buff_addr)
{
  int status;
  struct io_buff *next_addr;

  chk (buff_addr->io_status);
  chk (sys$qio (0, net_chan, IO$_WRITEVBLK, &buff_addr->status,
		net_output_ast, buff_addr,
		&buff_addr->data[0], buff_addr->io_byte_cnt, 0, 0, 0, 0));
  if ((status = allocate_io_buffer (&next_addr)) != LIB$_QUEWASEMP) {
    chk (ptd$read (0, ft_chan, ft_read_ast, next_addr,
		   &next_addr->io_status, CHAR_BUF_SIZE));
  } else
    read_stopped = TRUE;
}


/*
 *	f t _ e c h o _ a s t
 *
 * This routine is called if a write to the pseudo terminal used an ECHO
 * buffer.  If any data was echoed it needs to write the output to the
 * network. If no data was echoed then the I/O buffer needs to be freed 
 * so it can be used later.
 */
void
ft_echo_ast (struct io_buff *buff_addr)
{
  int status;

  if (buff_addr->io_byte_cnt != 0) {
    chk (sys$qio (0, net_chan, IO$_WRITEVBLK, &buff_addr->status,
		  net_output_ast, buff_addr, 
		  &buff_addr->data[0], buff_addr->io_byte_cnt, 0, 0, 0, 0));
  } else {
    free_io_buffer (buff_addr);
  }
}


/*
 *	a l l o c a t e _ i o _ b u f f e r
 *
 * This routine is used to get a free I/O buffer from the queue of
 * available I/O buffers.
 */
int
allocate_io_buffer (struct io_buff **buff_addr)
{
  int status;

#if defined(__DECC) || defined(__DECCXX)
  switch (lib$remqhi (&buffer_queue, (void *) buff_addr)) {
  case LIB$_SECINTFAI : lib$stop (LIB$_SECINTFAI);
  case LIB$_QUEWASEMP : return (LIB$_QUEWASEMP);
#else /* VAX C doesn't want the cast */
  switch (_REMQHI (&buffer_queue, buff_addr)) {
  case QI_SECINTFAI : lib$stop (LIB$_SECINTFAI);
  case QI_QUEWASEMP : return (LIB$_QUEWASEMP);
#endif
  }
  return (SS$_NORMAL);
}

/*
 *	f r e e _ i o _ b u f f e r
 *
 * This routine will place a free I/O buffer on the queue of available I/O
 * buffers.  It has the additional responsibility for restarting reads from the
 * pseudo terminal if they were stopped.  This routine disables AST delivery
 * while running to synchronize reading and resetting the read stopped flag.
 * This is a big hammer approach but is the easiest way to do it.
 *
 */
void
free_io_buffer (struct io_buff *buff_addr)
{
  int ast_stat;
  int status;

  ast_stat = sys$setast (0);
  if (!read_stopped) {
    lib$insqhi (buff_addr, &buffer_queue);
  } else {
    chk (ptd$read (0, ft_chan, ft_read_ast, buff_addr,
		   &buff_addr->io_status, CHAR_BUF_SIZE));
    read_stopped = FALSE;
  }
  if (ast_stat == SS$_WASSET)
    ast_stat = sys$setast (1);
}


/* 
 *	t i m e s t r
 *
 * Returns a pointer to a datestring 
 */
char *timestr (int atime[2])
{
  $DYNDESCRIPTOR(timbuf_d);

  if (!(lib$sys_asctim (&timbuf_d.dsc$w_length, &timbuf_d, atime, 0) & 1))
     return ("");
  *(timbuf_d.dsc$a_pointer + timbuf_d.dsc$w_length) = '\0';
  return (timbuf_d.dsc$a_pointer);
}

/*
 * 	s u b p r o c e s s _ e x i t
 *
 * This routine will be called when the subprocess has completed and
 * exited. It attempts to write an entry in the accounting log file
 */
void
subprocess_exit (void)
{
  unsigned int elapse[2];
  struct accdef *p;

  p = (struct accdef *)&term_mbx_buff;
  lib$sub_times (p->acc$q_termtime, p->acc$q_login, &elapse);
  open_log_file (DRLACP_ACCOUNTING);
  log_printf ("%s %12.12s %08x %08x %6d %6d %6d %s",
	      timestr ((int *) p->acc$q_login), p->acc$t_username, p->acc$l_finalsts, 
	      p->acc$l_pid, p->acc$l_biocnt, p->acc$l_diocnt,
	      p->acc$l_pageflts, timestr ((int *) &elapse));
  close_log_file ();
  sys$exit (SS$_NORMAL);
}

/*
 *	x o n _ a s t
 *
 * This routine is called when the pseudo terminal driver wants to signal
 * that it is ready to accept input. The routine will attempt to send an XON 
 * character to the network. This is done on a best effort basis - if it fails
 * it will not be retried.  This is done by sending XON DC1 to the network 
 * using SYS$QIO.
 */
void
xon_ast (void)
{
  char dc1 = XON;

  sys$qio (0, net_chan, IO$_WRITEVBLK, 0, 0, 0, &dc1, 1, 0, 0, 0, 0);
}

/*
 *	b e l l _ a s t
 *
 * This routine is called when the pseudo terminal driver wants to warn the
 * user to stop sending network data.  The routine will attempt to ring the
 * terminals bell.  This is done on a best effort basis - if it fails it will
 * not be retried.  This is done by send the BELL character to the terminal
 * using SYS$QIO.
 */
void
bell_ast (void)
{
  char bell = BELL;

  sys$qio (0, net_chan, IO$_WRITEVBLK, 0, 0, 0, &bell, 1, 0, 0, 0, 0);
}

/*
 *	x o f f _ a s t
 *
 * This routine is called when the pseudo terminal driver wants to signal
 * that it wants to stop accepting  keyboard input. The routine will attempt 
 * to send an XOFF character to the terminal.   This is done on a best effort
 * basis - if it fails it will not be retried.  This is done by sending XOFF
 * <DC3> to the terminal using SYS$QIO.
 */
void
xoff_ast (void)
{
  char dc3 = XOFF;

  sys$qio (0, net_chan, IO$_WRITEVBLK, 0, 0, 0, &dc3, 1, 0, 0, 0, 0);
}


/*
 *	m a i n
 */
main ()
{
  int buffer_pos;
  int got_buf_status;
  int status;
  static int exh_condition;
  struct exit_handler_block {
	struct exit_handler_block *flink;
	int (*exit_routine)();
	int arg_count;
	int *status_address;
	int exit_status;
  } exhblk;

  quadword privilege = { PRV$M_SYSPRV, 0 };

  /*
   * Disable the installed privilege
   * We only need SYSPRV to write to DRLACP_ACCOUNTING log file
   */
  sys$setprv (0, privilege, 0, 0);  	/* Disable SYSPRV */
  initialization ();

  exhblk.flink = 0;
  exhblk.exit_routine = exit_handler;
  exhblk.arg_count = 1;
  exhblk.status_address = &exh_condition;
  exhblk.exit_status = 0;

  chk (sys$dclexh(&exhblk));

  sys$hiber ();
}
