/******************************************************************************

				     PIPE.C

 ******************************************************************************/

/*

				      PIPE
				an X/Motif game

				 Michael Muller
				  Version 2.2

    This code has been tested and should work on ULTRIX, VMS, color, and 
    black & white systems.  Please report any problems.

    Feel free to reuse, distribute (as long as you don't profit), 
    or enhance this program.  Tell me if you do!

    Disclaimer: This was my first program using X *or* Motif *or* 
    an event-driven toolkit *or* C.  I learned a lot, but I now know 
    that I did many things the hard way.  I've since gone back and fixed 
    some of these oddities, but some are so entangled in the code 
    that I don't have the time or patience to work them out.  For 
    exmaple, I used global variables instead of passing values to 
    callbacks.  This can be avoided, but it's a lot of work because 
    Motif callbacks take only one parameter.

    History:
    01-Oct-1990	    Creation
    18-Jun-1991	    Version 1.0 (first stable version)
    08-Jul-1991	    Version 2.0 (major enhancements)
    19-Jul-1991	    Version 2.1 (bug fixes)
    08-Feb-1992     Version 2.2 (small clean-ups for CD distribution)

    Done since V1.0:
	o Reservoirs.
	o Bonus pipes.
        o Varying board width and height.
	o Fast flow.
	o Add a fluid flow countdown to the status panel.
	o Created a speedometer for the fluid.
	o Enable starting from a certain level.
	o Implement abort game.
	o On-line help.
	o High Scores maintained.
	o Weight the likelihood of each pipe appearing for each level.
	o Clear the preview window before each game.
        o The screen no longer flashes when the drawing area is [de]sensitized.
        o RNG without modulus (doesn't seem to help, but thanks to Dave Miller)
	o Change "bummer." to "psyche!" for level over.
	o Add application icon (thanks to Dick Schoeller).
	o You no longer lose your head start if you pause before flow begins.
	o Unix-compatible (thanks to Paul Douglas).
	o Stripes now visible in B&W
	o B&W pixels returned correctly (thanks to JP Tessier)
	o A number of unix bugs fixed (thanks to Marc Evans)
	o Level increments *after* Psyche! is pressed.
	o Tiles can't be placed out of bounds.
        o One UIL file for icons -- b&w uses same as color!

    XUI window manager incompatibilties:
        o The playingArea displays wherever it feels like. (XUI only)
	o The pause menu option doesn't iconize. (XUI only)

    Bugs:
	o The preview window is wider than the tiles. (used to be XUI only)

    Things to do/suggestions:
	o Make fast flow flow faster (calling w/out timeouts doesn't work).
	o Implement wrap-around edges in certain places for some levels.
	o What about the bonus Tetris-like level?  (LucasFilms did it...)
	o Create application resources for use with resource file
	o Finish pipe ("drain")
	o Two flows
	o T pipes
	o Customization (preferences)
	o Level info read from file for customization
	o Make erasing quicker on the starting levels
	o Allow customization of pipe colors

*/


/******************************************************************************

                                 Header Files

 ******************************************************************************/

#include <Mrm/MrmAppl.h>
#include <X11/Intrinsic.h>
#include <Xm/Xm.h>
#include <X11/Shell.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#ifndef VMS
#include <Xm/DialogS.h> /* for unix */
#include <Xm/ToggleBG.h> /* for unix */
#include <unistd.h> /* for unix (cuserid) */
#endif


/******************************************************************************

			     file names & locations

 ******************************************************************************/

#define PIPE_WIDGETS "pipe_widgets.uid"
#define PIPE_ICONS   "pipe_icons.uid"

#ifndef UID_LOC
#define UID_LOC ""
#endif

#ifndef HELP_FILE
#define HELP_FILE "pipe_help"
#endif

#ifndef HELP_LOC
#define HELP_LOC ""
#endif

#ifndef HIGH_SCORE_FILE
#define HIGH_SCORE_FILE "pipe_scores"
#endif

#ifndef HIGH_SCORE_LOC
#define HIGH_SCORE_LOC ""
#endif


/******************************************************************************

                               UIL declarations

 ******************************************************************************/

void RegisterWidgetCB();
void PlayingAreaInputCB();
void ExposeCB();
void LevelOverCB();
void PauseGameCB();
void HelpCB();
void HighScoreCB();
void AbortGameCB();
void StartGameCB();
void QuitGameCB();
void FastFlowCB();

static MRMRegisterArg G_CBroutines[] = {
  { "PlayingAreaInputCB", (caddr_t)PlayingAreaInputCB},
  { "ExposeCB", (caddr_t)ExposeCB},
  { "RegisterWidgetCB", (caddr_t)RegisterWidgetCB},
  { "LevelOverCB", (caddr_t)LevelOverCB},
  { "PauseGameCB", (caddr_t)PauseGameCB},
  { "AbortGameCB", (caddr_t)AbortGameCB},
  { "StartGameCB", (caddr_t)StartGameCB},
  { "QuitGameCB", (caddr_t)QuitGameCB},
  { "HelpCB", (caddr_t)HelpCB},
  { "HighScoreCB", (caddr_t)HighScoreCB},
  { "FastFlowCB", (caddr_t)FastFlowCB}
};

static int   G_num_CBroutines = (sizeof G_CBroutines / sizeof G_CBroutines[0]);


/******************************************************************************

                                    Macros

 ******************************************************************************/

#ifdef VMS
#  define RAND		rand()
#  define SEED_RAND	srand ((int) time(NULL))
#  define MAX_RAND      RAND_MAX
#else
#  define RAND		random()
#  define SEED_RAND	srandom ((unsigned) time(NULL))
#  define MAX_RAND      0x7FFFFFFF
#endif


/******************************************************************************

                                  Constants

 ******************************************************************************/

#define MENU_BAR_HEIGHT		35 /* KLUDGE */

#define ERASE_INCREMENT		4
#define ERASE_INTERVAL		200

#define RESERVOIR_FILL_INTERVAL	500
#define RESERVOIR_RADIUS	14

#define APP_X			200
#define APP_Y			200

#define MAX_ARGS		10

#define PIPETILE_SIZE		50
#define DIAMETER		6
#define SHORT_LEN		((PIPETILE_SIZE / 2) - (DIAMETER / 2))
#define LONG_LEN		((PIPETILE_SIZE / 2) + (DIAMETER / 2))

#define ERASED                  -1
#define PLUS			0
#define UPPER_LEFT		1
#define UPPER_RIGHT		2
#define LOWER_LEFT		3
#define LOWER_RIGHT		4
#define HORIZONTAL		5
#define VERTICAL		6
#define HORIZONTAL_LEFT		7
#define HORIZONTAL_RIGHT	8
#define VERTICAL_UP  		9
#define VERTICAL_DOWN  		10
#define START_UP		11
#define START_DOWN		12
#define START_LEFT		13
#define START_RIGHT		14
#define EMPTY                   15
#define WASTED                  16
#define OBSTACLE                17
#define HORIZONTAL_BONUS	18
#define VERTICAL_BONUS		19
#define HORIZONTAL_RESERVOIR	20
#define VERTICAL_RESERVOIR	21

#define MAX_PIPE		21
#define NUM_PIPE_TYPES		22

#define RIGHT			0
#define LEFT			1
#define UP			2
#define DOWN			3
#define ERROR			-1

#define MAX_TILES_SIDETOSIDE	15
#define MAX_TILES_UPANDDOWN	15

#define NUM_TILES_SIDETOSIDE	G_levelInfo[G_level].tilesAcross
#define NUM_TILES_UPANDDOWN	G_levelInfo[G_level].tilesUpNdown

#define PLAYINGAREA_HEIGHT	(NUM_TILES_UPANDDOWN * PIPETILE_SIZE)
#define PLAYINGAREA_WIDTH	(NUM_TILES_SIDETOSIDE * PIPETILE_SIZE)

#define STATUSAREA_WIDTH	200
#define STATUSAREA_HEIGHT	200

#define NUM_TILES_PREVIEWED	4

#define PREVIEWAREA_HEIGHT	(NUM_TILES_PREVIEWED * PIPETILE_SIZE)
#define PREVIEWAREA_WIDTH	PIPETILE_SIZE

#define FIVE_CROSS_SCORE	5000
#define BONUS_PIPE_SCORE	1000
#define CROSS_SCORE		500
#define NEW_TILE_SCORE		50
#define FF_MULTIPLIER           (G_fastFlow ? 2 : 1)
#define STARTING_EARLY_BONUS	100

#define BETWEEN_GAMES		1
#define BETWEEN_LEVELS		2
#define PLAYING			3
#define ERASING			4

#define FASTEST_FLUID_SPEED	10

char *LO_DIALOG_STR = "%s\n%s\nBonus for %d crosses: %d\nPenalty for %d unfilled pipes: %d";


/******************************************************************************

                              Type Declarations

 ******************************************************************************/

typedef struct {
  int    flow[4];
  Pixmap pixmap;
} PipeInfoRec;

typedef struct {
  char *description;
  int   headStart;
  int   fluidSpeed;
  int   numPipesToFill;
  int   numObstacles;
  int   numBonusPipes;
  int   numReservoirs;
  int   numHorizWraps;
  int   numVertWraps;
  int   tilesAcross;
  int   tilesUpNdown;
  int   pipeFrequency[NUM_PIPE_TYPES];
} LevelInfoRec;

typedef struct {
  int xTile,yTile;
  int Xpos,Ypos;
  int direction;
} FluidInfoRec;

typedef struct {
  int permanent;
  int flooded;
  int type;
} PipeLayoutRec;

typedef struct {
  int step;
  int xTile;
  int yTile;
} EraseInfo;

typedef struct SCORE_NODE {
  struct SCORE_NODE *next;
  char               name[20];
  int                score;
} ScoreNode,*ScoreNodePtr;


/******************************************************************************

                         Global Variable Declarations

 ******************************************************************************/

static Widget        G_appShell;

static Widget        G_statusShell;

static XtAppContext  G_appContext;

static GC            G_generic_GC;

static char         *G_username;

static Widget        G_main_wgt;
static Widget        G_levelOver_wgt;
static Widget        G_status_wgt;
static Widget        G_menuBar_wgt;
static Widget        G_startLevel_wgt;
static Widget        G_loButton_wgt;

static Widget        G_highScore_wgt;
static Widget        G_hsText_wgt;
static Widget        G_helpWindow_wgt;
static Widget        G_helpText_wgt;

static int           G_fastFlow = 0;

static Widget        G_playingArea_wgt;		/* playing area objects */
static Pixmap        G_playingAreaShadow;

static Widget        G_preview_wgt;		/* preview area objects */
static Pixmap        G_previewShadow;

static Widget        G_score_wgt;		/* label widgets */
static Widget        G_countdown_wgt;
static Widget        G_flowometer_wgt;
static Widget        G_crosses_wgt;
static Widget        G_pipes_wgt;
static Widget        G_pipesNeeded_wgt;
static Widget        G_level_wgt;
static Widget        G_levelOverLabel_wgt;

static Widget        G_fluid_wgt;

static Widget        G_start_wgt;		/* menu buttons */
static Widget        G_abort_wgt;
static Widget        G_pause_wgt;
static Widget        G_help_wgt;

static PipeInfoRec   G_pipeInfo[NUM_PIPE_TYPES];
static PipeLayoutRec G_pipeLayout[MAX_TILES_SIDETOSIDE][MAX_TILES_UPANDDOWN];
static int           G_previewLayout[NUM_TILES_PREVIEWED];

static Display      *G_display;
static Screen       *G_screen;

static FluidInfoRec  G_fluidInfo;
static Pixel         G_fluid_pxl;

static Pixel         G_replace_pxl;

static int           G_horizWarp[MAX_TILES_UPANDDOWN];
static int           G_vertWarp[MAX_TILES_SIDETOSIDE];

static LevelInfoRec *G_levelInfo;

static int           G_topLevel;

static XtIntervalId  G_eraseTimeOut = 0;
static XtIntervalId  G_countdownTimeOut = 0;
static XtIntervalId  G_fillresTimeOut = 0;
static XtIntervalId  G_fluidTimeOut = 0;

static int           G_pipesFilled;
static int           G_startingLevel = 0;
static int           G_level;
static int           G_score;
static int           G_crosses;

static int           G_countdown;

static int           G_gameOver;

static int           G_playingAreaSensitive;


/******************************************************************************

                               Application Icon

 ******************************************************************************/

static char G_pipeBitmapBits[] = {
   0xaa, 0x8a, 0xba, 0x8a, 0xba, 0xaa, 0x02, 0x55, 0x45, 0x5d, 0x15, 0x75,
   0x55, 0x01, 0xaa, 0x8a, 0xba, 0x8a, 0xba, 0xaa, 0x02, 0x55, 0x45, 0x5d,
   0x15, 0x75, 0x55, 0x01, 0xaa, 0x0a, 0xba, 0x8a, 0xbe, 0xaa, 0x02, 0x55,
   0x05, 0x5d, 0x15, 0x7d, 0x55, 0x01, 0xaa, 0x2a, 0xba, 0x8a, 0xae, 0xaa,
   0x02, 0x55, 0x15, 0x5d, 0x15, 0x5d, 0x55, 0x01, 0xaa, 0x2a, 0xba, 0x8a,
   0xae, 0xaa, 0x02, 0x55, 0x15, 0x5d, 0x15, 0x5d, 0x55, 0x01, 0xaa, 0x2a,
   0xba, 0x8a, 0xae, 0xaa, 0x02, 0x55, 0x15, 0x5d, 0x15, 0x5d, 0x55, 0x01,
   0x80, 0x2a, 0xba, 0x8a, 0xae, 0x0a, 0x00, 0x40, 0x15, 0x5d, 0x15, 0x5d,
   0x05, 0x00, 0x0a, 0x00, 0xba, 0x8a, 0x0e, 0x80, 0x02, 0x05, 0x00, 0x5d,
   0x15, 0x05, 0x40, 0x01, 0xaa, 0xaa, 0xba, 0x8a, 0xaa, 0xaa, 0x02, 0x55,
   0x55, 0x5d, 0x15, 0x55, 0x55, 0x01, 0xaa, 0xaa, 0xba, 0x8a, 0xaa, 0xaa,
   0x02, 0xff, 0xff, 0x5f, 0x95, 0xff, 0xff, 0x03, 0xff, 0xff, 0xbf, 0xca,
   0xff, 0xff, 0x03, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x01, 0xaa, 0xaa,
   0xaa, 0xaa, 0xaa, 0xaa, 0x02, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x01,
   0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0x02, 0x55, 0x55, 0x55, 0x55, 0x55,
   0x55, 0x01, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0x02, 0x55, 0x55, 0x55,
   0x55, 0x55, 0x55, 0x01, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0x02, 0x00,
   0x00, 0x50, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb8, 0x0a, 0x00, 0x00,
   0x00, 0x55, 0x55, 0x5d, 0x95, 0xaa, 0xaa, 0x02, 0xaa, 0xaa, 0xba, 0x0a,
   0x55, 0x55, 0x01, 0x55, 0x55, 0x5d, 0x95, 0xaa, 0xaa, 0x02, 0xfa, 0x7f,
   0xba, 0x0a, 0xfd, 0x7f, 0x01, 0xf5, 0x3f, 0x5d, 0x95, 0xfe, 0xbf, 0x02,
   0xbf, 0x2a, 0xba, 0x0a, 0x5d, 0xf5, 0x03, 0x7f, 0x15, 0x5d, 0x95, 0xae,
   0xfa, 0x03, 0xaa, 0x2a, 0xba, 0x0a, 0x5d, 0x55, 0x01, 0x55, 0x15, 0x5d,
   0x95, 0xae, 0xaa, 0x02, 0xaa, 0x2a, 0xba, 0x0a, 0x5d, 0x55, 0x01, 0x55,
   0x15, 0x5d, 0x95, 0xae, 0xaa, 0x02, 0xaa, 0x2a, 0xba, 0x0a, 0x5d, 0x55,
   0x01, 0x55, 0x15, 0x5d, 0x95, 0xae, 0xaa, 0x02, 0xaa, 0x0a, 0xba, 0x0a,
   0x7d, 0x55, 0x01, 0x55, 0x05, 0x5d, 0x95, 0xbe, 0xaa, 0x02, 0xaa, 0x8a,
   0xba, 0x0a, 0x75, 0x55, 0x01, 0x55, 0x45, 0x5d, 0x95, 0xba, 0xaa, 0x02,
   0xaa, 0x8a, 0xba, 0x0a, 0x75, 0x55, 0x01, 0x55, 0x45, 0x5d, 0x95, 0xba,
   0xaa, 0x02};



/******************************************************************************

                           Status Checking Routines

 ******************************************************************************/

void check(status,msg)
  int  status;
  char *msg;
{
  if (!status) {
    printf("\n%s\n\n",msg); 
    exit(1);
  }
}

void mrm_check(status,msg)
  int  status;
  char *msg;
{
  if (status != MrmSUCCESS) {
    printf("\n%s\n\n",msg); 
  }
}


/******************************************************************************

                                 RandFromRange

 ******************************************************************************/

/* 
    There have been complaints about this generator.  I can't tell if there's 
    really a problem or if it's just people being frustrated when they don't 
    get the pipes they need.  The pipes output should be tested for 
    independence and uniform distribution.
*/

int RandFromRange(l,h) 
  int l,h;
{
  int i;

  i = (h - l) + 1;

  return((int) (((float) RAND/(((float) MAX_RAND) + 1.)) * (float) i) + l);
}


/******************************************************************************

                               UnwindEventQueue

 ******************************************************************************/

void UnwindEventQueue (appContext, dpy)
  XtAppContext  appContext;
  Display      *dpy;
{
  XEvent        event;

  XSync(dpy, 0);

  while( XtAppPending( appContext ) != 0 ) {
    XtAppNextEvent( appContext, &event );
    XtDispatchEvent( &event );
  }

}


/******************************************************************************

                                   wprintf

 ******************************************************************************/

void wprintf(widget,str)
  Widget   widget;
  char    *str;
{
  Arg      arglist[MAX_ARGS];
  int      argcount;  
  XmString cstr;

  /* create the string */
  cstr = XmStringCreateLtoR(str,XmSTRING_DEFAULT_CHARSET);

  /* update the label */
  argcount = 0;
  XtSetArg(arglist[argcount],XmNlabelString,cstr); argcount++;
  XtSetValues(widget,arglist,argcount);

  /* free the string */
  XmStringFree(cstr);
}


/******************************************************************************

                                   Load Text

 ******************************************************************************/

/* KLUDGE -- check the file size without reading in all the characters... */

char *LoadText(fileName)
  char   *fileName;
{
  FILE   *fp;
  int    ccount,cpos;
  char   *str;

  /* init character counter and char position marker */
  ccount = cpos = 0;

  /* open the file containing help for this window */
  fp = fopen(fileName,"r");

  /* if the open was successful */
  if (fp != 0) {

    /* count the characters in the file */
    while (getc(fp) != EOF) ccount++;

    /* close the file */
    fclose(fp);
  }
  else {

    /* file could not be opened */
    printf("Could not open text file: %s (pass 1)\n",fileName);
    return;
  }

  /* allocate space for this string */
  str = (char *) malloc(ccount);

  /* ANOTHER pass, this time to load text */

  /* open the file containing help for this window */
  fp = fopen(fileName,"r");

  /* if the open was successful */
  if (fp != 0) {

    /* read the characters from the file into the character string */
    while ((str[cpos++] = getc(fp)) != EOF);

    /* plunk in the trailing 0 */
    str[cpos-1] = 0;

    /* close the file */
    fclose(fp);
  }
  else {

    /* file could not be opened */
    printf("Could not open text file: %s (pass 2)\n",fileName);
    return;
  }

  return(str);
}


/******************************************************************************

                                   Countdown

 ******************************************************************************/

void Countdown()
{
  void FluidInc();
  void CountdownInc();
  void FlowometerSet();

  G_countdownTimeOut = 0;

  if (!Iconized())
    CountdownInc(1);

  if (G_countdown == 0) {
    G_countdownTimeOut = 0;
    FlowometerSet(G_levelInfo[G_level].fluidSpeed);
    FluidInc(G_level);
  }
  else 
    G_countdownTimeOut = XtAppAddTimeOut(G_appContext,1000,Countdown,0);
    
}


/******************************************************************************

                                  FastFlowCB

 ******************************************************************************/

void FastFlowCB(widget,parameter,CB_info)
  Widget		        widget;
  int                           parameter;
  XmPushButtonCallbackStruct   *CB_info;
{
  void                          FluidInc();
  void                          CountdownInc();
  void                          ScoreInc();

  /* turn on fast flow */
  G_fastFlow = 1;

  /* if the fluid has not yet begin to flow */
  if (G_countdownTimeOut != 0) {

    /* remove the alarm to set it flowing sometime in the future */
    XtRemoveTimeOut(G_countdownTimeOut);

    /* give the player a bonus for starting early */
    ScoreInc(G_countdown*STARTING_EARLY_BONUS);

    /* set the countdown to zero */
    CountdownInc(G_countdown);

    /* start the fluid flowing */
    FluidInc(G_level);
  }

}


/******************************************************************************

                              NewStartingLevelCB

 ******************************************************************************/

void NewStartingLevelCB(widget,startingLevel,CB_info)
  Widget		        widget;
  int                           startingLevel;
  XmToggleButtonCallbackStruct *CB_info;
{
  if (CB_info->set) G_startingLevel = startingLevel;
}


/******************************************************************************

                            Level Creation Routine

 ******************************************************************************/

void CreateLevels()
{
  LevelInfoRec *level;
  int          l,p;
  Arg          argl[MAX_ARGS];
  int          argc;
  Widget       tb;

  /* There are 8 levels of play in the game. */
  G_topLevel = 8;

  /* Create space for the level information. */
  G_levelInfo = (LevelInfoRec *) malloc(sizeof(LevelInfoRec)*G_topLevel);

  /* Blank out the record. */
  memset(G_levelInfo,0,sizeof(LevelInfoRec)*G_topLevel);

  level = &G_levelInfo[0];
  level->description = "Simple";
  level->tilesAcross = 6;
  level->tilesUpNdown = 9;
  level->headStart = 40;
  level->fluidSpeed = 60;
  level->numPipesToFill = 15;
  level->pipeFrequency[PLUS] = 1;
  level->pipeFrequency[UPPER_LEFT] = 1;
  level->pipeFrequency[UPPER_RIGHT] = 1;
  level->pipeFrequency[LOWER_LEFT] = 1;
  level->pipeFrequency[LOWER_RIGHT] = 1;
  level->pipeFrequency[HORIZONTAL] = 1;
  level->pipeFrequency[VERTICAL] = 1;

  level = &G_levelInfo[1];
  level->tilesAcross = 10;
  level->tilesUpNdown = 10;
  level->description = "Obstacles";
  level->headStart = 40;
  level->fluidSpeed = 60;
  level->numPipesToFill = 25;
  level->numObstacles = 5;
  level->pipeFrequency[PLUS] = 1;
  level->pipeFrequency[UPPER_LEFT] = 1;
  level->pipeFrequency[UPPER_RIGHT] = 1;
  level->pipeFrequency[LOWER_LEFT] = 1;
  level->pipeFrequency[LOWER_RIGHT] = 1;
  level->pipeFrequency[HORIZONTAL] = 1;
  level->pipeFrequency[VERTICAL] = 1;

  level = &G_levelInfo[2];
  level->description = "Fast Flow";
  level->tilesAcross = 7;
  level->tilesUpNdown = 7;
  level->headStart = 25;
  level->fluidSpeed = 10;
  level->numReservoirs = 3;
  level->numPipesToFill = 20;
  level->pipeFrequency[PLUS] = 1;
  level->pipeFrequency[UPPER_LEFT] = 1;
  level->pipeFrequency[UPPER_RIGHT] = 1;
  level->pipeFrequency[LOWER_LEFT] = 1;
  level->pipeFrequency[LOWER_RIGHT] = 1;
  level->pipeFrequency[HORIZONTAL] = 1;
  level->pipeFrequency[VERTICAL] = 1;

  level = &G_levelInfo[3];
  level->tilesAcross = 8;
  level->tilesUpNdown = 12;
  level->description = "Bonus!";
  level->headStart = 20;
  level->fluidSpeed = 60;
  level->numPipesToFill = 30;
  level->numBonusPipes = 7;
  level->numObstacles = 3;
  level->pipeFrequency[PLUS] = 2;
  level->pipeFrequency[UPPER_LEFT] = 2;
  level->pipeFrequency[UPPER_RIGHT] = 2;
  level->pipeFrequency[LOWER_LEFT] = 2;
  level->pipeFrequency[LOWER_RIGHT] = 2;
  level->pipeFrequency[HORIZONTAL] = 2;
  level->pipeFrequency[VERTICAL] = 2;

  level = &G_levelInfo[4];
  level->description = "Marathon";
  level->tilesAcross = 12;
  level->tilesUpNdown = 14;
  level->headStart = 20;
  level->fluidSpeed = 30;
  level->numPipesToFill = 50;
  level->numObstacles = 5;
  level->pipeFrequency[PLUS] = 3;
  level->pipeFrequency[UPPER_LEFT] = 2;
  level->pipeFrequency[UPPER_RIGHT] = 2;
  level->pipeFrequency[LOWER_LEFT] = 2;
  level->pipeFrequency[LOWER_RIGHT] = 2;
  level->pipeFrequency[HORIZONTAL] = 1;
  level->pipeFrequency[VERTICAL] = 1;

  level = &G_levelInfo[5];
  level->description = "One Way";
  level->tilesAcross = 8;
  level->tilesUpNdown = 8;
  level->headStart = 20;
  level->fluidSpeed = 30;
  level->numPipesToFill = 20;
  level->pipeFrequency[PLUS] = 2;
  level->pipeFrequency[UPPER_LEFT] = 2;
  level->pipeFrequency[UPPER_RIGHT] = 2;
  level->pipeFrequency[LOWER_LEFT] = 2;
  level->pipeFrequency[LOWER_RIGHT] = 2;
  level->pipeFrequency[HORIZONTAL] = 2;
  level->pipeFrequency[VERTICAL] = 2;
  level->pipeFrequency[HORIZONTAL_LEFT] = 1;
  level->pipeFrequency[HORIZONTAL_RIGHT] = 1;
  level->pipeFrequency[VERTICAL_UP] = 1;
  level->pipeFrequency[VERTICAL_DOWN] = 1;

  level = &G_levelInfo[6];
  level->description = "Smorgasboard";
  level->tilesAcross = 10;
  level->tilesUpNdown = 10;
  level->headStart = 20;
  level->fluidSpeed = 30;
  level->numReservoirs = 3;
  level->numObstacles = 3;
  level->numBonusPipes = 3;
  level->numPipesToFill = 20;
  level->pipeFrequency[PLUS] = 2;
  level->pipeFrequency[UPPER_LEFT] = 2;
  level->pipeFrequency[UPPER_RIGHT] = 2;
  level->pipeFrequency[LOWER_LEFT] = 2;
  level->pipeFrequency[LOWER_RIGHT] = 2;
  level->pipeFrequency[HORIZONTAL] = 2;
  level->pipeFrequency[VERTICAL] = 2;
  level->pipeFrequency[HORIZONTAL_LEFT] = 1;
  level->pipeFrequency[HORIZONTAL_RIGHT] = 1;
  level->pipeFrequency[VERTICAL_UP] = 1;
  level->pipeFrequency[VERTICAL_DOWN] = 1;

  level = &G_levelInfo[7];
  level->description = "Nasty!";
  level->tilesAcross = 12;
  level->tilesUpNdown = 7;
  level->headStart = 5;
  level->fluidSpeed = 18;
  level->numPipesToFill = 50;
  level->numObstacles = 100;
  level->pipeFrequency[PLUS] = 2;
  level->pipeFrequency[UPPER_LEFT] = 2;
  level->pipeFrequency[UPPER_RIGHT] = 2;
  level->pipeFrequency[LOWER_LEFT] = 2;
  level->pipeFrequency[LOWER_RIGHT] = 2;
  level->pipeFrequency[HORIZONTAL] = 2;
  level->pipeFrequency[VERTICAL] = 2;
  level->pipeFrequency[HORIZONTAL_LEFT] = 1;
  level->pipeFrequency[HORIZONTAL_RIGHT] = 1;
  level->pipeFrequency[VERTICAL_UP] = 1;
  level->pipeFrequency[VERTICAL_DOWN] = 1;

  /* {3,2,5,4} ==> {3,3+2,3+2+5,3+2+5+4} */
  /*           ==> {3,5,10,14} */
  /* This allows RNG to pick a number between 0 and 1-the number in */
  /* the last array element, then use the the number of the cell whose */
  /* number is greater than the number generated as the number of the */
  /* next pipe type */

  for (l = 0; l < G_topLevel; l++)
    for (p = 1; p < NUM_PIPE_TYPES; p++)
      G_levelInfo[l].pipeFrequency[p] += G_levelInfo[l].pipeFrequency[p-1];

  /* create a menu option for each level so the user can start there */
  for (l = 0; l < G_topLevel; l++) {

    argc = 0;
    XtSetArg(argl[argc],XmNspacing,4); argc++;
    XtSetArg(argl[argc],XmNindicatorSize,11); argc++;
    tb = XmCreateToggleButtonGadget(G_startLevel_wgt,"tb",argl,argc);

    XtManageChild(tb);

    wprintf(tb,G_levelInfo[l].description);

    XtAddCallback(tb,XmNvalueChangedCallback,NewStartingLevelCB,l);

    if (l == 0) XmToggleButtonGadgetSetState(tb,True,True);


  }

}


/******************************************************************************

              Label Information Incrementation & Update Routines

 ******************************************************************************/

void ScoreInc(amount)
  int amount;
{
  char str[80];

  G_score += amount;

  if (G_score < 0) G_score = 0;

  sprintf(str,"Score: %d",G_score);
  wprintf(G_score_wgt,str);
}

void CrossInc(amount)
  int  amount;
{
  char str[80];

  sprintf(str,"Crosses: %d",G_crosses += amount);
  wprintf(G_crosses_wgt,str);
}

void PipesInc(amount)
  int  amount;
{
  char str[80];

  sprintf(str,"Pipes Filled: %d",G_pipesFilled += amount);
  wprintf(G_pipes_wgt,str);
}

void LevelInc(amount)
  int amount;
{
  char str[80];

  if (G_level+amount < G_topLevel) G_level += amount;

  sprintf(str,"Level %d: %s",G_level+1,G_levelInfo[G_level].description);
  wprintf(G_level_wgt,str);
}

void PipesNeededSet(number)
  int  number;
{
  char str[80];

  sprintf(str,"Pipes to Fill: %d",number);
  wprintf(G_pipesNeeded_wgt,str);
}

void CountdownInc(number)
  int  number;
{
  char str[80];

  G_countdown -= number;

  sprintf(str,"Seconds until fluid: %d",G_countdown);
  wprintf(G_countdown_wgt,str);
}

void FlowometerSet(number)
  int  number;
{
  char str[80];

  if (G_fastFlow)
    sprintf(str,"Fluid speed: %s","very fast");
  else if (number > 150) 
    sprintf(str,"Fluid speed: %s","very slow");
  else if (number < 1) 
    sprintf(str,"Fluid speed: 0");
  else 
    sprintf(str,"Fluid speed: %d",1000/number);

  wprintf(G_flowometer_wgt,str);
}


/******************************************************************************

                                   Iconized

 ******************************************************************************/

int Iconized()
{
  XWindowAttributes attributes;
  Window            window;
  int               status;

  window = XtWindow(XtParent(G_main_wgt));

  status = XGetWindowAttributes(G_display,window,&attributes);

  if ((status == 1) && (attributes.map_state != IsViewable))
    return(1);
  else
    return(0);
}


/******************************************************************************

                                  GetColors

 ******************************************************************************/

void GetColors(hierarchy,fg,bg)
  MrmHierarchy hierarchy;
  Pixel        *fg,*bg;
{
  int          mrm_stat;

  *fg = BlackPixel(G_display,DefaultScreen(G_display));
  *bg = WhitePixel(G_display,DefaultScreen(G_display));

  /* if it's a B&W monitor, don't bother fetching fluid & replace colors. */

  if (DisplayPlanes(G_display,DefaultScreen(G_display)) == 1) {

    G_fluid_pxl = *fg;
    G_replace_pxl = *fg;

  }

  /* it's a color display, fetch fluid & replace colors */

  else {

    mrm_stat = MrmFetchColorLiteral(hierarchy,
                                    "FLUID_COLOR",
                                    G_display,
                                    NULL,
                                    &G_fluid_pxl);
    mrm_check(mrm_stat,"Could not fetch fluid color.");

    mrm_stat = MrmFetchColorLiteral(hierarchy,
                                    "REPLACE_COLOR",
                                    G_display,
                                    NULL,
                                    &G_replace_pxl);
    mrm_check(mrm_stat,"Could not fetch replace color.");
  }

}


/******************************************************************************

                         Pipe Icon Fetching Routines

 ******************************************************************************/


void getPipePix(name,pixmap,hierarchy,fg,bg)
  char         *name;
  Pixmap       *pixmap;
  MrmHierarchy hierarchy;
  Pixel        fg,bg;
{
  int          mrm_stat;

  mrm_stat =MrmFetchIconLiteral(hierarchy,name,G_screen,G_display,fg,bg,pixmap);
  mrm_check(mrm_stat,"MRM error fetching icon.");
}

void SetupPointer(hierarchy,fg,bg)
  MrmHierarchy hierarchy;
  Pixel        fg,bg;
{
  Pixmap pointer_pixmap;
  Cursor pointer_cursor;
  XColor  fgColor,bgColor;
  
  getPipePix("WRENCH",&pointer_pixmap,hierarchy,fg,bg);

/* KLUDGE 

   I can't make this work.

#ifdef DEBUG
  printf("pixmap: %d\n",pointer_pixmap);
#endif

  fgColor.pixel = fg;
  bgColor.pixel = fg;
  XQueryColor(G_display,DefaultColormapOfScreen(G_screen),&fgColor);
  XQueryColor(G_display,DefaultColormapOfScreen(G_screen),&bgColor);

  pointer_cursor = XCreatePixmapCursor(G_display,pointer_pixmap,pointer_pixmap,
                          &fgColor,&bgColor,5,6);

#ifdef DEBUG
  printf("cursor: %d\n",pointer_cursor);
#endif

  XDefineCursor(G_display,XtWindow(G_playingArea_wgt),pointer_cursor);

*/
}


void initPipeVec(hierarchy,fg,bg)
  MrmHierarchy hierarchy;
  Pixel        fg,bg;
{

 /* The upper-left elbow pipe */

  G_pipeInfo[UPPER_LEFT].flow[LEFT]  = DOWN;
  G_pipeInfo[UPPER_LEFT].flow[RIGHT] = ERROR;
  G_pipeInfo[UPPER_LEFT].flow[UP]    = RIGHT;
  G_pipeInfo[UPPER_LEFT].flow[DOWN]  = ERROR;
  getPipePix("UPPER_LEFT",&G_pipeInfo[UPPER_LEFT].pixmap,hierarchy,fg,bg);

 /* The upper-right elbow pipe */

  G_pipeInfo[UPPER_RIGHT].flow[LEFT]  = ERROR;
  G_pipeInfo[UPPER_RIGHT].flow[RIGHT] = DOWN;
  G_pipeInfo[UPPER_RIGHT].flow[UP]    = LEFT;
  G_pipeInfo[UPPER_RIGHT].flow[DOWN]  = ERROR;
  getPipePix("UPPER_RIGHT",&G_pipeInfo[UPPER_RIGHT].pixmap,hierarchy,fg,bg);

 /* The lower_left elbow pipe */

  G_pipeInfo[LOWER_LEFT].flow[LEFT]  = UP;
  G_pipeInfo[LOWER_LEFT].flow[RIGHT] = ERROR;
  G_pipeInfo[LOWER_LEFT].flow[UP]    = ERROR;
  G_pipeInfo[LOWER_LEFT].flow[DOWN]  = RIGHT;
  getPipePix("LOWER_LEFT",&G_pipeInfo[LOWER_LEFT].pixmap,hierarchy,fg,bg);

 /* The lower-right elbow pipe */

  G_pipeInfo[LOWER_RIGHT].flow[LEFT]  = ERROR;
  G_pipeInfo[LOWER_RIGHT].flow[RIGHT] = UP;
  G_pipeInfo[LOWER_RIGHT].flow[UP]    = ERROR;
  G_pipeInfo[LOWER_RIGHT].flow[DOWN]  = LEFT;
  getPipePix("LOWER_RIGHT",&G_pipeInfo[LOWER_RIGHT].pixmap,hierarchy,fg,bg);

 /* The horizontal pipe */

  G_pipeInfo[HORIZONTAL].flow[LEFT]  = LEFT;
  G_pipeInfo[HORIZONTAL].flow[RIGHT] = RIGHT;
  G_pipeInfo[HORIZONTAL].flow[UP]    = ERROR;
  G_pipeInfo[HORIZONTAL].flow[DOWN]  = ERROR;
  getPipePix("HORIZONTAL",&G_pipeInfo[HORIZONTAL].pixmap,hierarchy,fg,bg);

 /* The vertical pipe */

  G_pipeInfo[VERTICAL].flow[LEFT]  = ERROR;
  G_pipeInfo[VERTICAL].flow[RIGHT] = ERROR;
  G_pipeInfo[VERTICAL].flow[UP]    = UP;
  G_pipeInfo[VERTICAL].flow[DOWN]  = DOWN;
  getPipePix("VERTICAL",&G_pipeInfo[VERTICAL].pixmap,hierarchy,fg,bg);

  /* The plus-shaped pipe. */

  G_pipeInfo[PLUS].flow[LEFT]  = LEFT;
  G_pipeInfo[PLUS].flow[RIGHT] = RIGHT;
  G_pipeInfo[PLUS].flow[UP]    = UP;
  G_pipeInfo[PLUS].flow[DOWN]  = DOWN;
  getPipePix("PLUS",&G_pipeInfo[PLUS].pixmap,hierarchy,fg,bg);

  /* The staring pipe pointed up. */

  G_pipeInfo[START_UP].flow[LEFT]  = ERROR;
  G_pipeInfo[START_UP].flow[RIGHT] = ERROR;
  G_pipeInfo[START_UP].flow[UP]    = ERROR;
  G_pipeInfo[START_UP].flow[DOWN]  = ERROR;
  getPipePix("START_UP",&G_pipeInfo[START_UP].pixmap,hierarchy,fg,bg);

  /* The staring pipe pointed down. */

  G_pipeInfo[START_DOWN].flow[LEFT]  = ERROR;
  G_pipeInfo[START_DOWN].flow[RIGHT] = ERROR;
  G_pipeInfo[START_DOWN].flow[UP]    = ERROR;
  G_pipeInfo[START_DOWN].flow[DOWN]  = ERROR;
  getPipePix("START_DOWN",&G_pipeInfo[START_DOWN].pixmap,hierarchy,fg,bg);

  /* The staring pipe pointed up. */

  G_pipeInfo[START_LEFT].flow[LEFT]  = ERROR;
  G_pipeInfo[START_LEFT].flow[RIGHT] = ERROR;
  G_pipeInfo[START_LEFT].flow[UP]    = ERROR;
  G_pipeInfo[START_LEFT].flow[DOWN]  = ERROR;
  getPipePix("START_LEFT",&G_pipeInfo[START_LEFT].pixmap,hierarchy,fg,bg);

  /* The staring pipe pointed up. */

  G_pipeInfo[START_RIGHT].flow[LEFT]  = ERROR;
  G_pipeInfo[START_RIGHT].flow[RIGHT] = ERROR;
  G_pipeInfo[START_RIGHT].flow[UP]    = ERROR;
  G_pipeInfo[START_RIGHT].flow[DOWN]  = ERROR;
  getPipePix("START_RIGHT",&G_pipeInfo[START_RIGHT].pixmap,hierarchy,fg,bg);

  /* The one-way left pipe. */

  G_pipeInfo[HORIZONTAL_LEFT].flow[LEFT]  = LEFT;
  G_pipeInfo[HORIZONTAL_LEFT].flow[RIGHT] = ERROR;
  G_pipeInfo[HORIZONTAL_LEFT].flow[UP]    = ERROR;
  G_pipeInfo[HORIZONTAL_LEFT].flow[DOWN]  = ERROR;
  getPipePix("HORIZONTAL_LEFT",&G_pipeInfo[HORIZONTAL_LEFT].pixmap,hierarchy,fg,bg);

  /* The one-way right pipe. */

  G_pipeInfo[HORIZONTAL_RIGHT].flow[LEFT]  = ERROR;
  G_pipeInfo[HORIZONTAL_RIGHT].flow[RIGHT] = RIGHT;
  G_pipeInfo[HORIZONTAL_RIGHT].flow[UP]    = ERROR;
  G_pipeInfo[HORIZONTAL_RIGHT].flow[DOWN]  = ERROR;
  getPipePix("HORIZONTAL_RIGHT",&G_pipeInfo[HORIZONTAL_RIGHT].pixmap,hierarchy,fg,bg);

  /* The one-way up pipe. */

  G_pipeInfo[VERTICAL_UP].flow[LEFT]  = ERROR;
  G_pipeInfo[VERTICAL_UP].flow[RIGHT] = ERROR;
  G_pipeInfo[VERTICAL_UP].flow[UP]    = UP;
  G_pipeInfo[VERTICAL_UP].flow[DOWN]  = ERROR;
  getPipePix("VERTICAL_UP",&G_pipeInfo[VERTICAL_UP].pixmap,hierarchy,fg,bg);

  /* The one-way down pipe. */

  G_pipeInfo[VERTICAL_DOWN].flow[LEFT]  = ERROR;
  G_pipeInfo[VERTICAL_DOWN].flow[RIGHT] = ERROR;
  G_pipeInfo[VERTICAL_DOWN].flow[UP]    = ERROR;
  G_pipeInfo[VERTICAL_DOWN].flow[DOWN]  = DOWN;
  getPipePix("VERTICAL_DOWN",&G_pipeInfo[VERTICAL_DOWN].pixmap,hierarchy,fg,bg);

  /* The pixmap displayed as an obstacle to building pipes. */

  G_pipeInfo[OBSTACLE].flow[LEFT]  = ERROR;
  G_pipeInfo[OBSTACLE].flow[RIGHT] = ERROR;
  G_pipeInfo[OBSTACLE].flow[UP]    = ERROR;
  G_pipeInfo[OBSTACLE].flow[DOWN]  = ERROR;
  getPipePix("OBSTACLE",&G_pipeInfo[OBSTACLE].pixmap,hierarchy,fg,bg);

  /* The horizontal bonus pipe. */

  G_pipeInfo[HORIZONTAL_BONUS].flow[LEFT]  = LEFT;
  G_pipeInfo[HORIZONTAL_BONUS].flow[RIGHT] = RIGHT;
  G_pipeInfo[HORIZONTAL_BONUS].flow[UP]    = ERROR;
  G_pipeInfo[HORIZONTAL_BONUS].flow[DOWN]  = ERROR;
  getPipePix("HORIZONTAL_BONUS",&G_pipeInfo[HORIZONTAL_BONUS].pixmap,hierarchy,fg,bg);

  /* The vertical bonus pipe. */

  G_pipeInfo[VERTICAL_BONUS].flow[LEFT]  = ERROR;
  G_pipeInfo[VERTICAL_BONUS].flow[RIGHT] = ERROR;
  G_pipeInfo[VERTICAL_BONUS].flow[UP]    = UP;
  G_pipeInfo[VERTICAL_BONUS].flow[DOWN]  = DOWN;
  getPipePix("VERTICAL_BONUS",&G_pipeInfo[VERTICAL_BONUS].pixmap,hierarchy,fg,bg);

  /* The horizontal reservoir pipe. */

  G_pipeInfo[HORIZONTAL_RESERVOIR].flow[LEFT]  = LEFT;
  G_pipeInfo[HORIZONTAL_RESERVOIR].flow[RIGHT] = RIGHT;
  G_pipeInfo[HORIZONTAL_RESERVOIR].flow[UP]    = ERROR;
  G_pipeInfo[HORIZONTAL_RESERVOIR].flow[DOWN]  = ERROR;
  getPipePix("HORIZONTAL_RESERVOIR",&G_pipeInfo[HORIZONTAL_RESERVOIR].pixmap,hierarchy,fg,bg);

  /* The vertical reservoir pipe. */

  G_pipeInfo[VERTICAL_RESERVOIR].flow[LEFT]  = ERROR;
  G_pipeInfo[VERTICAL_RESERVOIR].flow[RIGHT] = ERROR;
  G_pipeInfo[VERTICAL_RESERVOIR].flow[UP]    = UP;
  G_pipeInfo[VERTICAL_RESERVOIR].flow[DOWN]  = DOWN;
  getPipePix("VERTICAL_RESERVOIR",&G_pipeInfo[VERTICAL_RESERVOIR].pixmap,hierarchy,fg,bg);

  /* The pixmap displayed where there is no pipe. */
  getPipePix("EMPTY",&G_pipeInfo[EMPTY].pixmap,hierarchy,fg,bg);

  /* The pixmap displayed when a pipe is not filled at end of level. */
  getPipePix("WASTED",&G_pipeInfo[WASTED].pixmap,hierarchy,fg,bg);
}

/******************************************************************************

                                 DisplayTile

 ******************************************************************************/

void DisplayTile(pipeType,x,y,widget)
  int    pipeType;
  int    x,y;
  Widget widget;
{
  Window window;

#ifdef DEBUG
  printf("display tile %d at: %d %d\n",pipeType,x,y);
#endif

  if ((pipeType < 0) || (pipeType >= NUM_PIPE_TYPES)) 
    printf("Internal error.  Bad pipe id.\n");

  if (widget == G_playingArea_wgt) {

    /* first the live copy to the screen (if the window exists yet) */
    if (window = XtWindow(widget))
    XCopyArea(G_display,G_pipeInfo[pipeType].pixmap,
              window,G_generic_GC,
              0,0,PIPETILE_SIZE,PIPETILE_SIZE,
              x*PIPETILE_SIZE,y*PIPETILE_SIZE);

    /* then the safe copy for expose events */
    XCopyArea(G_display,G_pipeInfo[pipeType].pixmap,
              G_playingAreaShadow,G_generic_GC,
              0,0,PIPETILE_SIZE,PIPETILE_SIZE,
              x*PIPETILE_SIZE,y*PIPETILE_SIZE);
  }
  else if (widget == G_preview_wgt) {

    /* first the live copy to the screen (if the window exists yet) */
    if (window = XtWindow(widget))
    XCopyArea(G_display,G_pipeInfo[pipeType].pixmap,
              window,G_generic_GC,
              0,0,PIPETILE_SIZE,PIPETILE_SIZE,
              x*PIPETILE_SIZE,y*PIPETILE_SIZE);

    /* then the safe copy for expose events */
    XCopyArea(G_display,G_pipeInfo[pipeType].pixmap,
              G_previewShadow,G_generic_GC,
              0,0,PIPETILE_SIZE,PIPETILE_SIZE,
              x*PIPETILE_SIZE,y*PIPETILE_SIZE);
  }
}


/******************************************************************************

                                  RedrawplayingArea

 ******************************************************************************/

void RedrawplayingArea()
{
  int xTile,yTile;
  int pipeType;

  for (xTile = 0; xTile < NUM_TILES_SIDETOSIDE; xTile++)
    for (yTile = 0; yTile < NUM_TILES_UPANDDOWN; yTile++) {
      pipeType = G_pipeLayout[xTile][yTile].type;
      DisplayTile(pipeType,xTile,yTile,G_playingArea_wgt);
    }
}

/******************************************************************************

                              RedrawPreviewArea

 ******************************************************************************/

void RedrawPreviewArea()
{
  int yTile;

  for (yTile = 0; yTile < NUM_TILES_PREVIEWED; yTile++) 
    DisplayTile(G_previewLayout[yTile],0,yTile,G_preview_wgt);
}


/******************************************************************************

                                   NextTile

 ******************************************************************************/
/*

    Takes a tile off the head of the queue and adds one at random to the tail.

*/

int NextTile()
{
  int newTile;
  int nextTile;
  int tile;
  int r;

  nextTile = G_previewLayout[NUM_TILES_PREVIEWED-1];

  r = RandFromRange(0,G_levelInfo[G_level].pipeFrequency[MAX_PIPE]-1);

  newTile = 0;

  while (r >= G_levelInfo[G_level].pipeFrequency[newTile])
    newTile++;

  for (tile = NUM_TILES_PREVIEWED-1; tile > 0; tile--) 
    G_previewLayout[tile] = G_previewLayout[tile-1];

  G_previewLayout[0] = newTile;

  RedrawPreviewArea();

  return(nextTile);

}


/******************************************************************************

                              Drawing Primitives

 ******************************************************************************/

void line(x1,y1,x2,y2)
  int x1,y1,x2,y2;
{
  Window window;

# ifdef DEBUG
    printf("line: %d %d %d %d\n",x1,y1,x2,y2);
# endif

  if (window = XtWindow(G_playingArea_wgt))
    XDrawLine(G_display,window,G_generic_GC,x1,y1,x2,y2);

  XDrawLine(G_display,G_playingAreaShadow,G_generic_GC,x1,y1,x2,y2);

}

void rectangle(x,y,w,h)
  int x,y,w,h;
{
  Window window;

# ifdef DEBUG
    printf("rectangle: %d %d %d %d\n",x,y,w,h);
# endif

  if (window = XtWindow(G_playingArea_wgt))
    XDrawRectangle(G_display,window,G_generic_GC,x,y,w-1,h-1);

  XDrawRectangle(G_display,G_playingAreaShadow,G_generic_GC,x,y,w-1,h-1);

}

void filledCircle (x,y,r)
  int x,y,r;
{
  Window window;

# ifdef DEBUG
    printf("filled circle: %d %d %d \n",x,y,r);
# endif

  if (window = XtWindow(G_playingArea_wgt))
    XFillArc(G_display,window,G_generic_GC,x-r,y-r,r*2,r*2,0,360*64);

  XFillArc(G_display,G_playingAreaShadow,G_generic_GC,x-r,y-r,r*2,r*2,0,360*64);

}


/******************************************************************************

                                    erase

 ******************************************************************************/

void erase(eraseInfoRec)
  EraseInfo *eraseInfoRec;
{
  int        x,y,offset;

  G_eraseTimeOut = 0;

  if ((eraseInfoRec->step + ERASE_INCREMENT) > PIPETILE_SIZE) {

    G_playingAreaSensitive = True;

    G_pipeLayout[eraseInfoRec->xTile][eraseInfoRec->yTile].type = NextTile();

    DisplayTile(G_pipeLayout[eraseInfoRec->xTile][eraseInfoRec->yTile].type,
                eraseInfoRec->xTile,eraseInfoRec->yTile,
                G_playingArea_wgt);

  }
  else {

    eraseInfoRec->step += ERASE_INCREMENT;

    offset = eraseInfoRec->step / 2;
    x = eraseInfoRec->xTile * PIPETILE_SIZE + PIPETILE_SIZE / 2;
    y = eraseInfoRec->yTile * PIPETILE_SIZE + PIPETILE_SIZE / 2;

    XSetForeground(G_display,G_generic_GC,G_replace_pxl);
    rectangle(x-offset,y-offset,eraseInfoRec->step,eraseInfoRec->step);

    G_eraseTimeOut = XtAppAddTimeOut(G_appContext,ERASE_INTERVAL,erase,eraseInfoRec);
    
  }
}

/******************************************************************************

                              PlayingAreaInputCB

 ******************************************************************************/
/* 

    Processes input (mouse clicks) from the playing area.

*/


void PlayingAreaInputCB(widget,app_data,CB_data)
  Widget                       widget;
  caddr_t                      app_data;
  XmDrawingAreaCallbackStruct *CB_data;
{
  XButtonEvent                *btn_event;
  int                          xTile,yTile;
  static EraseInfo             eraseInfoRec;


#   ifdef DEBUG
      printf("Entering input callback.\n");
#   endif

  btn_event = (XButtonEvent *) CB_data->event;

  xTile = btn_event->x / PIPETILE_SIZE;
  yTile = btn_event->y / PIPETILE_SIZE;

  if ( (G_playingAreaSensitive) && 
       (CB_data->event->type == ButtonPress) &&
       (NUM_TILES_UPANDDOWN > yTile) && 
       (NUM_TILES_SIDETOSIDE > xTile) ) {

#   ifdef DEBUG
      printf("Button input received.\n");
#   endif

    if (!G_pipeLayout[xTile][yTile].permanent && 
        !G_pipeLayout[xTile][yTile].flooded) {

      if (G_pipeLayout[xTile][yTile].type == EMPTY) {

        G_pipeLayout[xTile][yTile].type = NextTile();

        DisplayTile(G_pipeLayout[xTile][yTile].type, 
                    xTile,yTile,
                    G_playingArea_wgt);
      }
      else {

        G_pipeLayout[xTile][yTile].type = ERASED;

        G_playingAreaSensitive = False;

        eraseInfoRec.step = 0;
        eraseInfoRec.xTile = xTile;
        eraseInfoRec.yTile = yTile;
        erase(&eraseInfoRec);
      }

    }
  }

#   ifdef DEBUG
      printf("Exiting input callback.\n");
#   endif

}

/******************************************************************************

                               RegisterWidgetCB

 ******************************************************************************/


void RegisterWidgetCB(widget,widget_name,Xdata)
  Widget  widget;
  char    *widget_name;
  caddr_t Xdata;
{

# ifdef DEBUG
    printf("Entering widget register callback.\n");
# endif

  if (strcmp(widget_name,"PLAYINGAREA") == 0) G_playingArea_wgt = widget;
  else if (strcmp(widget_name,"MAIN") == 0) G_main_wgt = widget; 
  else if (strcmp(widget_name,"PREVIEW") == 0) G_preview_wgt = widget; 
  else if (strcmp(widget_name,"SCORE_LABEL") == 0) G_score_wgt = widget; 
  else if (strcmp(widget_name,"CROSSES_LABEL") == 0) G_crosses_wgt = widget;
  else if (strcmp(widget_name,"PIPES_LABEL") == 0) G_pipes_wgt = widget; 
  else if (strcmp(widget_name,"PIPES_NEEDED_LABEL") == 0) G_pipesNeeded_wgt = widget; 
  else if (strcmp(widget_name,"LEVEL_LABEL") == 0) G_level_wgt = widget; 
  else if (strcmp(widget_name,"COUNTDOWN_LABEL") == 0) G_countdown_wgt = widget; 
  else if (strcmp(widget_name,"FLOW_LABEL") == 0) G_flowometer_wgt = widget; 
  else if (strcmp(widget_name,"PAUSE") == 0) G_pause_wgt = widget;
  else if (strcmp(widget_name,"ABORT") == 0) G_abort_wgt = widget;
  else if (strcmp(widget_name,"START") == 0) G_start_wgt = widget;
  else if (strcmp(widget_name,"MENU_BAR") == 0) G_menuBar_wgt = widget;
  else if (strcmp(widget_name,"LO_LABEL") == 0) G_levelOverLabel_wgt = widget;
  else if (strcmp(widget_name,"STARTING_LEVEL") == 0) G_startLevel_wgt = widget;
  else if (strcmp(widget_name,"HELP_TEXT") == 0) G_helpText_wgt = widget;
  else if (strcmp(widget_name,"HIGH_SCORES_TEXT") == 0) G_hsText_wgt = widget;
  else if (strcmp(widget_name,"LO_BUTTON") == 0) G_loButton_wgt = widget;
  else if (strcmp(widget_name,"FLUID") == 0) G_fluid_wgt = widget;

# ifdef DEBUG

  else printf("Uh-oh! No match found in register!\n");

    printf("Exiting widget register callback.\n");
# endif
}


/******************************************************************************

                                   ExposeCB

 ******************************************************************************/


void ExposeCB(widget,app_data,CB_data)
  Widget                      widget;
  caddr_t                     app_data;
  XmDrawingAreaCallbackStruct *CB_data;
{
  XExposeEvent *expose_event;


#   ifdef DEBUG
      printf("Entering expose callback.\n");
#   endif

  expose_event = (XExposeEvent *) CB_data->event;

  if (expose_event->window == XtWindow(G_playingArea_wgt))
    XCopyArea(G_display,G_playingAreaShadow,XtWindow(G_playingArea_wgt),G_generic_GC,
              0,0,PLAYINGAREA_WIDTH,PLAYINGAREA_HEIGHT,0,0);
  else
    XCopyArea(G_display,G_previewShadow,XtWindow(G_preview_wgt),G_generic_GC,
              0,0,PREVIEWAREA_WIDTH,PREVIEWAREA_HEIGHT,0,0);

#   ifdef DEBUG
      printf("Exiting expose callback.\n");
#   endif

}

/******************************************************************************

                             CreateShadowPixmaps

 ******************************************************************************/


void CreateShadowPixmaps()
{
  Window       window,root;
  int          x,y;
  unsigned int width,height,border,depth;

  XGetGeometry(G_display,DefaultRootWindow(G_display),
               &root,&x,&y,&width,&height,&border,&depth);

  G_playingAreaShadow = XCreatePixmap(G_display,
                                      root,
                                      MAX_TILES_SIDETOSIDE*PIPETILE_SIZE,
                                      MAX_TILES_UPANDDOWN*PIPETILE_SIZE,
                                      depth);

  G_previewShadow = XCreatePixmap(G_display,
                                  root,
                                  PIPETILE_SIZE,
                                  NUM_TILES_PREVIEWED*PIPETILE_SIZE,
                                  depth);

}

/******************************************************************************

                                   GameOver

 ******************************************************************************/

void GameOver()
{
  G_gameOver = True;

  /* Do not allow playing tiles between games. */
  G_playingAreaSensitive = False;
}


/******************************************************************************

                              RemoveUnusedPipes

 ******************************************************************************/

int RemoveUnusedPipes()
{
  int x,y;
  int pipes_wasted = 0;

  for (x=0; x<NUM_TILES_SIDETOSIDE; x++)
    for (y=0; y<NUM_TILES_UPANDDOWN; y++)
      if (( G_pipeLayout[x][y].type != EMPTY) &&
          (!G_pipeLayout[x][y].flooded) && 
          (!G_pipeLayout[x][y].permanent)) 
      {
        pipes_wasted++;
#ifndef LEAVE_UNUSED_TILES
        DisplayTile(WASTED,x,y,G_playingArea_wgt);
#endif
      }

  return(pipes_wasted);
}


/******************************************************************************

                                   LevelOver

 ******************************************************************************/

void LevelOver(msg)
  char *msg;
{
  int   wasted_pipes,bonus,penalty;
  char  lo_text[200];
  void  UpdateHighScores();

  /* If there's an outstanding timeout to perform an erase, then remove it. */
  if (G_eraseTimeOut) {
    XtRemoveTimeOut(G_eraseTimeOut);
    G_eraseTimeOut = 0;
  }

  bonus = (G_crosses / 5)* FIVE_CROSS_SCORE;
  ScoreInc(bonus);

  wasted_pipes = RemoveUnusedPipes();
  penalty = wasted_pipes * NEW_TILE_SCORE * -2;
  ScoreInc(penalty);

  if (G_pipesFilled >= G_levelInfo[G_level].numPipesToFill) {
    sprintf(lo_text,LO_DIALOG_STR,
            "Level Over",msg,G_crosses,bonus,wasted_pipes,penalty);
    wprintf(G_loButton_wgt,"Psyche!");
    G_playingAreaSensitive = False;
  }
  else {
    sprintf(lo_text,LO_DIALOG_STR,
            "Game Over" ,msg,G_crosses,bonus,wasted_pipes,penalty);
    wprintf(G_loButton_wgt,"Bummer.");
    GameOver();
  }

  wprintf(G_levelOverLabel_wgt,lo_text);

  XtManageChild(G_levelOver_wgt);

  UnwindEventQueue(G_appContext,G_display);

  if (G_gameOver) UpdateHighScores(0);


}

/******************************************************************************

                                 FillReservoir

 ******************************************************************************/

void FillReservoir(radius)
  int radius;
{
  void FluidInc();
  int  nextFill;

  G_fillresTimeOut = 0;

  if (radius > RESERVOIR_RADIUS) {
    FluidInc(G_level);
  }
  else {
    XSetForeground(G_display,G_generic_GC,G_fluid_pxl);

    filledCircle(G_fluidInfo.xTile*PIPETILE_SIZE+(PIPETILE_SIZE / 2),
                 G_fluidInfo.yTile*PIPETILE_SIZE+(PIPETILE_SIZE / 2),
                 radius);

    if (G_fastFlow)
      nextFill = 0;
    else
      nextFill = RESERVOIR_FILL_INTERVAL;

    G_fillresTimeOut = XtAppAddTimeOut(G_appContext,nextFill,FillReservoir,radius+1);
  }
}


/******************************************************************************

                                   FluidInc

 ******************************************************************************/

void FluidInc(level)
  int  level;
{
  int  xTile,yTile;
  int  X,Y;
  int  pipeType;
  int  new_pipe;
  int  new_direction;
  char str[80];
  int  next_inc;
  int  halfway_through;
  int  when;

# ifdef DEBUG
    printf("Fluid Increment\n");
# endif

  G_fluidTimeOut = 0;

  if (G_fastFlow) {
    next_inc = 0;
  }
  else {
    next_inc = G_levelInfo[level].fluidSpeed - G_pipesFilled;
    if (next_inc < FASTEST_FLUID_SPEED) next_inc = FASTEST_FLUID_SPEED;
  }

#ifdef DEBUG
  next_inc = 500;
#endif

  if (Iconized()) {
    G_fluidTimeOut = XtAppAddTimeOut(G_appContext,250,FluidInc,level);
    return;
  }

  /* Use temporary variables for the tile coordinates */
  xTile = G_fluidInfo.xTile;  
  yTile = G_fluidInfo.yTile;  
  X     = G_fluidInfo.Xpos;
  Y     = G_fluidInfo.Ypos;

  /* Make sure we haven't run off the end of the tile. */
       if (X > PIPETILE_SIZE-1) { xTile++; X = 0; }
  else if (X < 0)               { xTile--; X = PIPETILE_SIZE - 1; }
  else if (Y > PIPETILE_SIZE-1) { yTile++; Y = 0; }
  else if (Y < 0)               { yTile--; Y = PIPETILE_SIZE - 1; }

  /* Update the fluid rec in case the XY pos changed due to wraparound. */
  G_fluidInfo.Xpos = X;
  G_fluidInfo.Ypos = Y;

  /* If we're out of bounds and there's no wraparound, GAME OVER! */
  if (xTile > NUM_TILES_SIDETOSIDE-1)
    if (G_horizWarp[yTile]) xTile = 0;
    else { LevelOver("Out of bounds to right."); return; }
  else if (xTile < 0)
    if (G_horizWarp[yTile]) xTile = NUM_TILES_SIDETOSIDE-1;
    else { LevelOver("Out of bounds to left."); return; }
  else if (yTile > NUM_TILES_UPANDDOWN-1)
    if (G_vertWarp[xTile]) yTile = 0;
    else { LevelOver("Out of bounds above."); return; }
  else if (yTile < 0)
    if (G_vertWarp[xTile]) yTile = NUM_TILES_UPANDDOWN-1;
    else { LevelOver("Out of bounds below."); return; }

  new_pipe = FALSE;

  /* If we're on a new tile, update the fluid rec and inc pipefilled counter. */
  if (yTile != G_fluidInfo.yTile) { 
    G_fluidInfo.yTile = yTile;
    PipesInc(1);
    new_pipe = TRUE;
  }
  else if (xTile != G_fluidInfo.xTile) {
    G_fluidInfo.xTile = xTile;
    PipesInc(1);
    new_pipe = TRUE;
  }

  /* Find out through what kind of pipe we're flowing. */
  pipeType = G_pipeLayout[xTile][yTile].type;

#ifdef DEBUG
  printf("Pipe Type: %d\n",pipeType);
#endif

  /* If there is no pipe, GAME OVER! */
  if (pipeType == EMPTY) 
    { LevelOver("No pipe present."); return; }

  /* If there is no pipe, GAME OVER! */
  if (pipeType == ERASED) 
    { LevelOver("Pipe being erased."); return; }

  /* If we're on a new tile, make sure that we can enter from this direction. */
  if (new_pipe && (G_pipeInfo[pipeType].flow[G_fluidInfo.direction] == ERROR))
    { LevelOver("Wrong pipe present."); return; }

  if (new_pipe) {

    FlowometerSet(next_inc);

    if (G_pipeLayout[xTile][yTile].type == HORIZONTAL_BONUS ||
        G_pipeLayout[xTile][yTile].type == VERTICAL_BONUS)
      ScoreInc(BONUS_PIPE_SCORE * FF_MULTIPLIER);

    ScoreInc(NEW_TILE_SCORE * FF_MULTIPLIER);

    if (G_pipeLayout[xTile][yTile].flooded) {
#ifdef DEBUG
      printf("Cross pipe filled.\n");
#endif

      CrossInc(1);
      ScoreInc(CROSS_SCORE * FF_MULTIPLIER);
    }

    G_pipeLayout[xTile][yTile].flooded = TRUE;
  }

  /* Draw a vertical or horizontal line, depending on direction of flow. */
# define XOFF (xTile * PIPETILE_SIZE)
# define YOFF (yTile * PIPETILE_SIZE)
  XSetForeground(G_display,G_generic_GC,G_fluid_pxl);
  if ((G_fluidInfo.direction == UP) || (G_fluidInfo.direction == DOWN))
    line(XOFF+X,YOFF+Y,XOFF+X+DIAMETER-1,YOFF+Y);
  else
    line(XOFF+X,YOFF+Y,XOFF+X,YOFF+Y+DIAMETER-1);

  if (!G_fastFlow)
    XSync(G_display,FALSE);

#ifdef DEBUG
  printf("Direction: %d\n",G_fluidInfo.direction);
#endif

  /* find out if we're halfway through the pipe. */
  halfway_through = ( 
    ((G_fluidInfo.direction == UP) && (G_fluidInfo.Ypos == SHORT_LEN)) ||
    ((G_fluidInfo.direction == DOWN) && (G_fluidInfo.Ypos == LONG_LEN-1))||
    ((G_fluidInfo.direction == LEFT) && (G_fluidInfo.Xpos == SHORT_LEN)) ||
    ((G_fluidInfo.direction == RIGHT) && (G_fluidInfo.Xpos == LONG_LEN-1)) );

  /* is the fluid halfway through the pipe? */
  if (halfway_through) {

    /* which direction should fluid flow for second half of the pipe? */
    new_direction = G_pipeInfo[pipeType].flow[G_fluidInfo.direction];

    /* Make sure the fluid coordinates are correct. */

    if ((G_fluidInfo.direction == RIGHT) && (new_direction == DOWN))
      { G_fluidInfo.Xpos -= DIAMETER-1; G_fluidInfo.Ypos += DIAMETER-1; }

    else if ((G_fluidInfo.direction == RIGHT) && (new_direction == UP))
      { G_fluidInfo.Xpos -= DIAMETER-1; }

    else if ((G_fluidInfo.direction == UP) && (new_direction == RIGHT))
      { G_fluidInfo.Xpos += DIAMETER-1; }

    else if ((G_fluidInfo.direction == DOWN) && (new_direction == LEFT))
      { G_fluidInfo.Ypos -= DIAMETER-1; }

    else if ((G_fluidInfo.direction == DOWN) && (new_direction == RIGHT))
      { G_fluidInfo.Xpos += DIAMETER-1; G_fluidInfo.Ypos -= DIAMETER-1; }

    else if ((G_fluidInfo.direction == LEFT) && (new_direction == DOWN))
      { G_fluidInfo.Ypos += DIAMETER-1; }

    /* if we changed directions, the fluid info rec needs to be updated. */
    G_fluidInfo.direction = new_direction;

  }
                                                            
#ifdef DEBUG
  printf("Direction: %d\n",G_fluidInfo.direction);
#endif

  /* Move the flow of fluid one more unit. */
  switch (G_fluidInfo.direction) {
    case UP    : G_fluidInfo.Ypos--; break;
    case DOWN  : G_fluidInfo.Ypos++; break;
    case LEFT  : G_fluidInfo.Xpos--; break;
    case RIGHT : G_fluidInfo.Xpos++; break;
#ifdef DEBUG
    case ERROR : printf("Bad direction.\n");
#endif
  }

  /* if halfway through a reservoir pipe, begin filling and stop flowing */
  if (halfway_through && 
      (pipeType == VERTICAL_RESERVOIR || pipeType == HORIZONTAL_RESERVOIR) ) {

    if (G_fastFlow)
      when = 0;
    else
      when = RESERVOIR_FILL_INTERVAL;

    G_fillresTimeOut = XtAppAddTimeOut(G_appContext,when,FillReservoir,0);

  }
  else {

    /* Set timer to increment fluid again. */
    G_fluidTimeOut = XtAppAddTimeOut(G_appContext,next_inc,FluidInc,level);

  }

}

/******************************************************************************

                                  StartLevel

 ******************************************************************************/

void StartLevel()
{
  void SetupLevel();

  /* reset the number of pipes needed to fill to go on to the next level */
  PipesNeededSet(G_levelInfo[G_level].numPipesToFill);

  /* set up the headstart until the fluid begins. */
  G_countdown = G_levelInfo[G_level].headStart + 1;
  Countdown();

  /* this sets up the playing area for the next level. */
  SetupLevel();

  /* this allows clicks in the playing area to lay down tiles */
  G_playingAreaSensitive = True;

  /* While a level is underway, it is reasonable to accelerate the fluid. */
  XtSetSensitive(G_fluid_wgt,True);

}

/******************************************************************************

                                 ClearScreen

 ******************************************************************************/

void ClearScreen()
{
  int x,y;
  int tile;

  for (x = 0; x<NUM_TILES_SIDETOSIDE; x++)
    for (y = 0; y<NUM_TILES_UPANDDOWN; y++)
      DisplayTile(EMPTY,x,y,G_playingArea_wgt);

  for (y = 0; y<NUM_TILES_PREVIEWED; y++) 
    DisplayTile(EMPTY,0,y,G_preview_wgt);

}


/******************************************************************************

                                    HelpCB

 ******************************************************************************/

void HelpCB(help_wanted)
  int *help_wanted;
{
  if (*help_wanted)
    XtManageChild(G_helpWindow_wgt);
  else
    XtUnmanageChild(G_helpWindow_wgt);
}


/******************************************************************************
                                       
                                  HighScoreCB

 ******************************************************************************/

void HighScoreCB(hs_wanted)
  int *hs_wanted;
{
  if (*hs_wanted)
    XtManageChild(G_highScore_wgt);
  else
    XtUnmanageChild(G_highScore_wgt);
}


/******************************************************************************

                               UpdateHighScores

 ******************************************************************************/

/* 
   KLUDGE -- this does work, but doesn't update unles *you* got a high 
   score.  Also, VMS folks will often complain that high scores don't 
   work, but it almost always turns out that they don't have write 
   access to the directory where the high score file is installed.
*/

void UpdateHighScores (display)
  int           display;
{
  int           score = 0;
  char         *name = 0;
  ScoreNodePtr  thisNode,lastNode,firstNode,myNode,myPrev;
  static char  *highScoreStr = 0;
  char          str[40];
  FILE         *fp;
  char          otherName[40];
  int           otherScore;
  int           newHighScore;
  int           scoreOutput;
  char          highScoreFilename[80];

  /* use these insead of the globals... */
  name = G_username;
  score = G_score;

  /* blank out the high score list */
  firstNode = 0;
  lastNode  = 0;
  myNode    = 0;
  myPrev    = 0;

  /* open the high score file */
  strcpy(highScoreFilename,HIGH_SCORE_LOC);
  strcat(highScoreFilename,HIGH_SCORE_FILE);
  fp = fopen(highScoreFilename,"r");

  /* if the open was successful... */
  if (fp != 0) {

    /* read in the other high scores */
    while (fscanf(fp,"%s %d\n",otherName,&otherScore) == 2) {

      /* create the node */
      thisNode = (ScoreNodePtr) malloc(sizeof(ScoreNode));

      /* if this is the first, assign the head pointer to this node */
      if (firstNode == 0) firstNode = thisNode;

      /* otherwise, point the previous node to this one */
      else lastNode->next = thisNode;

      /* fill in this node */
      thisNode->next = 0;
      strcpy(thisNode->name,otherName);
      thisNode->score = otherScore;

      /* if this is the user's score, keep a pointer to this node */
      if (strcmp(otherName,name) == 0) {
        myNode = thisNode;
        myPrev = lastNode;
      }

      /* next time around, this node will be the last one... */
      lastNode = thisNode;

    }

    /* close the high score file */
    fclose(fp);

  }

  /* if this for display purposes only, then just do the display */

  if (display) {
    XmTextSetString(G_hsText_wgt," ");
    thisNode = firstNode;
    while (thisNode != 0) {
      sprintf(str,"%-13s %7d\n",thisNode->name,thisNode->score);
      XmTextInsert(G_hsText_wgt,0,str);
      thisNode = thisNode->next;
    }
  } 

  /* else if this is not just for display, then merge in the current score */

  else {

    newHighScore = 0;

    /* if the user's not in the list, then this is a high score for him/her */
    if (myNode == 0) {
      newHighScore = 1;
    }

    /* if the user is in the list, then it's only a high score if it's higher */
    else {
      if (score > myNode->score) {
        newHighScore = 1;

        /* pull this node out of the list */
        if (myPrev == 0) 
          firstNode = myNode->next;
        else 
          myPrev->next = myNode->next;

        /* free the node */
        free(myNode);

      }
    }

    if (newHighScore) {

      XmTextSetString(G_hsText_wgt," ");

      thisNode = firstNode;
      scoreOutput = 0;

      fp = 0;
      fp = fopen(highScoreFilename,"w");

      while (thisNode != 0) {

        if (score < thisNode->score && !scoreOutput) {
          scoreOutput = 1;
          if (fp != 0) fprintf(fp,"%-13s %7d\n",name,score);
          sprintf(str,"%-13s %7d\n",name,score);
          XmTextInsert(G_hsText_wgt,0,str);
        }

        if (fp != 0) fprintf(fp,"%-13s %7d\n",thisNode->name,thisNode->score);
        sprintf(str,"%-13s %7d\n",thisNode->name,thisNode->score);
        XmTextInsert(G_hsText_wgt,0,str);

        thisNode = thisNode->next;

      }

      if (!scoreOutput) {
        if (fp != 0) fprintf(fp,"%-13s %7d\n",name,score);
        sprintf(str,"%-13s %7d\n",name,score);
        XmTextInsert(G_hsText_wgt,0,str);
      }

      if (fp != 0) fclose(fp);

    }

  } /* end else not for display only */

  /* destroy the linked list */
  thisNode = firstNode;

  while (thisNode != 0) {
    firstNode = thisNode->next;
    free(thisNode);
    thisNode = firstNode;
  }

}


/******************************************************************************

                                 LevelOverCB

 ******************************************************************************/

void LevelOverCB()
{
  int pop_up;

  /* get rid of the "Level Over" dialog box */
  XtUnmanageChild(G_levelOver_wgt);

  /* The fluid cant be accelerated before it's started. */
  XtSetSensitive(G_fluid_wgt,False);

  /* if he didn't make it to thenext level, reset the game & show high scores */
  if (G_gameOver) {

    /* remove all the tiles from the previous game */
    ClearScreen();

    /* Allow a game to be started, now that this one is over. */
    XtSetSensitive(G_start_wgt,True);

    /* you can't abort a game that hasn't started yet... */
    XtSetSensitive(G_abort_wgt,False);

    /* pop up the high score window */
    pop_up = 1;
    HighScoreCB(&pop_up);

  }

  /* if he did make it to the next level, increment the level and start */
  else {

    LevelInc(1);

    StartLevel();

  }

}

/******************************************************************************

                                   ValidPos

 ******************************************************************************/

Boolean ValidPos(pipeType,x,y)
  int pipeType,x,y;
{

  /* note: pipe type is ignored for now... */

  if (G_pipeLayout[x][y].type != EMPTY) return (False);

  if ( (x >= NUM_TILES_SIDETOSIDE-1) || (x <= 0) ) return(False);
  if ( (y >= NUM_TILES_UPANDDOWN-1) || (y <= 0) ) return(False);

  if (G_pipeLayout[x-1][y+1].type != EMPTY) return (False);
  if (G_pipeLayout[x+1][y-1].type != EMPTY) return (False);
  if (G_pipeLayout[x-1][y-1].type != EMPTY) return (False);
  if (G_pipeLayout[x+1][y+1].type != EMPTY) return (False);
  if (G_pipeLayout[x+1][y].type != EMPTY) return (False);
  if (G_pipeLayout[x-1][y].type != EMPTY) return (False);
  if (G_pipeLayout[x][y+1].type != EMPTY) return (False);
  if (G_pipeLayout[x][y-1].type != EMPTY) return (False);

  return(True);
}

/******************************************************************************

                                 SetupLevel

 ******************************************************************************/

void SetupLevel()
{
  int x,y,i;
  int pipeType;
  Arg argl[MAX_ARGS];
  int argc;
  int fails,valid;

  argc = 0;  
  XtSetArg(argl[argc],XmNwidth,PLAYINGAREA_WIDTH); argc++;
  XtSetArg(argl[argc],XmNheight,PLAYINGAREA_HEIGHT + MENU_BAR_HEIGHT); argc++;
  XtSetValues(G_appShell,argl,argc);

  argc = 0;
  XtSetArg(argl[argc],XmNx,250+PLAYINGAREA_WIDTH); argc++;
  XtSetValues(G_statusShell,argl,argc);

  /* Blank out all the playing area tiles. */
  for (x=0; x<NUM_TILES_SIDETOSIDE; x++) 
    for (y=0; y<NUM_TILES_UPANDDOWN; y++) {
      G_pipeLayout[x][y].type = EMPTY;
      G_pipeLayout[x][y].permanent = FALSE;
      G_pipeLayout[x][y].flooded = FALSE;
    }

  /* Find a starting position and orientation. */
  do {
    pipeType = RandFromRange(START_UP,START_RIGHT);
    x        = RandFromRange(0,NUM_TILES_SIDETOSIDE-1);
    y        = RandFromRange(0,NUM_TILES_UPANDDOWN-1);
  } while (!ValidPos(pipeType,x,y));

  /* Record the chosen starting position. */
  G_fluidInfo.xTile = x;
  G_fluidInfo.yTile = y;

  /* Record the initial position. */
  G_pipeLayout[x][y].type = pipeType;
  G_pipeLayout[x][y].permanent = TRUE;
  G_pipeLayout[x][y].flooded = FALSE;

  /* Set up obstacles. */

  for (i = 0; i < G_levelInfo[G_level].numObstacles; i++) {

    fails = 0;

    /* choose positions for all the obstacles. */
    do {
      fails++;
      x = RandFromRange(0,NUM_TILES_SIDETOSIDE-1);
      y = RandFromRange(0,NUM_TILES_UPANDDOWN-1);
      valid = ValidPos(OBSTACLE,x,y);
    } while (!valid && fails < 10);

    if (valid) {
      /* Record the initial position. */
      G_pipeLayout[x][y].type = OBSTACLE;
      G_pipeLayout[x][y].permanent = TRUE;
      G_pipeLayout[x][y].flooded = FALSE;
    }

  }

  /* Set up reservoirs. */

  for (i = 0; i < G_levelInfo[G_level].numReservoirs; i++) {

    fails = 0;

    /* choose positions for all the obstacles. */
    do {
      fails++;
      x = RandFromRange(0,NUM_TILES_SIDETOSIDE-1);
      y = RandFromRange(0,NUM_TILES_UPANDDOWN-1);
      valid = ValidPos(OBSTACLE,x,y);
    } while (!valid && fails < 10);

    if (valid) {
      /* Record the initial position. */
      G_pipeLayout[x][y].type =  RandFromRange(HORIZONTAL_RESERVOIR,VERTICAL_RESERVOIR);
      G_pipeLayout[x][y].permanent = TRUE;
      G_pipeLayout[x][y].flooded = FALSE;
    }

  }

  /* Set up bonus pipes. */

  for (i = 0; i < G_levelInfo[G_level].numBonusPipes; i++) {

    fails = 0;

    /* choose positions for all the obstacles. */
    do {
      fails++;
      x = RandFromRange(0,NUM_TILES_SIDETOSIDE-1);
      y = RandFromRange(0,NUM_TILES_UPANDDOWN-1);
      valid = ValidPos(OBSTACLE,x,y);
    } while (!valid && fails < 10);

    if (valid) {
      /* Record the initial position. */
      G_pipeLayout[x][y].type =  RandFromRange(HORIZONTAL_BONUS,VERTICAL_BONUS);
      G_pipeLayout[x][y].permanent = TRUE;
      G_pipeLayout[x][y].flooded = FALSE;
    }

  }

  /* Clear and redraw the playing area. */
  RedrawplayingArea();

  /* Blank out all the preview tiles. */
  for (y=0; y<NUM_TILES_PREVIEWED; y++) {
    G_previewLayout[y] = EMPTY;
  }

  /* Clear and redraw the playing area. */
  RedrawPreviewArea();

  /* Get the fluid ready to flow. */
  switch(pipeType) {
    case START_UP:
        G_fluidInfo.direction = UP;
        G_fluidInfo.Ypos = SHORT_LEN-1;
        G_fluidInfo.Xpos = SHORT_LEN;
      break;
    case START_DOWN:
        G_fluidInfo.direction = DOWN;
        G_fluidInfo.Ypos = LONG_LEN+1;
        G_fluidInfo.Xpos = SHORT_LEN;
      break;
    case START_LEFT:
        G_fluidInfo.direction = LEFT;
        G_fluidInfo.Ypos = SHORT_LEN;
        G_fluidInfo.Xpos = SHORT_LEN-1;
      break;
    case START_RIGHT:
        G_fluidInfo.direction = RIGHT;
        G_fluidInfo.Ypos = SHORT_LEN;
        G_fluidInfo.Xpos = LONG_LEN+1;
      break;
  }

  /* Haven't filled up any pipes yet. */
  G_fastFlow = FALSE;
  PipesInc(G_pipesFilled = 0);
  CrossInc(G_crosses = 0);

  /* reset the flowometer */
  FlowometerSet(0);

  /* Fill in random tiles in the preview area. */
  for (y=0; y<NUM_TILES_PREVIEWED; y++) {
    NextTile();
  }
}

/******************************************************************************

                                CreateAppShell

 ******************************************************************************/

Widget CreateAppShell(hierarchy,fg,bg)
  MrmHierarchy hierarchy;
  Pixel        fg,bg;
{
  Arg          arglist[MAX_ARGS];
  int          argcount;  
  MrmType      class;
  Cardinal     mrm_stat;
  Widget       app_shell;
  int          menubar_height;
  Pixmap       icon_pixmap;

  /* create the application icon pixmap */
  icon_pixmap = XCreatePixmapFromBitmapData(G_display,
                                            DefaultRootWindow(G_display),
                                            G_pipeBitmapBits,
                                            PIPETILE_SIZE,PIPETILE_SIZE,
                                            BlackPixel(G_display,0),
                                            WhitePixel(G_display,0),
                                            1);

  /* create shell for the main window */
  argcount = 0;
  XtSetArg(arglist[argcount],XmNx,200); argcount++;
  XtSetArg(arglist[argcount],XmNy,100); argcount++;
  XtSetArg(arglist[argcount],XmNiconPixmap,icon_pixmap); argcount++;
  XtSetArg(arglist[argcount],XmNwidth,PIPETILE_SIZE*5); argcount++;
  XtSetArg(arglist[argcount],XmNheight,PIPETILE_SIZE*3+MENU_BAR_HEIGHT); argcount++;
  app_shell = XtAppCreateShell("Pipe",NULL,applicationShellWidgetClass,
                               G_display,arglist,argcount);

  /* Get the main window widget */
  mrm_stat = MrmFetchWidget(hierarchy,"main_window",app_shell,
                                                    &G_main_wgt,&class);
  mrm_check(mrm_stat,"Unable to fetch main window.");

  /* make the drawing area appearunderneath the menu bar instead of behind it */
  XmMainWindowSetAreas(G_main_wgt,
                       G_menuBar_wgt,
                       NULL,NULL,NULL,
                       G_playingArea_wgt);

  /* manage the main window */
  XtManageChild(G_main_wgt);

  return(app_shell);

}

/******************************************************************************

                            CreateLevelOverDialog

 ******************************************************************************/

void CreateLevelOverDialog(hierarchy,app_shell)
  MrmHierarchy hierarchy;
  Widget       app_shell;
{
  Arg          arglist[MAX_ARGS];
  int          argcount;  
  MrmType      class;
  Cardinal     mrm_stat;
  Widget       lo_shell;

  /* create a shell to hold the "levelOver" dialog box */
  argcount = 0;
  XtSetArg(arglist[argcount],XmNx,330); argcount++;
  XtSetArg(arglist[argcount],XmNy,260); argcount++;
  lo_shell = XmCreateDialogShell(app_shell,"Level Over",arglist,argcount);

  /* Get the levelOver dialog box widget */
  mrm_stat = MrmFetchWidget(hierarchy,"level_over_dialog",lo_shell,&G_levelOver_wgt,&class);
  mrm_check(mrm_stat,"Unable to fetch 'Level Over' dialog box.");

  /* realize the "Level Over" dialog boxes' shell */
  XtRealizeWidget(lo_shell);
}

/******************************************************************************

                             CreatePreviewWindow

 ******************************************************************************/

void CreatePreviewWindow(hierarchy,app_shell)
  MrmHierarchy hierarchy;
  Widget       app_shell;
{
  Arg          arglist[MAX_ARGS];
  int          argcount;  
  MrmType      class;
  Cardinal     mrm_stat;
  Widget       prv_shell;

  /* create a shell to hold the preview queue */
  argcount = 0;
  prv_shell = XmCreateDialogShell(app_shell,"Next",arglist,argcount);

  /* Get the preview window widget */
  mrm_stat = MrmFetchWidget(hierarchy,"preview_window",prv_shell,&G_preview_wgt,&class);
  mrm_check(mrm_stat,"Unable to fetch preview window.");

  argcount = 0;
  XtSetArg(arglist[argcount],XmNwidth,PREVIEWAREA_WIDTH); argcount++;
  XtSetArg(arglist[argcount],XmNheight,PREVIEWAREA_HEIGHT); argcount++;
  XtSetArg(arglist[argcount],XmNx,100); argcount++;
  XtSetArg(arglist[argcount],XmNy,100); argcount++;
  XtSetValues(prv_shell,arglist,argcount);

  /* realize the preview shell */
  XtRealizeWidget(prv_shell);
}

/******************************************************************************

                              CreateStatusWindow

 ******************************************************************************/

void CreateStatusWindow(hierarchy,app_shell)
  MrmHierarchy hierarchy;
  Widget       app_shell;
{
  Arg          arglist[MAX_ARGS];
  int          argcount;  
  MrmType      class;
  Cardinal     mrm_stat;

  /* create a shell to hold the status information */
  argcount = 0;
  G_statusShell = XmCreateDialogShell(app_shell,"Status",arglist,argcount);

  /* Get the preview window widget */
  mrm_stat = MrmFetchWidget(hierarchy,"status_window",G_statusShell,&G_status_wgt,&class);
  mrm_check(mrm_stat,"Unable to fetch status window.");

  /* set the position and size */
  argcount = 0;
  XtSetArg(arglist[argcount],XmNx,250+PIPETILE_SIZE*5); argcount++;
  XtSetArg(arglist[argcount],XmNy,100); argcount++;
  XtSetValues(G_statusShell,arglist,argcount);

  /* realize the status shell */
  XtRealizeWidget(G_statusShell);
}


/******************************************************************************

                             CreateHighScoreWindow

 ******************************************************************************/

void CreateHighScoreWindow(hierarchy,app_shell)
  MrmHierarchy hierarchy;
  Widget       app_shell;
{
  Arg          arglist[MAX_ARGS];
  int          argcount;  
  MrmType      class;
  Cardinal     mrm_stat;
  Widget       hs_shell;

  /* create a shell to hold the status information */
  argcount = 0;
  XtSetArg(arglist[argcount],XmNx,200); argcount++;
  XtSetArg(arglist[argcount],XmNy,100); argcount++;
  XtSetArg(arglist[argcount],XmNallowShellResize,True); argcount++;
  hs_shell = XmCreateDialogShell(app_shell,"High Scores",arglist,argcount);

  /* Get the preview window widget */
  mrm_stat = MrmFetchWidget(hierarchy,"high_scores_window",hs_shell,&G_highScore_wgt,&class);
  mrm_check(mrm_stat,"Unable to fetch high score window.");

  /* realize the high scores shell */
  XtRealizeWidget(hs_shell);

  /* load the high scores window */
  UpdateHighScores(1);

}


/******************************************************************************

                               CreateHelpWindow

 ******************************************************************************/

void CreateHelpWindow(hierarchy,app_shell)
  MrmHierarchy  hierarchy;
  Widget        app_shell;
{
  Arg           arglist[MAX_ARGS];
  int           argcount;  
  MrmType       class;
  Cardinal      mrm_stat;
  Widget        help_shell;
  char         *helpText;
  char          help_filename[80];

  /* create a shell to hold the status information */
  argcount = 0;
  help_shell = XmCreateDialogShell(app_shell,"Pipe Help",arglist,argcount);

  /* Get the preview window widget */
  mrm_stat = MrmFetchWidget(hierarchy,"help_window",help_shell,&G_helpWindow_wgt,&class);
  mrm_check(mrm_stat,"Unable to fetch help window.");

  /* load and pop up the help window right away */
  strcpy(help_filename,HELP_LOC);
  strcat(help_filename,HELP_FILE);
  helpText = LoadText(help_filename);
  if (!helpText) helpText = "Could not load help file.\nBad installation?";
  XmTextSetString(G_helpText_wgt,helpText);

  /* set the width and height of the shell */
  argcount = 0;
  XtSetArg(arglist[argcount],XmNx,150); argcount++;
  XtSetArg(arglist[argcount],XmNy,150); argcount++;
  XtSetArg(arglist[argcount],XmNwidth,600); argcount++;
  XtSetArg(arglist[argcount],XmNheight,300); argcount++;
  XtSetValues(help_shell,arglist,argcount);  

  /* realize the help shell */
  XtRealizeWidget(help_shell);

}


/******************************************************************************

                                   SetupMrm

 ******************************************************************************/

MrmHierarchy SetupMrm(CBroutines,num_CBroutines)
  MRMRegisterArg      CBroutines;
  int                 num_CBroutines;
{
  MrmHierarchy        hierarchy;
  Cardinal            mrm_stat;
  char               *uid_filenames[2];

  uid_filenames[0] = (char *) malloc(80);
  uid_filenames[1] = (char *) malloc(80);

  strcpy(uid_filenames[0],UID_LOC);
  strcat(uid_filenames[0],PIPE_WIDGETS);
  strcpy(uid_filenames[1],UID_LOC);
  strcat(uid_filenames[1],PIPE_ICONS);

  /* Initialize the Motif Resource Manager */
  MrmInitialize();

  /* Open the file o' widgets */
  mrm_stat = MrmOpenHierarchy(2,uid_filenames,NULL,&hierarchy);
  mrm_check(mrm_stat,"The UID could not be opened.");

  /* Register the names to bind */
  mrm_stat = MrmRegisterNames(CBroutines,num_CBroutines);
  mrm_check(mrm_stat,"The callbacks could not be registered.");

  return(hierarchy);
}


/******************************************************************************

                                  PauseGameCB

 ******************************************************************************/

void PauseGameCB()
{
  Window window;

  window = XtWindow(XtParent(G_main_wgt));

  XIconifyWindow(G_display,window,DefaultScreen(G_display));
}


/******************************************************************************

                                  AbortGameCB

 ******************************************************************************/

void AbortGameCB()
{

  /* remove all the timeouts */

  if (G_fluidTimeOut) 
    XtRemoveTimeOut(G_fluidTimeOut);

  if (G_eraseTimeOut) 
    XtRemoveTimeOut(G_eraseTimeOut);

  if (G_fillresTimeOut) 
    XtRemoveTimeOut(G_fillresTimeOut);

  if (G_countdownTimeOut) 
    XtRemoveTimeOut(G_countdownTimeOut);


  GameOver();
  LevelOverCB();

}


/******************************************************************************

                                  QuitGameCB

 ******************************************************************************/

void QuitGameCB()
{
  exit(0);
}


/******************************************************************************

                                 StartGameCB

 ******************************************************************************/

void StartGameCB()
{
  /* assume starting at level 0 */
  G_level = G_startingLevel;
  LevelInc(0);
  ScoreInc(G_score = 0);

  /* start the level */
  StartLevel();

  /* Disable the start button while a game is underway */
  XtSetSensitive(G_start_wgt,False);

  /* Now that a game has tsarted, you can abort it. */
  XtSetSensitive(G_abort_wgt,True);

  /* New game, go it's no longer over. */
  G_gameOver = False;
}


/******************************************************************************

                                 Main Routine

 ******************************************************************************/

int main(argc,argv)
  unsigned int  argc;
  char          *argv[];
{
  MrmHierarchy  hierarchy;
  Pixel         fg_pxl,bg_pxl;
  
  /* Seed the random number generator */
  SEED_RAND;

  /* determine the plumber's name */
  G_username = cuserid(0);

  /* Initialize the Xwindows toolkit */
  XtToolkitInitialize();

  /* get the application context */
  G_appContext = XtCreateApplicationContext();

  /* Open a connection to the workstation */
  G_display = XtOpenDisplay(G_appContext,NULL,"Pipe","PipeClass",NULL,0,&argc,argv);
  check(G_display,"The display could not be opened.");

  /* Set up the Motif Resource Manager, open UID file, and register callbacks */
  hierarchy = SetupMrm(G_CBroutines,G_num_CBroutines);

  /* Disable event buffering while debugging. */
# ifdef DEBUG
    XSynchronize(G_display,1);
# endif

  G_screen = XDefaultScreenOfDisplay(G_display);

  /* Get the colors to be used for drawing lines. */
  GetColors(hierarchy,&fg_pxl,&bg_pxl);

  /* Create the windows */
  G_appShell = CreateAppShell(hierarchy,fg_pxl,bg_pxl);
  CreatePreviewWindow(hierarchy,G_appShell);
  CreateLevelOverDialog(hierarchy,G_appShell);
  CreateStatusWindow(hierarchy,G_appShell);
  CreateHighScoreWindow(hierarchy,G_appShell);
  CreateHelpWindow(hierarchy,G_appShell);

  /* set up the list of pipe types and information held in G_pipeInfo. */
  initPipeVec(hierarchy,fg_pxl,bg_pxl);

  /* create a graphics context for drawing */
  G_generic_GC = XCreateGC(G_display,DefaultRootWindow(G_display),0,0);
  XSetState(G_display,G_generic_GC,fg_pxl,bg_pxl,GXcopy,AllPlanes);

  /* create the pixmaps that hold game image when occluded */
  CreateShadowPixmaps();

  /* Create levels (and buttons to select starting level). */
  CreateLevels();

  /* Create a snazzy pointer in the shape of a pipe wrench (NOT!) */
  SetupPointer(hierarchy,fg_pxl,bg_pxl);

  /* disable input to the playing area */
  G_playingAreaSensitive = False;

  /* you can't abort a game that hasn't started yet... */
  XtSetSensitive(G_abort_wgt,False);

  /* don't allow the fluid to be accelerated between levels */
  XtSetSensitive(G_fluid_wgt,False);

  /* clear the screen so it looks OK while there's no game going on... */
  ClearScreen();

  /* realize the application shell */
  XtRealizeWidget(G_appShell);
  XtManageChild(G_preview_wgt);
  XtManageChild(G_status_wgt);

  /* Enter the event-handling infinite loop */
  XtAppMainLoop(G_appContext);
}
