/* tkImgTGA.c: TCL/Tk Targa TGA photo image format handler
**
** Written and Copyright (C) 1996 by Michael J. Gourlay
**
** PROVIDED AS IS.  NO WARRENTIES, EXPRESS OR IMPLIED.
**
** This code won't be used by tkmorph to load TGA images because the
** photo image type doesn't handle the alpha channel.  I wrote this code
** to see how to add more photo image formats.
*/

#include <stdio.h>
#include <stdlib.h>

#include <tcl.h>
#include <tk.h>

#include "tkImgTGA.h"
#include "rgba_image.h"
#include "tga.h"




/* matchFileTGA: TCL/Tk Photo image file match procedure for Targa images
**
** Argument list is as dictated by the TCL/Tk photo image format.
**
** Although the man pages do not specify the following information, I
** assume it is true.  John Ousterhout and his minions ought to be
** spanked for such poor documentation.  Documentation is TCL's cash
** cow.  Why pay for it when it's bad?
**
** Assumptions:
**  The file is already open, and is pointing to the beginning.
**  The file is rewound by the caller (i.e., the file position is set to
**    the beginning) after this routine is called by Tk (which implies a
**    seek'able file).
*/
static int
matchFileTGA(FILE *fP, char *filename, char *format_str,
             int *widthP, int *heightP)
{
  int tga_return;
  tga_hdr_t tga_hdr;
  RgbaImageT img;

  if(tga_return = load_tga_header(&tga_hdr, &img, fP)) {
    /* Failed to positively identify this as a TGA image file */
    return(0);
  } else {
    *widthP = img.ncols;
    *heightP = img.nrows;

    return(1);
  }
}




/* readFileTGA: TCL/Tk Photo image file read procedure for Targa images
** Assumption (not documented explicitly):
**   The file is open and pointing to the beginning.
**
** Argument list is as dictated by the TCL/Tk photo image format.
**
** width, height: dimensions of the portion of the incoming image being used
** src_x, src_y: upper left corner of the portion of the incoming image
** dest_x, dest_y: where, in the destination Tk Photo Image, to put
**                 the incoming image upper left corner
**
*/
static int
readFileTGA(Tcl_Interp *interp, FILE *fP, char *filename,
            char *format_str, Tk_PhotoHandle photo_img_H,
            int dest_x, int dest_y,
            int width, int height,
            int src_x, int src_y)
{
  int tga_return;
  tga_hdr_t tga_hdr;
  RgbaImageT rgba_img;

  if(tga_return = load_tga_header(&tga_hdr, &rgba_img, fP)) {
    /* Failed to positively identify this as a TGA image file */
    Tcl_AppendResult(interp, "could not read TGA header from file '",
                     filename, "'", NULL);
    return TCL_ERROR;

  } else {
    Tk_PhotoImageBlock block;
    int yi;

    /* Make sure that the requested width and height fit within the
    ** incoming image.
    */
    if ((src_x + width) > rgba_img.ncols) {
        width = rgba_img.ncols - src_x;
    }
    if ((src_y + height) > rgba_img.nrows) {
        height = rgba_img.nrows - src_y;
    }

    /* If requested subimage has no area then we're done */
    if ((width <= 0) || (height <= 0)
        || (src_x >= rgba_img.ncols) || (src_y >= rgba_img.nrows)) {
        return TCL_OK;
    }

    /* Make sure Photo Image is at least large enough for incoming
    ** image portion
    */
    Tk_PhotoExpand(photo_img_H, dest_x + width, dest_x + height);

    /* Allocate new memory for the new image planes */
    if(rgbaImageAlloc(&rgba_img, "readFileTGA")) return TCL_ERROR;

    /* Load the TGA image */
    if(load_tga_image(&tga_hdr, &rgba_img, fP)) {
      Tcl_AppendResult(interp, "error reading TGA image file '",
                       filename, "'", NULL);
      rgbaImageFree(&rgba_img);
      return TCL_ERROR;
    }

    /* Set the Tk Photo Image block parameters for the Tk_PhotoPutBlock */
    /* pixelSize: increment to the horizontally adjacent pixel */
    block.pixelSize = 1; /* NOT REALLY PIXEL SIZE */
    /* pitch: increment the the vertically adjacent pixel */
    block.pitch = block.pixelSize * rgba_img.ncols;
    /* offset: offsets to red, green, and blue pixels */
    block.offset[0] = 0;
    block.offset[1] = rgba_img.ncols * rgba_img.nrows;
    block.offset[2] = 2 * block.offset[1];
    block.width = rgba_img.ncols;
    block.height = 1;


    /* Put each line of the incoming image into the Tk Photo Image */
    for(yi=0; yi<height; yi++) {
      block.pixelPtr = &(rgba_img.ri[(src_y+yi) * rgba_img.ncols + src_x]);
      Tk_PhotoPutBlock(photo_img_H, &block, dest_x, dest_y+yi, width, 1);
    }

    /* Free the rgba image */
    rgbaImageFree(&rgba_img);
  }

  return TCL_OK;
}




/* Create the Tk TGA photo image handler */
void
createTGAImageFormat()
{
  Tk_PhotoImageFormat format;


  format.name = "tga";
  format.fileMatchProc = matchFileTGA;
  format.stringMatchProc = NULL;
  format.fileReadProc = readFileTGA;
  format.stringReadProc = NULL;
  format.fileWriteProc = NULL;
  format.stringWriteProc = NULL;

  Tk_CreatePhotoImageFormat(&format);
}
