/*
 *   libdi - CD Audio Player Device Interface Library
 *
 *   Copyright (C) 1993-1996  Ti Kan
 *   E-mail: ti@amb.org
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   Siemens Nixdorf Informationssysteme SINIX support
 *
 *   Contributing author: Eckhard Einert
 *   E-Mail: einert.pad@sni.de
 *
 *   This software fragment contains code that interfaces the CD player
 *   application to the SNI SINIX operating system.  The names "SNI" and
 *   "SINIX" are used here for identification purposes only.
 */
#ifndef LINT
static char *_os_sinix_c_ident_ = "@(#)os_sinix.c	6.8 96/01/07";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "libdi_d/libdi.h"
#include "libdi_d/scsipt.h"

#if defined(SNI) && defined(DI_SCSIPT) && !defined(DEMO_ONLY)

extern appdata_t	app_data;
extern bool_t		scsipt_notrom_error;
extern FILE		*errfp;
extern di_client_t	*di_clinfo;

STATIC int		pthru_fd = -1;	/* Passthrough device file desc */


/*
 * pthru_send
 *	Build SCSI CDB and send command to the device.
 *
 * Args:
 *	opcode - SCSI command opcode
 *	addr - The "address" portion of the SCSI CDB
 *	buf - Pointer to data buffer
 *	size - Number of bytes to transfer
 *	rsvd - The "reserved" portion of the SCSI CDB
 *	length - The "length" portion of the SCSI CDB
 *	param - The "param" portion of the SCSI CDB
 *	control - The "control" portion of the SCSI CDB
 *	rw - Data transfer direction flag (READ_OP or WRITE_OP)
 *	prnerr - Whether an error message should be displayed
 *		 when a command fails
 *
 * Return:
 *	TRUE - command completed successfully
 *	FALSE - command failed
 */
bool_t
pthru_send(
	byte_t		opcode,
	word32_t	addr,
	byte_t		*buf,
	word32_t	size,
	byte_t		rsvd,
	word32_t	length,
	byte_t		param,
	byte_t		control,
	byte_t		rw,
	bool_t		prnerr
)
{
	struct scsictl	req;
	int		cmdlen;

	if (pthru_fd < 0 || scsipt_notrom_error)
		return FALSE;

	memset(&req, 0, sizeof(req));

	req.scs_addr = (int)0;
	req.scs_bytes = 0;
	req.scs_flags = 0;
	req.scs_status = 0;
	req.scs_spare = 0;

	/* Set up scsictl */

	if (buf != NULL && size != 0) {
		req.scs_flags |= (rw == READ_OP) ? SCSFDATAOUT : SCSFNODATA;
		req.scs_addr = (caddr_t) buf;
		req.scs_bytes = (ulong) size;
	}

	/* set up SCSI CDB */
	switch (opcode & 0xf0) {
	case 0xa0:
	case 0xe0:
		/* 12-byte commands */
		req.scsi0 = opcode;
		req.scsi1 = param;
		req.scsi2 = (addr >> 24) & 0xff;
		req.scsi3 = (addr >> 16) & 0xff;
		req.scsi4 = (addr >> 8) & 0xff;
		req.scsi5 = (addr & 0xff);
		req.scsi6 = (length >> 24) & 0xff;
		req.scsi7 = (length >> 16) & 0xff;
		req.scsi8 = (length >> 8) & 0xff;
		req.scsi9 = length & 0xff;
		req.scsi10 = rsvd;
		req.scsi11 = control;

		cmdlen = 12;
		break;

	case 0xc0:
	case 0xd0:
	case 0x20:
	case 0x30:
	case 0x40:
		/* 10-byte commands */
		req.scsi0 = opcode;
		req.scsi1 = param;
		req.scsi2 = (addr >> 24) & 0xff;
		req.scsi3 = (addr >> 16) & 0xff;
		req.scsi4 = (addr >> 8) & 0xff;
		req.scsi5 = addr & 0xff;
		req.scsi6 = rsvd;
		req.scsi7 = (length >> 8) & 0xff;
		req.scsi8 = length & 0xff;
		req.scsi9 = control;
		switch (opcode) {
		case 0x42:
			req.scs_flags = SCSFDATAOUT;
			break;
		case 0x43:
			req.scs_flags = SCSFDATAOUT;
			break;
		default:
			/* 47 */
			req.scs_flags = SCSFNODATA;
		}	

		cmdlen = 10;
		break;

	case 0x00:
		req.scs_flags = SCSFNODATA;
	case 0x10:
		/* 6-byte commands */
		req.scsi0 = opcode;
		req.scsi1 = param;
		req.scsi2 = (addr >> 8) & 0xff;
		req.scsi3 = addr & 0xff;
		req.scsi4 = length & 0xff;
		req.scsi5 = control;
		switch (opcode) {
		case 0x15:
			req.scs_flags = SCSFDATAIN;
			break;
		case 0x12:
			req.scs_flags = SCSFDATAOUT;
			break;
		case 0x1a:
			req.scs_flags = SCSFDATAOUT;
			break;
		default:
			/* 1b,2e */
			req.scs_flags = SCSFNODATA;
		}	

		cmdlen = 6;
		break;

	default:
		if (app_data.scsierr_msg && prnerr)
			fprintf(errfp, "0x%02x: Unknown SCSI opcode\n",
				opcode);
		return FALSE;
	}

	DBGDUMP("SCSI CDB bytes", &req.scsi0, cmdlen);

	/* Send the command down via the "pass-through" interface */
	if (ioctl(pthru_fd, IOSSCSICMD, &req) < 0) {
		if (app_data.scsierr_msg && prnerr)
			perror("IOSSCSICMD ioctl failed");
		return FALSE;
	}
	
	return TRUE;
}


/*
 * pthru_open
 *	Open SCSI pass-through device
 *
 * Args:
 *	path - device path name string
 *
 * Return:
 *	TRUE - open successful
 *	FALSE - open failed
 */
bool_t
pthru_open(char *path)
{
	struct stat	stbuf;
	char		errstr[ERR_BUF_SZ];

	/* Check for validity of device node */
	if (stat(path, &stbuf) < 0) {
		sprintf(errstr, app_data.str_staterr, path);
		DI_FATAL(errstr);
		return FALSE;
	}

	if (!S_ISCHR(stbuf.st_mode)) {
		sprintf(errstr, app_data.str_noderr, path);
		DI_FATAL(errstr);
		return FALSE;
	}

	if ((pthru_fd = open(path, O_RDONLY)) < 0) {
		DBGPRN(errfp, "Cannot open %s: errno=%d\n", path, errno);
		return FALSE;
	}

	return TRUE;
}


/*
 * pthru_close
 *	Close SCSI pass-through device
 *
 * Args:
 *	Nothing.
 *
 * Return:
 *	Nothing.
 */
void
pthru_close(void)
{
	if (pthru_fd >= 0) {
		close(pthru_fd);
		pthru_fd = -1;
	}
}


/*
 * pthru_vers
 *	Return OS Interface Module version string
 *
 * Args:
 *	Nothing.
 *
 * Return:
 *	Module version text string.
 */
char *
pthru_vers(void)
{
	return ("OS Interface module for SNI SINIX\n");
}

#endif	/* SNI DI_SCRIPT DEMO_ONLY */

