/*
 *   libdi - CD Audio Player Device Interface Library
 *
 *   Copyright (C) 1993-1996  Ti Kan
 *   E-mail: ti@amb.org
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   BSDI BSD/OS support
 *
 *   Contributing author: Danny Braniss
 *   E-Mail: danny@cs.huji.ac.il
 *
 *   This software fragment contains code that interfaces the CD player
 *   application to the BSDI BSD/OS (version 2.0 or later) operating system.
 */
#ifndef LINT
static char *_os_bsdi_c_ident_ = "@(#)os_bsdi.c	6.13 96/01/07";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "libdi_d/libdi.h"
#include "libdi_d/scsipt.h"

#if defined(bsdi) && defined(DI_SCSIPT) && !defined(DEMO_ONLY)

extern appdata_t	app_data;
extern bool_t		scsipt_notrom_error;
extern FILE		*errfp;
extern di_client_t	*di_clinfo;

STATIC int		pthru_fd = -1;


/*
 * pthru_send
 *	Build SCSI CDB and send command to the device.
 *
 * Args:
 *	opcode - SCSI command opcode
 *	addr - The "address" portion of the SCSI CDB
 *	buf - Pointer to data buffer
 *	size - Number of bytes to transfer
 *	rsvd - The "reserved" portion of the SCSI CDB
 *	length - The "length" portion of the SCSI CDB
 *	param - The "param" portion of the SCSI CDB
 *	control - The "control" portion of the SCSI CDB
 *	rw - Data transfer direction flag (READ_OP or WRITE_OP)
 *	prnerr - Whether an error message should be displayed
 *		 when a command fails
 *
 * Return:
 *	TRUE - command completed successfully
 *	FALSE - command failed
 */
bool_t
pthru_send(
	byte_t		opcode,
	word32_t	addr,
	byte_t		*buf,
	word32_t	size,
	byte_t		rsvd,
	word32_t	length,
	byte_t		param,
	byte_t		control,
	byte_t		rw,
	bool_t		prnerr
)
{
	struct scsi_cdb	cdb;
	int		blen,
			n,
			len;
	byte_t		ch = 0;

	if (pthru_fd < 0 || scsipt_notrom_error)
		return FALSE;

	memset(&cdb, 0, sizeof(struct scsi_cdb));

	/* Set up SCSI CDB */
	switch (opcode & 0xf0) {
	case 0xa0:
	case 0xe0:
		/* 12-byte commands */
		cdb.cdb_bytes[0] = opcode;
		cdb.cdb_bytes[1] = param;
		cdb.cdb_bytes[2] = (addr >> 24) & 0xff;
		cdb.cdb_bytes[3] = (addr >> 16) & 0xff;
		cdb.cdb_bytes[4] = (addr >> 8) & 0xff;
		cdb.cdb_bytes[5] = (addr & 0xff);
		cdb.cdb_bytes[6] = (length >> 24) & 0xff;
		cdb.cdb_bytes[7] = (length >> 16) & 0xff;
		cdb.cdb_bytes[8] = (length >> 8) & 0xff;
		cdb.cdb_bytes[9] = length & 0xff;
		cdb.cdb_bytes[10] = rsvd;
		cdb.cdb_bytes[11] = control;

		len = 12;
		break;

	case 0xc0:
	case 0xd0:
	case 0x20:
	case 0x30:
	case 0x40:
		cdb.cdb_bytes[0] = opcode;
		cdb.cdb_bytes[1] = param;
		cdb.cdb_bytes[2] = (addr >> 24) & 0xff;
		cdb.cdb_bytes[3] = (addr >> 16) & 0xff;
		cdb.cdb_bytes[4] = (addr >> 8) & 0xff;
		cdb.cdb_bytes[5] = addr & 0xff;
		cdb.cdb_bytes[6] = rsvd;
		cdb.cdb_bytes[7] = (length >> 8) & 0xff;
		cdb.cdb_bytes[8] = length & 0xff;
		cdb.cdb_bytes[9] = control;

		len = 10;
		break;

	case 0x00:
	case 0x10:
		/* 6-byte commands */
		cdb.cdb_bytes[0] = opcode;
		cdb.cdb_bytes[1] = param;
		cdb.cdb_bytes[2] = (addr >> 8) & 0xff;
		cdb.cdb_bytes[3] = addr & 0xff;
		cdb.cdb_bytes[4] = length & 0xff;
		cdb.cdb_bytes[5] = control;

		len = 6;
		break;

	default:
		if (app_data.scsierr_msg && prnerr)
			fprintf(errfp, "0x%02x: Unknown SCSI opcode\n",
				opcode);
		return FALSE;
	}

	DBGDUMP("SCSI CDB bytes", (byte_t *) cdb.cdb_bytes, len);

	/* Make sure that this command is supported by the driver */
	n = opcode;
	if (ioctl(pthru_fd, SDIOCADDCOMMAND, &n) < 0) {
		if (app_data.scsierr_msg && prnerr)
			perror("SDIOCADDCOMMAND ioctl failed");
		return FALSE;
	}

	/* Send the command down via the "pass-through" interface */
	if (ioctl(pthru_fd, SDIOCSCSICOMMAND, (struct scsi_cdb *) &cdb) < 0) {
		if (app_data.scsierr_msg && prnerr)
			perror("SDIOCSCSICOMMAND ioctl failed");
		return FALSE;
	}

	if (size == 0) {
		blen = 1;
		buf = &ch;
	}
	else {
		blen = size;
	}

	switch (rw) {
	case READ_OP:
		n = read(pthru_fd, buf, blen);
		if (n != blen && n != size) {
			if (app_data.scsierr_msg && prnerr)
				perror("data read failed");
			return FALSE;
		}
		break;
	case WRITE_OP:
		n = write(pthru_fd, buf, blen);
		if (n != blen && n != size) {
			if (app_data.scsierr_msg && prnerr)
				perror("data write failed");
			return FALSE;
		}
		break;
	default:
		break;
	}

	return TRUE;
}


/*
 * pthru_open
 *	Open SCSI pass-through device
 *
 * Args:
 *	path - device path name string
 *
 * Return:
 *	TRUE - open successful
 *	FALSE - open failed
 */
bool_t
pthru_open(char *path)
{
	struct stat	stbuf;
	char		errstr[ERR_BUF_SZ];
	int		val;

	/* Check for validity of device node */
	if (stat(path, &stbuf) < 0) {
		sprintf(errstr, app_data.str_staterr, path);
		DI_FATAL(errstr);
		return FALSE;
	}
	if (!S_ISCHR(stbuf.st_mode)) {
		sprintf(errstr, app_data.str_noderr, path);
		DI_FATAL(errstr);
		return FALSE;
	}

	if ((pthru_fd = open(path, O_RDWR | O_NONBLOCK)) < 0 &&
	    (pthru_fd = open(path, O_RDONLY | O_NONBLOCK)) < 0) {
		DBGPRN(errfp, "Cannot open %s: errno=%d\n", path, errno);
		return FALSE;
	}

	/* Query if the driver is already in "format mode" */
	val = 0;
	if (ioctl(pthru_fd, SDIOCGFORMAT, &val) < 0) {
		perror("SDIOCGFORMAT ioctl failed");
		return FALSE;
	}

	if (val != 0) {
		DBGPRN(errfp, "Device %s is in use by process %d\n",
			path, val);
		return FALSE;
	}

	/* Put the driver in "format mode" for SCSI pass-through */
	val = 1;
	if (ioctl(pthru_fd, SDIOCSFORMAT, &val) < 0) {
		perror("SDIOCSFORMAT ioctl failed");
		return FALSE;
	}

	return TRUE;
}


/*
 * pthru_close
 *	Close SCSI pass-through device
 *
 * Args:
 *	Nothing.
 *
 * Return:
 *	Nothing.
 */
void
pthru_close(void)
{
	int	val;

	if (pthru_fd >= 0) {
		val = 0;

		/* Put the driver back to normal mode */
		ioctl(pthru_fd, SDIOCSFORMAT, &val);

		close(pthru_fd);
		pthru_fd = -1;
	}
}


/*
 * pthru_vers
 *	Return OS Interface Module version string
 *
 * Args:
 *	Nothing.
 *
 * Return:
 *	Module version text string.
 */
char *
pthru_vers(void)
{
	return ("OS Interface module for BSDI BSD/OS\n");
}

#endif	/* bsdi DI_SCSIPT DEMO_ONLY */

