/*
** lock.c: this module provides a simple locking mechanism
**
**  	   LockScoreFile(filename) locks the given file. returns -1 on error.
**         UnlockScoreFile(filename) unlocks the given file.
**  	   LockFile(fd) locks the given file descriptor using fcntl.
**  	   UnlockFile(fd) unlocks the given file descriptor.
**
** the lockfilename is built from the given parameter by adding ".lock" at
** the end of it. the callers process id will be written in the file.
** when the file is unlocked again the pid will be read and checked
** before the file is unlinked.
**
** LockFile and UnlockFile use fcntl on an open file descriptor.
**
** Copyright (C) 1993, Gerald Vogt <vogt@isa.de>
** 
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
*/

#include <stdio.h>

#ifdef VMS
#include "fcntl.h"
#else
#include <fcntl.h>
#include <malloc.h>
#endif

#include "xblockbuster.h"
#include "lock.h"

#define MAXTRY 30

#ifdef LOCK_FILE
char	*CreateLockFileName(scorefile)
char	*scorefile;
{
    char    *buffer;

    buffer = malloc(strlen(scorefile) + 6);

    if (buffer != (char *) NULL) {
	strcpy(buffer, scorefile);
	strcat(buffer, ".lock");
    }

    return(buffer);
}

int LockScoreFile(scorefile)
char	*scorefile;
{
    char    *filename;
    int	    ret;
    char    string[10];
    int	    tryCount;
#ifndef VMS
	extern char *sys_errlist[];
	extern int errno;
#endif

    filename = CreateLockFileName(scorefile);
   
    if (filename == (char *) NULL)
      return(-1);

    tryCount = 0;

    do {
	ret = open(filename, O_WRONLY | O_CREAT | O_EXCL, 0664);
	tryCount++;
	
	if (ret == -1) {
#ifndef VMS
		fprintf(stderr,"Could not open %s:%s\n",filename,sys_errlist[errno]);
#else
		fprintf(stderr,"Could not open %s\n",filename);
#endif
	    if (tryCount == MAXTRY) {
		fprintf(stderr, "Giving up.\n");
		free(filename);
		return(-1);
	    }
	    sleep(1);
	}
    } while (ret == -1);

    sprintf(string, "%d", getpid());
    write(ret, string, strlen(string) + 1);

    close(ret);
    free(filename);
    return(0);
}
      
void	UnlockScoreFile(scorefile)
char	*scorefile;
{
    char    *filename;
    int	    ret, len;
    char    string[10];
	extern int atoi();

    filename = CreateLockFileName(scorefile);
    
    if (filename == (char *) NULL)
      return;

    ret = open(filename, O_RDONLY);

    if (ret == -1) {
	perror("open score lock file");
	free(filename);
	return;
    }

    len = read(ret, string, 10);

    if (len == -1) {
	perror("reading score lock file");
	close(ret);
	free(filename);
	return;
    }

    len = atoi(string);

    close(ret);

    if (len == getpid())
      unlink(filename);
    else
      fprintf(stderr, "Score lock pid does not match\n");

    free(filename);
}
#endif

#ifdef LOCK_FCNTL
int LockFile(fd)
int fd;
{
    int	    ret;
    int	    tryCount;
    struct flock fl;

    tryCount = 0;

    do {
	fl.l_type = F_WRLCK;
	fl.l_whence = 0;
	fl.l_start = 0L;
	fl.l_len = 0L;
	
	ret = fcntl(fd, F_SETLK, &fl);

	tryCount++;
	
	if (ret == -1) {
	    perror("fcntl");

	    if (tryCount == MAXTRY) {
		fprintf(stderr, "Giving up.\n");
		return(-1);
	    }
	    sleep(1);
	}
    } while (ret == -1);

    return(0);
}
      
void	UnlockFile(fd)
int 	fd;
{
    int	    ret;
    struct flock fl;

    fl.l_type = F_UNLCK;
    fl.l_whence = 0;
    fl.l_start = 0L;
    fl.l_len = 0L;

    ret = fcntl(fd, F_SETLK, &fl);

    if (ret == -1) {
	perror("fcntl");
    }
}
#endif
