/*
 * main.c -- main routine for GhostView-VMS.
 * Copyright (C) 1995  Johannes Plass
 *   Author: Johannes Plass
 *           Department of Physics
 *           Johannes-Gutenberg University, Mainz, Germany
 * Internet: plass@dipmza.physik.uni-mainz.de
*/

/*
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
*/

#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef VMS
#   include <X11_DIRECTORY/Intrinsic.h>
#   include <X11_DIRECTORY/cursorfont.h>
#   include <X11_DIRECTORY/StringDefs.h>
#   include <X11_DIRECTORY/Shell.h>
#   include <XAW_DIRECTORY/Cardinals.h>
#   if defined(USE_XAW3D) && defined(USE_FORM3D)
#   include <XAW_DIRECTORY/ThreeD.h>
#      ifdef _FORM3D_LOCAL_
#         include "Form3d.h"
#      else
#         include <XAW_DIRECTORY/Form3d.h>
#      endif
#   endif
#   include <XAW_DIRECTORY/Form.h>
#   include <XAW_DIRECTORY/Viewport.h>
#   include <XAW_DIRECTORY/Box.h>
#   include <XAW_DIRECTORY/MenuButton.h>
#   include <XAW_DIRECTORY/SimpleMenu.h>
#   include <XAW_DIRECTORY/SmeBSB.h>
#   include <XAW_DIRECTORY/SmeLine.h>
#   include <XAW_DIRECTORY/Label.h>
#   include <XAW_DIRECTORY/AsciiText.h>
#   include <XAW_DIRECTORY/Command.h>
#   include <XAW_DIRECTORY/Panner.h>
#   include <XAW_DIRECTORY/Xawinit.h>
#   ifdef _LAYOUT_LOCAL_
#      include "Layout.h"
#   else
#      include <XAW_DIRECTORY/Layout.h>
#   endif
#   ifdef _FILESEL_LOCAL_
#      include "Filesel.h"
#   else
#      include <XAW_DIRECTORY/Filesel.h>
#   endif
#   include <unixlib.h>
#else
#   include <X11/Intrinsic.h>
#   include <X11/cursorfont.h>
#   include <X11/StringDefs.h>
#   include <X11/Shell.h>
#   include <X11/Xaw/Cardinals.h>
#   if defined(USE_XAW3D) && defined(USE_FORM3D)
#      ifdef _FORM3D_LOCAL_
#         include "Form3d.h"
#      else
#         include <X11/Xaw/Form3d.h>
#      endif
#   endif
#   include <X11/Xaw/Form.h>
#   include <X11/Xaw/Viewport.h>
#   include <X11/Xaw/Box.h>
#   include <X11/Xaw/MenuButton.h>
#   include <X11/Xaw/SimpleMenu.h>
#   include <X11/Xaw/SmeBSB.h>
#   include <X11/Xaw/SmeLine.h>
#   include <X11/Xaw/Label.h>
#   include <X11/Xaw/AsciiText.h>
#   include <X11/Xaw/Command.h>
#   include <X11/Xaw/Panner.h>
#   include <X11/Xaw/Xawinit.h>
extern char *getenv();
#endif

/*
#define MESSAGES
#define MESSAGE_NO_ESC
*/
#include "message.h"

#include "gv.h"
#include "dialog.h"
#include "error.h"
#include "note.h"
#include "info.h"
#include "options.h"
#include "popup.h"
#include "ps.h"
#include "version.h"

#define GV_DEFAULT_VERSION "?"

static XtResource resources[] = {
    {"autoCenter", "AutoCenter", XtRBoolean, sizeof(Boolean),
     XtOffsetOf(AppResources, auto_center), XtRImmediate, (XtPointer)True},
    {"minimumMagstep", "minimumMagstep", XtRInt, sizeof(int),
     XtOffsetOf(AppResources, minimum_magstep), XtRImmediate, (XtPointer)-5},
    {"maximumMagstep", "maximumMagstep", XtRInt, sizeof(int),
     XtOffsetOf(AppResources, maximum_magstep), XtRImmediate, (XtPointer)5},
    {"magstep", "MagstRESET()ep", XtRInt, sizeof(int),
     XtOffsetOf(AppResources, magstep), XtRImmediate, (XtPointer)0},
    {"orientation", "Orientation", XtRPageOrientation,
     sizeof(XtPageOrientation), XtOffsetOf(AppResources, orientation),
     XtRImmediate, (XtPointer) XtPageOrientationPortrait},
    {"page", "Page", XtRString, sizeof(String),
     XtOffsetOf(AppResources, page), XtRImmediate, NULL},
    {"pageMedia", "PageMedia", XtRString, sizeof(String),
     XtOffsetOf(AppResources, pagemedia), XtRImmediate, "A4"},
    {"forceOrientation", "Force", XtRBoolean, sizeof(Boolean),
     XtOffsetOf(AppResources, force_orientation), XtRImmediate,
     (XtPointer)False},
    {"forcePageMedia", "Force", XtRBoolean, sizeof(Boolean),
     XtOffsetOf(AppResources, force_pagemedia), XtRImmediate, (XtPointer)False},
#ifdef USE_SWAP_LANDSCAPE
    {"swapLandscape", "SwapLandscape", XtRBoolean, sizeof(Boolean),
     XtOffsetOf(AppResources, swap_landscape), XtRImmediate, (XtPointer)False},
#endif
#ifndef VMS
# if defined(SVR4) || defined(SYSV) || defined(USG)
    {"printerVariable", "PrinterVariable", XtRString, sizeof(String),
     XtOffsetOf(AppResources, printer_variable), XtRImmediate, "LPDEST"},
# else
    {"printerVariable", "PrinterVariable", XtRString, sizeof(String),
     XtOffsetOf(AppResources, printer_variable), XtRImmediate, "PRINTER"},
# endif
    {"defaultPrinter", "DefaultPrinter", XtRString, sizeof(String),
     XtOffsetOf(AppResources, default_printer), XtRImmediate, NULL},
#endif /* VMS */
    {"printCommand", "PrintCommand", XtRString, sizeof(String),
     XtOffsetOf(AppResources, print_command), XtRImmediate,"?"},
    {"scratchDir", "ScratchDir", XtRString, sizeof(String),
     XtOffsetOf(AppResources, scratch_dir), XtRImmediate,"?"},
    {"defaultSaveDir", "DefaultSaveDir", XtRString, sizeof(String),
     XtOffsetOf(AppResources, default_save_dir), XtRImmediate,"?"},
    {"confirmPrint", "ConfirmPrint", XtRBoolean, sizeof(Boolean),
     XtOffsetOf(AppResources, confirm_print), XtRImmediate, (XtPointer)True},
    {"version", "Version", XtRString, sizeof(String),
     XtOffsetOf(AppResources, version), XtRImmediate,GV_DEFAULT_VERSION},
    {"autoResize", "AutoResize", XtRBoolean, sizeof(Boolean),
     XtOffsetOf(AppResources, auto_resize), XtRImmediate, (XtPointer)True},
    {"maximumWidth", "MaximumWidth", XtRString, sizeof(String),
     XtOffsetOf(AppResources, maximum_width), XtRImmediate, (XtPointer)"screen-20"},
    {"maximumHeight", "MaximumHeight", XtRString, sizeof(String),
     XtOffsetOf(AppResources, maximum_height), XtRImmediate, (XtPointer)"screen-44"},
    {"selectedBitmap", "SelectedBitmap", XtRBitmap, sizeof(Pixmap),
     XtOffsetOf(AppResources,selected_bitmap), XtRImmediate, (XtPointer)None},
    {"documentBitmap", "DocumentBitmap", XtRBitmap, sizeof(Pixmap),
     XtOffsetOf(AppResources,document_bitmap), XtRImmediate, (XtPointer)None},
    {"forcedBitmap", "ForcedBitmap", XtRBitmap, sizeof(Pixmap),
     XtOffsetOf(AppResources,forced_bitmap),   XtRImmediate, (XtPointer)None},
    {"locatorFormat", "LocatorFormat", XtRString, sizeof(String),
     XtOffsetOf(AppResources,locator_format),   XtRImmediate, (XtPointer)"%d x %d"},
};

static XrmOptionDescRec options[] = {
 { "-monochrome"	, "*Ghostview.palette"		, XrmoptionNoArg	, "Monochrome"	},
 { "-grayscale"		, "*Ghostview.palette"		, XrmoptionNoArg	, "Grayscale"	},
 { "-color"		, "*Ghostview.palette"		, XrmoptionNoArg	, "Color"	},
 { "-page"		, ".page"			, XrmoptionSepArg	, NULL		},
 { "-center"		, ".autoCenter"			, XrmoptionNoArg	, "True"	},
 { "-nocenter"		, ".autoCenter"			, XrmoptionNoArg	, "False"	},
 { "-quiet"		, "*Ghostview.quiet"		, XrmoptionNoArg	, "True"	},
 { "-noquiet"		, "*Ghostview.quiet"		, XrmoptionNoArg	, "False"	},
 { "-safer"		, "*Ghostview.safer"		, XrmoptionNoArg	, "True"	},
 { "-nosafer"		, "*Ghostview.safer"		, XrmoptionNoArg	, "False"	},
 { "-arguments"		, "*Ghostview.arguments"	, XrmoptionSepArg	, NULL		},
 { "-xdpi"		, "*Ghostview.xdpi"		, XrmoptionSepArg	, NULL		},
 { "-ydpi"		, "*Ghostview.ydpi"		, XrmoptionSepArg	, NULL		},
 { "-dpi"		, "*Ghostview.Resolution"	, XrmoptionSepArg	, NULL		},
 { "-resolution"	, "*Ghostview.Resolution"	, XrmoptionSepArg	, NULL		},
 { "-letter"            , ".pageMedia"                  , XrmoptionNoArg        , "Letter"      },
 { "-tabloid"           , ".pageMedia"                  , XrmoptionNoArg        , "Tabloid"     },
 { "-ledger"            , ".pageMedia"                  , XrmoptionNoArg        , "Ledger"      },
 { "-legal"             , ".pageMedia"                  , XrmoptionNoArg        , "Legal"       },
 { "-statement"         , ".pageMedia"                  , XrmoptionNoArg        , "Statement"   },
 { "-executive"         , ".pageMedia"                  , XrmoptionNoArg        , "Executive"   },
 { "-a3"                , ".pageMedia"                  , XrmoptionNoArg        , "A3"          },
 { "-a4"                , ".pageMedia"                  , XrmoptionNoArg        , "A4"          },
 { "-a5"                , ".pageMedia"                  , XrmoptionNoArg        , "A5"          },
 { "-b4"                , ".pageMedia"                  , XrmoptionNoArg        , "B4"          },
 { "-b5"                , ".pageMedia"                  , XrmoptionNoArg        , "B5"          },
 { "-folio"             , ".pageMedia"                  , XrmoptionNoArg        , "Folio"       },
 { "-quarto"            , ".pageMedia"                  , XrmoptionNoArg        , "Quarto"      },
 { "-10x14"             , ".pageMedia"                  , XrmoptionNoArg        , "10x14"       },
 { "-forceorientation"	, ".forceOrientation"		, XrmoptionNoArg	, "True"	},
 { "-forcemedia"	, ".forcePageMedia"		, XrmoptionNoArg	, "True"	},
 { "-force"		, ".Force"			, XrmoptionNoArg	, "True"	},
 { "-magstep"		, ".magstep"			, XrmoptionSepArg	, NULL		},
 { "-ad"		, ".userDefaults"		, XrmoptionSepArg	, NULL		},
#ifdef USE_SWAP_LANDSCAPE
 { "-swap"		, ".swapLandscape"		, XrmoptionNoArg	, "True"	},
 { "-noswap"		, ".swapLandscape"		, XrmoptionNoArg	, "False"	},
#endif
};

static XtActionsRec actions[] = {
 { "GV_Center"		, action_center			},
 { "GV_CheckFile"	, action_checkFile		},
 { "GV_ClearBackground"	, action_clearBackground	},
 { "GV_Copyright"	, action_handleVersionPopup	},
 { "GV_DeleteWindow"	, action_deleteWindow		},
 { "GV_Dialog"		, action_handleDialogPopup	},
 { "GV_DismissPopup"	, action_dismissPopup		},
 { "GV_EraseLocator"	, action_erase_locator		},
 { "GV_Force"		, action_force			},
 { "GV_Info"		, action_handleInfoPopup	},
 { "GV_MovePage"	, action_movePage		},
 { "GV_Next"		, action_next			},
 { "GV_Note"		, action_handleNotePopup	},
 { "GV_Open"		, action_open			},
 { "GV_Options"		, action_handleOptionPopup	},
 { "GV_Panner"		, action_panner			},
 { "GV_PopupMenu"	, action_popup_menu		},
 { "GV_Previous"	, action_prev			},
 { "GV_Print"		, action_print			},
 { "GV_Quit"		, action_quit			},
 { "GV_Redisplay"	, action_redisplay		},
 { "GV_Reopen"		, action_reopen			},
 { "GV_Save"		, action_save			},
 { "GV_Scroll"		, action_scroll			},
 { "GV_SetMagstep"	, action_set_magstep		},
 { "GV_SetOrientation"	, action_set_orientation	},
 { "GV_SetPageMark"	, action_setPageMark		},
 { "GV_SetPageMedia"	, action_set_pagemedia		},
 { "GV_Show"		, action_showThisPage		},
#ifdef USE_SWAP_LANDSCAPE
 { "GV_SwapLandscape"	, action_swap_landscape		},
#endif
};

#ifdef FALLBACK_RESOURCES
String fallback_resources[] = {
#   include FALLBACK_RESOURCES
    NULL
};
#endif

/*### Procedure and Macro Declarations ###########################################*/

static void output_message();
static char *GetPseudoResource();

#ifdef max
#   undef max
#endif
#define max(a,b) ((a)>(b)?(a):(b))
#ifdef min
#   undef min
#endif
#define min(a,b) ((a)<(b)?(a):(b))

/*### Global Variables ##########################################################*/

Bool			show_date;
Bool			show_locator;
Bool			show_title;
Bool			show_toggleCurrentPage;
Bool			show_toggleAllPages;
Bool			show_toggleEvenPages;
Bool			show_toggleOddPages;
Bool			show_unmarkAllPages;
Bool			show_saveMarkedPages;
Bool			show_saveAllPages;
Bool			show_printMarkedPages;
Bool			show_printAllPages;     
Bool			show_openFile;
Bool			show_autoResize;
Bool			show_showThisPage;
Bool			show_updateFile;
Bool			show_checkFile;
Bool			show_panner;

float                   default_xdpi;           /* default xdpi from ghostview widget */
float                   default_ydpi;           /* default ydpi from ghostview widget */
int                     num_ghosts;             /* number of ghostview widgets active */
FILE                    *psfile;                /* file to display */
String                  filename;               /* its filename */
String                  oldfilename;            /* previous filename */
int                     current_page;           /* current page being displayed */
int                     current_magstep;        /* current magnification */
XtPageOrientation       current_orientation;    /* current orientation */
int                     current_pagemedia;      /* index into document media or papersizes */
int                     current_llx;            /* current bounding box */
int                     current_lly;
int                     current_urx;
int                     current_ury;
int                     base_papersize;         /* tells where papersizes begins */
String                  toc_text;               /* page labels (Table of Contents) */
int                     toc_length;             /* length of page label text */
int                     toc_entry_length;       /* length of one entry */
int                     gv_print_mode;          /* printing mode */
int                     gv_save_mode;           /* saving mode */
time_t                  mtime;                  /* last modified time of input file */
struct document         *doc;                   /* document structure */
struct document         *olddoc;                /* document structure */
Atom                    wm_delete_window;       /* Atom sent to destroy a window */
XErrorHandler           old_Xerror;             /* standard error handler */
Boolean                 dying;                  /* whether an X error caused our exit */
XErrorEvent             bomb;                   /* what the error was */

int			gv_force;
int                     gv_pagemedia;
int                     gv_forced_pagemedia;
int                     gv_default_pagemedia;
XtPageOrientation       gv_orientation;
XtPageOrientation       gv_forced_orientation;
XtPageOrientation       gv_default_orientation;

String			gv_class = GV_CLASS;
String			gv_name;
String			gv_application_name = GV_APPLICATION_NAME;
String			gv_user_defaults_file;

/* if a page is requested but gv is busy the following variable
holds the number of the requested page.
As soon as gv is notified that rendering a page is complete
this variable is checked and the corresponding page
will be displayed if necessary
*/
int			gv_pending_page_request=NO_CURRENT_PAGE;

XtAppContext            app_con;
AppResources            app_res;

char *			ghostviewShellTitle;

/*### The Widget Tree ###########################################################*/

Widget   toplevel;
Widget      control;
Widget         titlebutton;
Widget            titlemenu;
Widget         datebutton;
Widget            datemenu;
Widget         locator;
Widget         fileButton;
Widget            fileMenu;
Widget               openEntry;
Widget               reopenEntry;
Widget               printAllEntry;
Widget               printMarkedEntry;
Widget               saveAllEntry;
Widget               optionEntry;
Widget               copyrightEntry;
Widget               quitEntry;
Widget         pageButton;
Widget            pageMenu;
Widget               nextEntry;
Widget               showEntry;
Widget               prevEntry;
Widget               centerEntry;
Widget         magstepButton;
Widget            magstepMenu;
Widget               *magstepEntry;
Widget         orientationButton;
Widget            orientationMenu;
Widget               portraitEntry;
Widget               landscapeEntry;
Widget               upsidedownEntry;
Widget               seascapeEntry;
#ifdef USE_SWAP_LANDSCAPE
   Widget               swapEntry;
#endif
Widget         pagemediaButton;
Widget            pagemediaMenu;
Widget               *pagemediaEntry;
Widget      toc3d;
Widget         toc;
Widget      w_toggleCurrentPage;
Widget      w_toggleAllPages;
Widget      w_toggleEvenPages;
Widget      w_toggleOddPages;
Widget      w_unmarkAllPages;
Widget      w_saveMarkedPages;
Widget      w_saveAllPages;
Widget      w_printMarkedPages;
Widget      w_printAllPages; 
Widget      w_openFile;
Widget      w_autoResize;
Widget      w_showThisPage;
Widget      w_updateFile;
Widget      w_checkFile;

Widget      panner3d;
Widget         panner;

Widget      pageview;
Widget         page;

/*### Popup Children ############################################################*/

Widget      infopopup    = NULL;
Widget      optionpopup  = NULL;
Widget      dialogpopup  = NULL;
Widget      notepopup    = NULL;
Widget      versionpopup = NULL;

Widget   FileSel_popup   = NULL;
Widget      FileSel;


/*#################################################################################
   GV_XrmCombineFileDataBase
#################################################################################*/

#if (XlibSpecificationRelease<5) && defined(VMS)
   void
   GV_XrmCombineFileDatabase(filename,target_db,override)
   char* filename;
   XrmDatabase *target_db;
   Bool override;
   {
     if (filename) {
        fprintf(stderr,"  %s: Cannot load resource file %s (see GV_FAQ.TXT)\n",gv_application_name,filename);
     }
   }
#else
#  define GV_XrmCombineFileDatabase XrmCombineFileDatabase
#endif

/*#################################################################################
   Main
#################################################################################*/

int
main(argc, argv)
int  argc;
char *argv[];
{

  {
    struct stat      sbuf;
    Display    *dpy;
    Screen     *scr;
    Arg     args[20];
    Cardinal      n;
    Widget     lineEntry;
    int        number;
    int        i;
    String     s1, s2;
    XTextProperty nameprop;
    Dimension     width, height;
    Dimension     button_width;
    static     String nothing = "";
    static     XawTextSelectType sarry[] = {XawselectLine, XawselectAll, XawselectNull};
    Widget   cont_child[50];
    Cardinal cont_child_num=0;
    XrmDatabase db;
    Dimension maximum_width,maximum_height;
    unsigned int gwidth=0,gheight=0;

    MAINBEGINMESSAGE(main)

    INFMESSAGE(initializing toolkit and the application)
    XtToolkitInitialize();
    app_con = XtCreateApplicationContext();
    XtAppAddActions(app_con, actions, XtNumber(actions));

#   ifdef FALLBACK_RESOURCES
       INFMESSAGE(setting fallback resources)
       XtAppSetFallbackResources(app_con, fallback_resources);
#   endif

/*### opening display #######################################################*/

    INFMESSAGE(opening display)
    dpy = XtOpenDisplay(app_con,NULL,NULL,gv_class,options,XtNumber(options),&argc,argv);
    if (dpy==NULL) { fprintf(stderr, "  %s: unable to open DISPLAY.\n", gv_application_name); exit(1); }
    XtGetApplicationNameAndClass(dpy,&gv_name,&gv_class);
    SMESSAGE(gv_name) SMESSAGE(gv_class)
    db=XtDatabase(dpy);
    if (!db) {
       fprintf(stderr,"  %s:  No database associated with display\n",gv_application_name);
       XtDestroyApplicationContext(app_con);
       exit(1);
    }

/*### additional resources ##################################################*/

    gv_user_defaults_file=NULL;
#   ifdef VMS
       {
          char tmp[GV_MAX_FILENAME_LENGTH];
          char *pos;
          filename = GetPseudoResource(dpy,"userDefaults","UserDefaults");
          if (filename) {
             pos=strrchr(filename,']');
             if (!pos) pos=strrchr(filename,':');
             if (!pos) { sprintf(tmp,"DECW$USER_DEFAULTS:%s",filename); pos=filename; }
             else      { strcpy(tmp,filename);                                        }
             if (!strchr(pos,'.')) strcat(tmp,".DAT");
             GV_XrmCombineFileDatabase(tmp,&db,True);
          } else {
            sprintf(tmp,"DECW$USER_DEFAULTS:%s.DAT",gv_class);
          }
          gv_user_defaults_file=XtNewString(tmp);
          INFSMESSAGE(user resources will be saved to:,gv_user_defaults_file)

          /* explicitly load class resources if necessary */
          pos = GetPseudoResource(dpy,"version","Version");
          if (!pos) {
             sprintf(tmp,"DECW$USER_DEFAULTS:%s_CLASS.DAT",gv_class);
             INFSMESSAGE(loading class resources:,tmp)
             GV_XrmCombineFileDatabase(tmp,&db,False);
             sprintf(tmp,"DECW$SYSTEM_DEFAULTS:%s_CLASS.DAT",gv_class);
             INFSMESSAGE(loading class resources:,tmp)
             GV_XrmCombineFileDatabase(tmp,&db,False);
          }

          pos = GetPseudoResource(dpy,"geometry","Geometry");
          if (pos) {
             int gx=0,gy=0;
             int flag;
             char g[20];
             INFSMESSAGE(found geometry resource,pos)
             flag=XParseGeometry(pos,&gx,&gy,&gwidth,&gheight);
             IIMESSAGE(gx,gy) IIMESSAGE(gwidth,gheight)
             if (gwidth)  gwidth   = max(gwidth,GV_MINIMUM_SIZE);
             if (gheight) gheight  = max(gheight,GV_MINIMUM_SIZE);
                                                                                sprintf(tmp,"");
             if (flag&WidthValue) {
                sprintf(g,"%d",(Dimension)gwidth);                              strcat(tmp,g);
             }
             if (flag&HeightValue) {
                sprintf(g,"x%d",(Dimension)gheight);                            strcat(tmp,g);
             }
             if (flag&XValue) {
                sprintf(g,"%s%d",((flag&XNegative) ? (gx ? "":"-") : "+"),gx);   strcat(tmp,g);
             }
             if (flag&YValue) {
                sprintf(g,"%s%d",((flag&YNegative) ? (gy ? "":"-") : "+"),gy);   strcat(tmp,g);
             }
             if (tmp[0]!='\0') {
                char s[255];
                INFSMESSAGE(corrected geometry,tmp)
                sprintf(s,"%s.geometry",gv_name);
                SMESSAGE(s) SMESSAGE(tmp)
                XrmPutStringResource(&db,s,tmp);
                pos = GetPseudoResource(dpy,"geometry","Geometry");
#               ifdef MESSAGES
                   if (pos) { INFSMESSAGE(check:,pos) }
#               endif
             }
          }
       }
#   else  /* VMS */
       /* This has to be worked out better:
          We just set the the filename gv_user_defaults_file (used for saving options)
          to the value of the queried resource. If no value is found no saving
          can be done.
       */
       filename = GetPseudoResource(dpy,"userDefaults","UserDefaults");
       if (filename) {
          INFMESSAGE(checking for user specific resources)
          gv_user_defaults_file=XtNewString(filename);
          INFSMESSAGE(user resources will be saved to:,gv_user_defaults_file)
       }
#   endif /* VMS */ 
    filename = NULL;

/*### checking for possible file #######################################################*/

    if (argc>2) output_message(dpy,"cmdline-error",0);
    if (argc==2) {
       INFMESSAGE(checking file)
       filename = XtNewString(argv[1]);
       if (!strcmp(filename,"-?") || !strcmp(filename,"-h")) output_message(dpy,"usage",0);
       if (!strcmp(filename,"-help")) output_message(dpy,"help",0);
       if (strcmp(filename, "-")) {
#         ifdef VMS
             INFSMESSAGE(opening:,filename)
             if ((psfile = fopen(filename, "r", "mbc=100")) == NULL) {
                char *p;
                INFMESSAGE(failed)
                p=strrchr(filename,']');
                if (!p) p=strrchr(filename,':');
                if (!p) p=filename;
                if (!strchr(p,'.')) {
                   XtFree(filename);
                   filename = strcpy(XtMalloc(((unsigned)strlen(argv[1])+4)),argv[1]);
                   strcat(filename,".ps");
                }
                INFSMESSAGE(opening:,filename)
                if ((psfile = fopen(filename, "r", "mbc=100")) == NULL) {
                   open_fail_error(errno,GV_ERROR_OPEN_FAIL,filename,True);
                   exit(1);
                }
             }
#         else
             INFSMESSAGE(opening:,filename)
             if ((psfile = fopen(argv[1], "r")) == NULL) {
                fprintf(stderr, "  %s:  Cannot open ",gv_application_name); perror(argv[1]);
                exit(1);
             }
#         endif
          stat(filename, &sbuf);
          mtime = sbuf.st_mtime;
       }
    }

/*### initializing widget set and creating application shell #########################*/

    XawInitializeWidgetSet();
    XtSetTypeConverter(XtRString,XtRPageOrientation,XmuCvtStringToPageOrientation, NULL,0,XtCacheAll, NULL);

    old_Xerror = XSetErrorHandler(catch_Xerror);
    scr = DefaultScreenOfDisplay(dpy);
    wm_delete_window = XInternAtom(dpy, "WM_DELETE_WINDOW", False);

    INFMESSAGE(creating the application shell)
								n=0;
             XtSetArg(args[n], XtNallowShellResize, True);	n++;
    toplevel = XtAppCreateShell(NULL,gv_class,applicationShellWidgetClass,dpy,args,n);

/*### Getting Resources ##############################################################*/

    INFMESSAGE(retrieving and analyzing resources)

    XtGetApplicationResources(toplevel,(XtPointer) &app_res,resources,XtNumber(resources),NULL,ZERO);
    if (!strcmp(app_res.version,GV_DEFAULT_VERSION)) output_message(dpy,"missing-resources",0);
    else if ((!strrchr(app_res.version,' ')) ||
            ((int)(1000*atof(strrchr(app_res.version,' ')))<(int)(1000*atof(strrchr(versionCompatibility,' '))))
            ) output_message(dpy,"version-mismatch",0);

#   ifndef VMS /* a small memory leakage */
       if (s1 = getenv(app_res.printer_variable)) app_res.default_printer = s1;
#   endif

/*### Parsing maximum width, maximum height resources, creating control ####################*/

    {
       char *pos;
       int width,height;
       char* max_size_screen = "screen";
       String geometry;

       INFMESSAGE(parsing maximum size resources)
       pos= strstr(app_res.maximum_width,max_size_screen);
       if (pos) { width=WidthOfScreen(XtScreen(toplevel))+atoi(pos+strlen(max_size_screen)); }
       else     { width=atoi(app_res.maximum_width); }
       maximum_width = (width > 0 ? (Dimension) width : 0);
       maximum_width = max(maximum_width,GV_MINIMUM_SIZE);
       if (maximum_width<(Dimension)gwidth) maximum_width=(Dimension)gwidth;
       pos= strstr(app_res.maximum_height,max_size_screen);
       if (pos) { height=HeightOfScreen(XtScreen(toplevel))+atoi(pos+strlen(max_size_screen)); }
       else     { height=atoi(app_res.maximum_height); }
       maximum_height = (height > 0 ? (Dimension) height : 0);
       maximum_height = max(maximum_height,GV_MINIMUM_SIZE);
       if (maximum_height<(Dimension)gheight) maximum_height=(Dimension)gheight;
       IIMESSAGE(maximum_width,maximum_height)

       INFMESSAGE(creating control)
                               				n=0;
       if (gwidth) {
          app_res.auto_resize=False;
          XtSetArg(args[n], XtNresizeWidth, False);	n++;
          XtSetArg(args[n], XtNwidth, (Dimension)gwidth);n++;
          INFIMESSAGE(forcing width for control:,gwidth)
       } else {
          XtSetArg(args[n], XtNresizeWidth, True);	n++;
       }
       if (gheight) {
          app_res.auto_resize=False;
          XtSetArg(args[n], XtNresizeHeight, False);	n++;
          XtSetArg(args[n], XtNheight, (Dimension)gheight);n++;
          INFIMESSAGE(forcing height for control:,gheight)
       } else {
          XtSetArg(args[n], XtNresizeHeight, True);	n++;
       }
       XtSetArg(args[n], XtNmaximumWidth, maximum_width); n++;
       XtSetArg(args[n], XtNmaximumHeight,maximum_height);n++; 
       XtSetArg(args[n], XtNminimumWidth, GV_MINIMUM_SIZE);n++;
       XtSetArg(args[n], XtNminimumHeight,GV_MINIMUM_SIZE);n++; 
       XtSetArg(args[n], XtNconditionedResize,False);	n++;
       control = XtCreateWidget("control",layoutWidgetClass,toplevel,args,n);
    }

/*### Creating the Menu ###############################################################*/

    INFMESSAGE(menus)

							n=0;
	    XtSetArg(args[0], XtNresize, False);	n++;
    fileButton = XtCreateManagedWidget("fileButton",menuButtonWidgetClass,control,args,n);
    cont_child[cont_child_num] = fileButton; cont_child_num++;

							n=0;
    fileMenu = XtCreatePopupShell("menu", simpleMenuWidgetClass,fileButton,args,n);
    openEntry = XtCreateManagedWidget("open", smeBSBObjectClass,fileMenu,args,n);
            XtAddCallback(openEntry, XtNcallback, cb_openFile, (XtPointer)NULL);
    reopenEntry = XtCreateManagedWidget("reopen", smeBSBObjectClass,fileMenu,args,n);
            XtAddCallback(reopenEntry, XtNcallback,cb_reopen, (XtPointer)NULL);
    printAllEntry = XtCreateManagedWidget("printAllPages", smeBSBObjectClass,fileMenu,args,n);
            XtAddCallback(printAllEntry, XtNcallback, cb_print,(XtPointer)PAGE_MODE_ALL);
    printMarkedEntry = XtCreateManagedWidget("printMarkedPages", smeBSBObjectClass,fileMenu,args,n);
            XtAddCallback(printMarkedEntry, XtNcallback,cb_print,(XtPointer)(PAGE_MODE_MARKED|PAGE_MODE_CURRENT));
    saveAllEntry = XtCreateManagedWidget("saveAllPages", smeBSBObjectClass,fileMenu,args,n);
            XtAddCallback(saveAllEntry, XtNcallback, cb_save, (XtPointer)PAGE_MODE_ALL);
    saveMarkedEntry = XtCreateManagedWidget("saveMarkedPages", smeBSBObjectClass,fileMenu,args,n);
            XtAddCallback(saveMarkedEntry, XtNcallback, cb_save,(XtPointer)(PAGE_MODE_MARKED|PAGE_MODE_CURRENT));
    lineEntry = XtCreateManagedWidget("line", smeLineObjectClass,fileMenu,args,n);
    optionEntry = XtCreateManagedWidget("options", smeBSBObjectClass,fileMenu,args,n);
            XtAddCallback(optionEntry, XtNcallback, cb_popupOptionPopup,(XtPointer)NULL);
    copyrightEntry = XtCreateManagedWidget("copyright", smeBSBObjectClass,fileMenu,args,n);
            XtAddCallback(copyrightEntry, XtNcallback,cb_popupVersionPopup,(XtPointer)NULL);
    quitEntry = XtCreateManagedWidget("quit", smeBSBObjectClass,fileMenu,args,n);
            XtAddCallback(quitEntry, XtNcallback, cb_quitGhostview, (XtPointer)0);

							n=0;
	    XtSetArg(args[0], XtNresize, False);	n++;
    pageButton = XtCreateManagedWidget("pageButton", menuButtonWidgetClass,control,args,n);
    cont_child[cont_child_num] = pageButton; cont_child_num++;

							n=0;
    pageMenu = XtCreatePopupShell("menu", simpleMenuWidgetClass,pageButton,args,n);
    nextEntry = XtCreateManagedWidget("next", smeBSBObjectClass,pageMenu,args,n);
            XtAddCallback(nextEntry, XtNcallback, cb_showNextPage, (XtPointer)0);
    showEntry = XtCreateManagedWidget("show", smeBSBObjectClass,pageMenu,args,n);
            XtAddCallback(showEntry, XtNcallback, cb_redisplay, (XtPointer)0);
    prevEntry = XtCreateManagedWidget("prev", smeBSBObjectClass,pageMenu,args,n);
            XtAddCallback(prevEntry, XtNcallback, cb_showPreviousPage, (XtPointer)0);
    lineEntry = XtCreateManagedWidget("line", smeLineObjectClass,pageMenu,args,n);
    centerEntry = XtCreateManagedWidget("center", smeBSBObjectClass,pageMenu,args,n);
           XtAddCallback(centerEntry, XtNcallback, cb_centerPage, (XtPointer)0);

							n=0;
	XtSetArg(args[0], XtNresize, False);		n++;
    magstepButton = XtCreateManagedWidget("magstepButton",menuButtonWidgetClass,control,args,n);
    cont_child[cont_child_num] =magstepButton; cont_child_num++;

							n=0;
    magstepMenu = XtCreatePopupShell("menu", simpleMenuWidgetClass,magstepButton,args,n);
    magstepEntry = (Widget *) XtMalloc((app_res.maximum_magstep-app_res.minimum_magstep+1) *sizeof(Widget));
           XtSetArg(args[n], XtNleftMargin, 20);                        n++;
    for (i = app_res.minimum_magstep; i <= app_res.maximum_magstep; i++) {
       char buf[16];
       sprintf(buf, "%d", i);
       magstepEntry[i-app_res.minimum_magstep] = XtCreateManagedWidget(buf, smeBSBObjectClass,magstepMenu,args,n);
       XtAddCallback(magstepEntry[i-app_res.minimum_magstep], XtNcallback,cb_setMagstep, (XtPointer)i);
    }

    {
       char *name=XtNewString(GV_DOCUMENT_MENU_NAME);

 							n=0;
	   XtSetArg(args[n], XtNresize, False);		n++;
	   XtSetArg(args[n], XtNmenuName, name);	n++;
       orientationButton = XtCreateManagedWidget("orientationButton",menuButtonWidgetClass,control,args,n);
       cont_child[cont_child_num] = orientationButton; cont_child_num++;

							n=0;
       orientationMenu = XtCreatePopupShell(name, simpleMenuWidgetClass,orientationButton,args,n);
							n=0;
           XtSetArg(args[n], XtNleftMargin, 20);	n++;
       portraitEntry = XtCreateManagedWidget("portrait",smeBSBObjectClass,orientationMenu,args,n);
           XtAddCallback(portraitEntry,XtNcallback,cb_setOrientation,(XtPointer)XtPageOrientationPortrait);
       landscapeEntry = XtCreateManagedWidget("landscape",smeBSBObjectClass,orientationMenu,args,n);
           XtAddCallback(landscapeEntry,XtNcallback,cb_setOrientation,(XtPointer)XtPageOrientationLandscape);
       upsidedownEntry = XtCreateManagedWidget("upsidedown",smeBSBObjectClass,orientationMenu,args,n);
           XtAddCallback(upsidedownEntry,XtNcallback,cb_setOrientation, (XtPointer)XtPageOrientationUpsideDown);
       seascapeEntry = XtCreateManagedWidget("seascape",smeBSBObjectClass,orientationMenu,args,n);
           XtAddCallback(seascapeEntry,XtNcallback,cb_setOrientation,(XtPointer)XtPageOrientationSeascape);
       gv_default_orientation = app_res.orientation;
#ifdef USE_SWAP_LANDSCAPE
       lineEntry = XtCreateManagedWidget("line",smeLineObjectClass,orientationMenu,args,ZERO);
       swapEntry = XtCreateManagedWidget("swap",smeBSBObjectClass,orientationMenu,args,n);
           XtAddCallback(swapEntry,XtNcallback,cb_swapLandscape,(XtPointer)0);

       if (app_res.swap_landscape) {
          XtSetArg(args[0], XtNlabel, &s1);
          XtGetValues(landscapeEntry, args, ONE);
          s1 = XtNewString(s1);
          XtSetArg(args[0], XtNlabel, &s2);
          XtGetValues(seascapeEntry, args, ONE);
          s2 = XtNewString(s2);
          XtSetArg(args[0], XtNlabel, s2);
          XtSetValues(landscapeEntry, args, ONE);
          XtSetArg(args[0], XtNlabel, s1);
          XtSetValues(seascapeEntry, args, ONE);
          XtFree(s1);
          XtFree(s2);
       }
#endif
							n=0;
	  XtSetArg(args[0], XtNresize, False);		n++;
	  XtSetArg(args[n], XtNmenuName, name);		n++;
       pagemediaButton = XtCreateManagedWidget("pagemediaButton",menuButtonWidgetClass,control,args,n);
       cont_child[cont_child_num] = pagemediaButton; cont_child_num++;

       gv_default_pagemedia = 0;
       for (i = 0; papersizes[i].name; i++) {
           if (!strcmp(app_res.pagemedia, papersizes[i].name)) {
              gv_default_pagemedia = i; break;
           }
       }
    }

/*### Optional Widgets ##############################################################*/

    {
       char *layout;
       char *widgetname;

       layout = GetPseudoResource(dpy,"control.layout","Layout.Layout");
       if (!layout) { fprintf(stderr,"  %s: Layout resource not found\n",versionIdentification[0]); }

       INFMESSAGE(optional widgets: labels)
       {
          char wn[20];
          int i=1;
						n=0;
          XtSetArg(args[0], XtNresize, True);	n++;
          while (i<10) {
             sprintf(wn,"label%d",i);
             if (!strstr(layout,wn)) break;
             INFIMESSAGE(creating label,i)
             cont_child[cont_child_num]=XtCreateWidget(wn,labelWidgetClass,control,args,n);
             cont_child_num++;
             ++i;
          }
       }

       INFMESSAGE(optional widgets: lines)
       {
          char wn[20];
          int i=1;
						n=0;
          XtSetArg(args[0], XtNresize, True);	n++;
          while (i<10) {
             sprintf(wn,"line%d",i);
             if (!strstr(layout,wn)) break;
             INFIMESSAGE(creating label,i)
             cont_child[cont_child_num]=XtCreateWidget(wn,threeDWidgetClass,control,args,n);
             cont_child_num++;
             ++i;
          }
       }

       {
          int i=0;
          struct { Widget		*widget;
                   Bool 		*show;
                   String		name;
                   XtCallbackProc	callback;
                   XtPointer		client_data;
          } b[] = {
            { &w_toggleCurrentPage , &show_toggleCurrentPage , "toggleCurrent" , cb_setPageMark , (XtPointer)(SPM_CURRENT|SPM_TOGGLE) },
            { &w_toggleAllPages    , &show_toggleAllPages    , "toggleAll"     , cb_setPageMark , (XtPointer)(SPM_ALL|SPM_TOGGLE)     },
            { &w_toggleEvenPages   , &show_toggleEvenPages   , "toggleEven"    , cb_setPageMark , (XtPointer)(SPM_EVEN|SPM_TOGGLE)    },
            { &w_toggleOddPages    , &show_toggleOddPages    , "toggleOdd"     , cb_setPageMark , (XtPointer)(SPM_ODD|SPM_TOGGLE)     },
            { &w_unmarkAllPages    , &show_unmarkAllPages    , "unmarkAll"     , cb_setPageMark , (XtPointer)(SPM_ALL|SPM_UNMARK)     },
            { &w_printMarkedPages  , &show_printMarkedPages  , "printMarked"   , cb_print       , (XtPointer)(PAGE_MODE_MARKED|PAGE_MODE_CURRENT)},
            { &w_printAllPages     , &show_printAllPages     , "printAll"      , cb_print       , (XtPointer)PAGE_MODE_ALL           },
            { &w_saveMarkedPages   , &show_saveMarkedPages   , "saveMarked"    , cb_save        , (XtPointer)(PAGE_MODE_MARKED|PAGE_MODE_CURRENT)},
            { &w_saveAllPages      , &show_saveAllPages      , "saveAll"       , cb_save        , (XtPointer)PAGE_MODE_ALL            },
            { &w_openFile          , &show_openFile          , "openFile"      , cb_openFile ,    (XtPointer)NULL },
            { &w_autoResize        , &show_autoResize        , "autoResize"    , cb_autoResize ,  (XtPointer)NULL },
            { &w_showThisPage      , &show_showThisPage      , "redisplay"     , cb_redisplay ,   (XtPointer)NULL },
            { &w_updateFile        , &show_updateFile        , "updateFile"    , cb_checkFile ,   (XtPointer)CHECK_FILE_VERSION },
            { &w_checkFile         , &show_checkFile         , "checkFile"     , cb_checkFile ,   (XtPointer)CHECK_FILE_DATE },
            { NULL         , NULL        , NULL     , NULL ,  NULL },
          };
          INFMESSAGE(optional widgets: buttons)
						n=0;
          XtSetArg(args[0], XtNresize, True);	n++;
          while (b[i].widget) {
             INFSMESSAGE(creating widget,b[i].name)
             *(b[i].show) = strstr(layout,b[i].name) ? True : False;
             if (*(b[i].show)) {
                *(b[i].widget) = XtCreateWidget(b[i].name,commandWidgetClass,control,args,n);
                XtAddCallback(*(b[i].widget),XtNcallback,b[i].callback,b[i].client_data);
                cont_child[cont_child_num] = *(b[i].widget); cont_child_num++;
             }
             ++i;
          }
       }


       INFMESSAGE(optional widgets: informational widgets)
#      define _mw_(widget,show,name)						\
         widgetname=name;							\
         show = strstr(layout,widgetname) ? True : False;			\
         if (show) {								\
            widget = XtCreateWidget(widgetname,labelWidgetClass,control,args,n);\
            cont_child[cont_child_num] = widget; cont_child_num++;		\
         }

	      						n=0;
       _mw_( titlebutton , show_title   , "titleButton" );
       _mw_( datebutton  , show_date    , "dateButton"  );

       widgetname="locator";
       show_locator = strstr(layout,widgetname) ? True : False;
       if (show_locator) {
          char buf[MAX_LOCATOR_LENGTH];
          sprintf(buf,app_res.locator_format,9999,9999);
          XtSetArg(args[n], XtNlabel,buf);		n++;
          locator = XtCreateWidget(widgetname,labelWidgetClass,control,args,n);
          cont_child[cont_child_num] =locator; cont_child_num++;
       }
#      undef _mw_


       INFMESSAGE(panner)
       widgetname="panner3d";
       show_panner = strstr(layout,widgetname) ? True : False;
       if (show_panner) {
									n=0;
#         if defined(USE_XAW3D) && defined(USE_FORM3D)
             panner3d = XtCreateWidget(widgetname,form3dWidgetClass,control,args,n);
#         else
             panner3d = XtCreateWidget(widgetname,formWidgetClass,control,args,n);
#         endif
          cont_child[cont_child_num] = panner3d; cont_child_num++;

									n=0;
               XtSetArg(args[n], XtNfromVert, NULL);			n++;
               XtSetArg(args[n], XtNfromHoriz, NULL);			n++;
               XtSetArg(args[n], XtNtop, XtChainTop);			n++;
               XtSetArg(args[n], XtNbottom, XtChainBottom);		n++;
               XtSetArg(args[n], XtNleft, XtChainLeft);			n++;
               XtSetArg(args[n], XtNright, XtChainRight);		n++;
               XtSetArg(args[n], XtNresizable, True);			n++;
               XtSetArg(args[n], XtNresize, False);			n++;
          panner = XtCreateManagedWidget("panner", pannerWidgetClass,panner3d, args, n);
            XtAddCallback(panner, XtNreportCallback,cb_movePanner,NULL);
       }

    } /* end of optional widgets */

/*### Table of Contents ###########################################################*/

    INFMESSAGE(table of contents)

									n=0;
            XtSetArg(args[n], XtNresize, True);				n++;
            XtSetArg(args[n], XtNresizable, True);			n++;
#if defined(USE_XAW3D) && defined(USE_FORM3D)
    toc3d = XtCreateWidget("toc3d",form3dWidgetClass,control,args,n);
#else
    toc3d = XtCreateWidget("toc3d",formWidgetClass,control,args,n);
#endif
    cont_child[cont_child_num] = toc3d; cont_child_num++;

									n=0;
            XtSetArg(args[n], XtNfromVert, NULL);			n++;
            XtSetArg(args[n], XtNfromHoriz, NULL);			n++;
            XtSetArg(args[n], XtNtop, XtChainTop);			n++;
            XtSetArg(args[n], XtNbottom, XtChainBottom);		n++;
            XtSetArg(args[n], XtNleft, XtChainLeft);			n++;
            XtSetArg(args[n], XtNright, XtChainRight);			n++;
            XtSetArg(args[n], XtNresizable, False);			n++;
            XtSetArg(args[n], XtNdisplayCaret, False);			n++;
            XtSetArg(args[n], XtNuseStringInPlace, True);		n++;
            XtSetArg(args[n], XtNlength, 0);				n++;
            XtSetArg(args[n], XtNstring, nothing);			n++;
            XtSetArg(args[n], XtNselectTypes, sarry);			n++;
            XtSetArg(args[n], XtNscrollVertical, XawtextScrollAlways);	n++;
            XtSetArg(args[n], XtNwidth,TOC3D_INITIAL_WIDTH);	n++;
            XtSetArg(args[n], XtNheight,TOC3D_INITIAL_HEIGHT);	n++;
    toc = XtCreateManagedWidget("toc", asciiTextWidgetClass,toc3d, args, n);

/*### The Page View ###########################################################*/

    INFMESSAGE(viewport)
									n=0;
            XtSetArg(args[n], XtNallowHoriz,True);			n++;
            XtSetArg(args[n], XtNallowVert, True);			n++;
            XtSetArg(args[n], XtNresizable, True);			n++;
    pageview = XtCreateWidget("pageview", viewportWidgetClass,control,args,n);
    if (show_panner) XtAddCallback(pageview, XtNreportCallback,cb_moveSlider,(XtPointer)NULL);
    cont_child[cont_child_num] = pageview; cont_child_num++;
                                                                        n=0;
            XtSetArg(args[n], XtNautoSetup,False);			n++;
    page = XtCreateManagedWidget("page", ghostviewWidgetClass,pageview, args,n);
            num_ghosts++;
            XtAddCallback(page, XtNcallback, cb_trackAndZoom, (XtPointer)0);
            XtAddCallback(page, XtNdestroyCallback, cb_destroyGhost, (XtPointer)page);
            XtAddCallback(page, XtNmessageCallback, cb_message, (XtPointer)page);
            XtAddCallback(page, XtNoutputCallback,cb_appendInfoPopup, (XtPointer)0);

									n=0;
            XtSetArg(args[n], XtNxdpi, &default_xdpi);			n++;
            XtSetArg(args[n], XtNydpi, &default_ydpi);			n++;
            XtGetValues(page, args, n);

/*### remaining initialization #####################################################*/

    INFMESSAGE(additional initialization)
    GhostviewDisableInterpreter(page);
    setup_ghostview();
    number = find_page(app_res.page);
    if (toc_text) {
       if (number >= doc->numpages) number = doc->numpages - 1;
       if (number < 0) number = 0;
    }
    /* Coerce magstep to fall in range */
    if (app_res.magstep < app_res.minimum_magstep) app_res.magstep = app_res.minimum_magstep;
    if (app_res.magstep > app_res.maximum_magstep) app_res.magstep = app_res.maximum_magstep;

/*### managing the children ######################################################*/

    INFMESSAGE(managing children of control)
    XtManageChildren((WidgetList)cont_child,cont_child_num);
    INFMESSAGE(managing control)
    XtManageChild(control);
    XtSetMappedWhenManaged(toplevel, False);
    INFMESSAGE(realizing toplevel)
    XtRealizeWidget(toplevel);
    XSetWMProtocols(dpy, XtWindow(toplevel), &wm_delete_window, 1);

/*### Creating the File Selection Popup ###########################################*/

    INFMESSAGE(creating file selection popup)
                              				n=0;
            XtSetArg(args[n], XtNallowShellResize,True);n++;
    FileSel_popup = XtCreatePopupShell("fileSelPopup",transientShellWidgetClass,toplevel,args,n);
                              				n=0;
            XtSetArg(args[n], XtNviewMode, XawFileSelectionRescan); n++;
            XtSetArg(args[n], XtNbuttons,  2);		n++;
            XtSetArg(args[n], XtNpreferredButton, 1);	n++;
    FileSel = XtCreateManagedWidget("fileSel",file_selectionWidgetClass,FileSel_popup,args,n);
	XtAddCallback(XtNameToWidget(FileSel,"button2"), XtNcallback,cb_popdownPopup,FileSel_popup);
	XtAddCallback(XtNameToWidget(FileSel,"button2"), XtNcallback,cb_popdownNotePopup,(XtPointer)NULL);

    XtRealizeWidget(FileSel_popup);
    XSetWMProtocols(dpy,XtWindow(FileSel_popup),&wm_delete_window,1);

/*### now we become visible ######################################################*/

    INFMESSAGE(setting window title)

#   ifdef SHOW_COMPILE_DATE
       ghostviewShellTitle = (char*) XtMalloc(strlen(__DATE__)+strlen(versionIdentification[0])+17);
       sprintf(ghostviewShellTitle,"%s (compiled on %s)",versionIdentification[0],__DATE__);
#   else
       ghostviewShellTitle = (char*) XtMalloc(strlen(versionIdentification[0])+1);
       sprintf(ghostviewShellTitle,"%s",versionIdentification[0]);
#   endif
    if (XStringListToTextProperty(&ghostviewShellTitle,1,&nameprop)) {
       XSetWMName(dpy, XtWindow(toplevel),&nameprop);
    }

    INFMESSAGE(switching off resize for buttons and labels)
							n=0;
	XtSetArg(args[0], XtNresize, False);		n++;
    if (show_toggleCurrentPage)	XtSetValues(w_toggleCurrentPage,args,n);
    if (show_toggleAllPages)	XtSetValues(w_toggleAllPages,args,n);
    if (show_toggleEvenPages)	XtSetValues(w_toggleEvenPages,args,n);
    if (show_toggleOddPages)	XtSetValues(w_toggleOddPages,args,n);
    if (show_unmarkAllPages)	XtSetValues(w_unmarkAllPages,args,n);
    if (show_saveMarkedPages)	XtSetValues(w_saveMarkedPages,args,n);
    if (show_printMarkedPages)	XtSetValues(w_printMarkedPages,args,n);
    if (show_printAllPages)	XtSetValues(w_printAllPages,args,n);
    if (show_openFile)		XtSetValues(w_openFile,args,n);
    if (show_autoResize)	XtSetValues(w_autoResize,args,n);
    if (show_showThisPage)	XtSetValues(w_showThisPage,args,n);
    if (show_updateFile)	XtSetValues(w_updateFile,args,n);
    if (show_checkFile)		XtSetValues(w_checkFile,args,n);
    if (show_date)		XtSetValues(datebutton,args,n);
    if (show_title)		XtSetValues(titlebutton,args,n);
    if (show_locator) {
       				XtSetArg(args[n], XtNlabel,"");		n++;
				XtSetValues(locator,args,n);
    }

    setup_layout_ghostview();
    if (filename) current_page=number;
    show_page(REQUEST_SETUP);

    /* must allow control to resize */
    LayoutWidgetAllowResize((LayoutWidget)control,True,True);

    INFMESSAGE(mapping toplevel)
    XtMapWidget(toplevel);
  }

  INFMESSAGE(waiting for events now)
  XtAppMainLoop(app_con);

  /* should never get here */
  return 1;
}

/*#################################################################################
   output_message
#################################################################################*/

#define MAX_RECORD_LENGTH 256

static void
output_message(dpy,identifier,go_back)
   Display *dpy;
   char *identifier;
   Boolean go_back;
{
   FILE *helpfile;
   char *helpfilename;
   char line[MAX_RECORD_LENGTH];
   int  show=0;

   helpfilename = GetPseudoResource(dpy,"messageFile","MessageFile");
   if (!helpfilename) helpfilename = GV_DEFAULT_MESSAGE_FILE;

   if ((helpfile=fopen(helpfilename,"r"))==NULL) {
      open_fail_error(errno,GV_ERROR_OPEN_FAIL,helpfilename,True);
      show=3;
   }

   while (show<2 && fgets(line,MAX_RECORD_LENGTH,helpfile)) {
      if (line[0]=='#' && strstr(line,identifier)) ++show;
      else if (show==1 && line[0]!='!') {
           if      (line[0]=='C')  fprintf(stderr,&(line[1]),gv_class);
           else if (line[0]=='N')  fprintf(stderr,&(line[1]),gv_name);
           else if (line[0]=='A')  fprintf(stderr,&(line[1]),gv_application_name);
           else if (line[0]=='V')  fprintf(stderr,&(line[1]),versionIdentification[0]);
           else if (line[0]=='\n') fprintf(stderr,line);
           else                    fprintf(stderr,&(line[1]));
      } else if (show==2) break;
   }
   fclose(helpfile);
   if (go_back && show<3) return;
   fclose(psfile);
   XtDestroyApplicationContext(app_con);
   exit(1);
}   

/*#################################################################################
    GetPseudoResource
#################################################################################*/

static char *GetPseudoResource(dpy,name_val,class_val)
Display *dpy;
char *name_val;
char *class_val;
{
   char *result;
   XrmDatabase db=XrmGetDatabase(dpy);

   BEGINMESSAGE(GetPseudoResource)
   if (db) {
      XrmValue rm_value;
      String str_type;
      char str_name[GV_MAX_FILENAME_LENGTH],str_class[GV_MAX_FILENAME_LENGTH];
      Bool success=False;
  
      sprintf(str_name, "%s.%s",gv_name,name_val);
      sprintf(str_class,"%s.%s",gv_class,class_val);
      success = XrmGetResource(db,str_name,str_class,&str_type,&rm_value);
      if  (success && rm_value.size>1) { result = (char*) rm_value.addr; INFMESSAGE(found entry)    }
      else                             { result = NULL;                  INFMESSAGE(no entry found) }
   }
   ENDMESSAGE(GetPseudoResource)
   return(result);
}
