/* (C) Copyright 1991 Andrew Plotkin. Permission is
 given to copy and use, as long as this copyright
 notice is retained. */

#include <stdio.h>
#include <math.h>    
#include <X11/Xlib.h>
#include "spheral.h"

typedef struct _ballp {
    int x, y, z;
} ballp;

point plist[MAXBALLS];	    
point traplist[MAXBALLS];   
point templist[MAXBALLS];   
int aspectflag = 0;
double aspect;
double offx, offy, offz;
double focallen, boardscale;
int halfboardx, halfboardy;
fieldplist fieldpts;
fieldrlist fieldrads;

int colors[12] = {1, 3, 7, 15, 1, 3, 7, 15, 1, 3, 7, 15};

extern GC gcwhite, gcblack, gcfield, gcadd,
gccopy, gcballs[];
extern Pixmap ballpm[NUMBALLSIZES][NUMSHADES];
extern short ballpmflag[NUMBALLSIZES][NUMSHADES];
extern int errdiflag; 

extern void updatepiece(), round_piece(), measure_curpiece(),
rotate_axis();
extern void create_ballpm();

void startpiece()
{
    register int ix;
    int res;
    piecelist *p;
    int flix = (random() % 2)*2-1;
    int fliy = (random() % 2)*2-1;
    int fliz = (random() % 2)*2-1;

    curpiece = random() % numpieces;

    p = &(pieces[curpiece]);
    for (ix=0; ix<p->numballs; ix++) {
	plist[ix].x = flix*p->points[ix].x;
	plist[ix].y = fliy*p->points[ix].y;
	plist[ix].z = fliz*p->points[ix].z;
	plist[ix].w = p->points[ix].w;
	traplist[ix].w = plist[ix].w;
    };
    {
	int initx = fieldx/4;
	int inity = initx;
	int initz = fieldz+4;
	offz = ((double)initz) - 0.5*inity - 0.5;
	offy = ROOTHALF*((double)inity);
	offx = ((double)initx) + 0.5*inity + 0.5;
    }
    updatepiece();

    res = collision(0);
    while (res!=OUT_NOT && res!=OUT_COLLIDE) {
	switch (res) {
	    case OUT_UP:
		offz = offz - 1.0 ;
		break;
	    case OUT_DOWN:
		fprintf(stderr,
			"spheral: piece too long for board\n");
		exit(-1);
		break;
	    case OUT_NORTHEAST:
		offx = offx - 0.5;
		offy = offy - ROOTHALF;
		offz = offz + 0.5;
		break;
	    case OUT_SOUTH:
		offx = offx + 0.5;
		offy = offy + ROOTHALF;
		offz = offz - 0.5;
		break;
	    case OUT_NORTHWEST:
		offx = offx + 1.0;
		break;
	}
	updatepiece();
	res = collision(0);
    };
    if (res==(OUT_COLLIDE)) curpiece = (-2);
}

void rotate_piece(pls, axis, theta)
/* works on plist */
point pls[];
int axis;   /* 4-axis system */
double theta;
{
#define PHI (0.7853981634)
    switch (axis) {
	case 1:
	    rotate_axis(pls, 2, theta);
	    break;
	case 2:
	    rotate_axis(pls, 2, PHI);
	    rotate_axis(pls, 1, theta);
	    rotate_axis(pls, 2, -PHI);
	    break;
	case 3:
	case 4:
	    rotate_axis(pls, 2, -PHI);
	    rotate_axis(pls, 1, -theta);
	    rotate_axis(pls, 2, PHI);
	    break;
    }
}

void reverse_piece(pls, axis)
point pls[];
int axis;
{
    register int ix;

    switch (axis) {
	case -1:
	    for (ix=0; ix<pieces[curpiece].numballs; ix++) {
		pls[ix].x = (-pls[ix].x);
	    }
	    break;
    }
}

void rotate_axis(pls, axis, theta)
point pls[];
int axis;   /* 3-axis system */
double theta;
{
    register int ix;
    double t1, t2;
    double sinth = sin(theta);
    double costh = cos(theta);

    switch (axis) {
	case 1:
	    for (ix=0; ix<pieces[curpiece].numballs; ix++) {
		t1 = pls[ix].y;
		t2 = pls[ix].z;
		pls[ix].y = costh*t1 - sinth*t2;
		pls[ix].z = sinth*t1 + costh*t2;
	    }
	    break;
	case 2:
	    for (ix=0; ix<pieces[curpiece].numballs; ix++) {
		t1 = pls[ix].x;
		t2 = pls[ix].z;
		pls[ix].x = costh*t1 - sinth*t2;
		pls[ix].z = sinth*t1 + costh*t2;
	    }
	    break;
	case 3:
	    for (ix=0; ix<pieces[curpiece].numballs; ix++) {
		t1 = pls[ix].x;
		t2 = pls[ix].y;
		pls[ix].x = costh*t1 - sinth*t2;
		pls[ix].y = sinth*t1 + costh*t2;
	    }
	    break;
    }
}

void updatepiece()
/* create traplist from plist+offsets */
{
    register int ix;

    if (!aspectflag) {
	for (ix=0; ix<pieces[curpiece].numballs; ix++) {
	    traplist[ix].x = plist[ix].x + offx;
	    traplist[ix].y = plist[ix].y + offy;
	    traplist[ix].z = plist[ix].z + offz;
	};
    }
    else {
	for (ix=0; ix<pieces[curpiece].numballs; ix++) {
	    traplist[ix].x = aspect*plist[ix].x + offx;
	    traplist[ix].y = plist[ix].y + offy;
	    traplist[ix].z = plist[ix].z + offz;
	};
    }
}

void updatetemp_tra(xa, ya, za, taxi, tdir)
/* create templist from traplist, adding (whole)
 moves and turns */
double xa, ya, za;
int taxi; /* positive for rotation;
 negative for reflection */
int tdir;
{
    register int ix;

    if (taxi==0) {
	for (ix=0; ix<pieces[curpiece].numballs; ix++) {
	    templist[ix].x = traplist[ix].x + xa;
	    templist[ix].y = traplist[ix].y + ya;
	    templist[ix].z = traplist[ix].z + za;
	};
    }
    else if (taxi<0) {
	switch (taxi) {
	    case -1:
		for (ix=0; ix<pieces[curpiece].numballs; ix++) {
		    templist[ix].x = -plist[ix].x;
		    templist[ix].y = plist[ix].y;
		    templist[ix].z = plist[ix].z;
		};
		for (ix=0; ix<pieces[curpiece].numballs; ix++) {
		    templist[ix].x += offx + xa;
		    templist[ix].y += offy + ya;
		    templist[ix].z += offz + za;
		};		
		break;
	}
    }
    else {
	for (ix=0; ix<pieces[curpiece].numballs; ix++) {
	    templist[ix].x = plist[ix].x;
	    templist[ix].y = plist[ix].y;
	    templist[ix].z = plist[ix].z;
	};
	rotate_piece(templist, taxi, ((double)(tdir))*PI/2.0);
	for (ix=0; ix<pieces[curpiece].numballs; ix++) {
	    templist[ix].x += offx + xa;
	    templist[ix].y += offy + ya;
	    templist[ix].z += offz + za;
	};
    }
}

void round_piece() /* round off piece (plist) to grid */
{
    register int ix;

    offx = floor(offx*2.0 + 0.5) / 2.0;
    offy = floor(offy/(0.5*ROOTHALF) + 0.5) * (0.5*ROOTHALF);
    offz = floor(offz*2.0 + 0.5) / 2.0;

    for (ix=0; ix<pieces[curpiece].numballs; ix++) {
	plist[ix].x = floor(2.0*plist[ix].x + 0.5)/2.0;
	plist[ix].y = floor(plist[ix].y/(0.5*ROOTHALF) + 0.5) * (0.5*ROOTHALF);
	plist[ix].z = floor(2.0*plist[ix].z + 0.5)/2.0;
    };
}

int collision(listflag)
/* Check a list for collision. Check traplist if
 listflag=0, templist if listflag=1.
 Returns (in this priority)
 1: out-of-field up;
 2: ...down;
 3: ...northeast;
 4: ...south;
 6: ...northwest;
 -1 for ball overlap; 
 0 for ok; 
 */
int listflag;
{
    register int ix;
    static ballp balls[MAXBALLS];
    point *pls;

    if (listflag==0) pls = traplist;
    else pls = templist;

    for (ix=0; ix<pieces[curpiece].numballs; ix++) {
	balls[ix].y = (int)(pls[ix].y/ROOTHALF+100.5) - 100;
	balls[ix].z = (int)(pls[ix].z+0.5*balls[ix].y+100.5) - 100;
	balls[ix].x = (int)(pls[ix].x-0.5*balls[ix].y+100.5) - 100;
	/*printf("%.3f, %.3f, %.3f == %d, %d, %d\n", pls[ix].x,
	 pls[ix].y, pls[ix].z, balls[ix].x, balls[ix].y, balls[ix].z);*/

	if (balls[ix].z < 0) return OUT_DOWN;
	if (balls[ix].z >= fieldz) return OUT_UP;
	if (balls[ix].y < 0) return OUT_SOUTH;
	if (balls[ix].y >= fieldx-balls[ix].x) return OUT_NORTHEAST;
	if (balls[ix].x < 0) return OUT_NORTHWEST;
    };
    for (ix=0; ix<pieces[curpiece].numballs; ix++) {
	if (field[balls[ix].x][balls[ix].y][balls[ix].z] & F_ON)
	    return (OUT_COLLIDE);
    };

    return OUT_NOT;
}

void setup_fieldpts() /* just that */
{
    register int ix, iy, iz;
    double xc, yc, zc, wcoord;

    if (errdiflag) {
	XClearWindow(dpy, win);
	XDrawImageString(dpy, win, gcwhite, 50, 50,
	"Please wait about 15 seconds; computing pixmaps",
	strlen("Please wait about 15 seconds; computing pixmaps"));
    }

    for (iz=0; iz<fieldz; iz++)
	for (ix=0; ix<fieldx; ix++)
	    for (iy=0; iy<fieldx-ix; iy++) {
		zc = (double)iz - 0.5*iy;
		yc = ROOTHALF*iy;
		xc = (double)ix + 0.5*iy;

		wcoord = -(zc+fieldoffz)/focallen + 1.0;
		fieldpts[ix][iy][iz].x = halfboardx +
		  (int)(boardscale*(xc+fieldoffx)/wcoord);
		fieldpts[ix][iy][iz].y = halfboardy -
		  (int)(boardscale*(yc+fieldoffy)/wcoord);
		fieldrads[ix][iy][iz] = (int)(0.5*boardscale/wcoord);

		if (errdiflag &&
		    !ballpmflag[fieldrads[ix][iy][iz]][iz%NUMSHADES]) {
		    create_ballpm(fieldrads[ix][iy][iz], iz%NUMSHADES);
		}
	    }
    
}

void add_ball(cx, cy, cz) /* draw a ball on fieldpm */
int cx, cy, cz;
{
    int rad = fieldrads[cx][cy][cz];
    int x1 = fieldpts[cx][cy][cz].x;
    int y1 = fieldpts[cx][cy][cz].y;

    if (!errdiflag) {
	GC gcball, gcrim;

	if (monomode) {
	    gcball = (gcballs[colors[cz]]);
	    gcrim = (colors[cz]<8) ? (gcfield) : (gcwhite);
	}
	else {
	    gcball = (gcballs[cz]);
	    gcrim = (gcwhite);
	}

	XFillArc(dpy, fieldpm, gcball, x1-rad, y1-rad,
		 rad*2, rad*2, 0, 23040); 
	/*XFillArc(dpy, fieldpm, gcspot, fieldpts[cx][cy][cz].x,
	 fieldpts[cx][cy][cz].y, rad*2/3, rad*2/3, 0, 23040); */
	XDrawArc(dpy, fieldpm, gcrim, x1-rad, y1-rad,
		 rad*2, rad*2, 0, 23040); 
    }
    else {
	if (!ballpmflag[rad][cz%NUMSHADES]) {
	    fprintf(stderr,
		    "spheral: missing ball pixmap (%d, %d)\n",
		    rad, cz%NUMSHADES);
	    exit(-1);
	}
	XFillArc(dpy, fieldpm, gcblack, x1-rad, y1-rad,
		rad*2, rad*2, 0, 23040); 
	XCopyArea(dpy, ballpm[rad][cz%NUMSHADES], fieldpm,
		gcadd, 0, 0, rad*2, rad*2, x1-rad, y1-rad);
    }
}

void add_balls(lev)
/* add balls to fieldpm. lev is the smallest z-coord affected */
int lev;
{
    register int ix, iy, iz;
    int div;

    for (iz=lev; iz<fieldz; iz++) 
	for (iy=fieldx-1; iy>=0; iy--) {
	    div = (fieldx-iy)/2;
	    for (ix=0; ix<div; ix++) {
		if (field[ix][iy][iz] & F_ON) {
		    add_ball(ix, iy, iz);
		}
	    }
	    for (ix=(fieldx-iy-1); ix>=div; ix--) {		
		if (field[ix][iy][iz] & F_ON) {
		    add_ball(ix, iy, iz);
		}
	    }

	}
}

void plop_piece() /* plop traplist onto fieldpm. */
{
    register int ix, iy, iz;
    int cubx, cuby, cubz;
    int dirx, diry;
    int lev = fieldz+1;
    int full;

    for (ix=0; ix<pieces[curpiece].numballs; ix++) {
	cuby = (int)(traplist[ix].y/ROOTHALF+100.5) - 100;
	cubz = (int)(traplist[ix].z+0.5*cuby+100.5) - 100;
	cubx = (int)(traplist[ix].x-0.5*cuby+100.5) - 100;
	if (cubz < lev) lev = cubz;
	/*if (cubz+1 > meterlev) meterlev = cubz+1;*/
	field[cubx][cuby][cubz] |= F_ON;
    };
    add_balls(lev);

    /* check settling */
    while (1) {
	for (iz=fieldz-1; iz>=0; iz--) {
	    full=1;
	    for (ix=0; full && (ix<fieldx); ix++) {
		for (iy=0; full && (iy<fieldx-ix); iy++) {
		    if (!(field[ix][iy][iz] & F_ON)) full = 0;
		}
	    }
	    if (full) break;
	}
	if (iz<0) break;
	for (; iz<fieldz-1; iz++) 
	    for (ix=0; ix<fieldx; ix++)
		for (iy=0; iy<fieldx-ix; iy++)
		    field[ix][iy][iz] = field[ix][iy][iz+1];
	for (ix=0; ix<fieldx; ix++)
	    for (iy=0; iy<fieldx-ix; iy++)
		field[ix][iy][fieldz-1] = 0;
	score += 15*fieldx*fieldx;
	if (dropticks > 10) dropticks -= 6;
	/*meterlev--;*/
	setup_fieldpm();
	add_balls(0);
    }
}

void measure_curpiece(psum)
/* locate balls (from traplist) in screen coords */
piecesum *psum;
{
    register int ix, jx;
    int numballs;
    double wcoord;
    double dist[MAXBALLS], lowdist;
    short arr[MAXBALLS], lowval, temp;

    numballs = pieces[curpiece].numballs;
    psum->numballs = numballs;

    if (!aspectflag) {
	psum->scalex = 1.0;
	psum->scaley = 1.0;
    }
    else {
	psum->scalex = aspect;
	psum->scaley = 1.0;
    }

    for (ix=0; ix<numballs; ix++) {
	arr[ix] = ix;
	dist[ix] = (traplist[ix].z+fieldoffz)*(traplist[ix].z+fieldoffz) +
	  (traplist[ix].x+fieldoffx)*(traplist[ix].x+fieldoffx) +
	  (traplist[ix].y+fieldoffy)*(traplist[ix].y+fieldoffy);
	/*printf("%.3f, %.3f, %.3f = %.3f\n",
	 (traplist[ix].z+fieldoffz), */
    };
    
    for (ix=0; ix<numballs-1; ix++) {
	lowdist = dist[arr[ix]];
	lowval = ix;
	for (jx=ix+1; jx<numballs; jx++) {
	    if (dist[arr[jx]] > lowdist) {
		lowdist = dist[arr[jx]];
		lowval = jx;
	    };
	};
	if (ix!=lowval) {
	    temp = arr[ix];
	    arr[ix] = arr[lowval];
	    arr[lowval] = temp;
	}
    }

    for (ix=0; ix<numballs; ix++) {
	jx = arr[ix];
	wcoord = -(traplist[jx].z+fieldoffz)/focallen + 1.0;
	psum->p[ix].x = halfboardx +
	  (int)(boardscale*(traplist[jx].x+fieldoffx)/wcoord);
	psum->p[ix].y = halfboardy -
	  (int)(boardscale*(traplist[jx].y+fieldoffy)/wcoord);	
	psum->rad[ix] = (int)(0.5*boardscale/wcoord);
    };
}

