/* heap.c : heap manager for the tabulator. */
/* Copyright 1993 Stefane Fermigier. */

#include <stdio.h>
#ifndef VMS
#include <malloc.h>
#endif
#include "midifile.h"
#include "tabul.h"

/* From main.c */
void Fail();

static struct event *heap;
static heap_num = -1; /* number of events in the heap */
static heap_size;

void HeapInit (size)
int size;
{
	heap_size = size;
	heap_num = -1;
	heap = (struct event *) malloc (sizeof(struct event) * size);
	if (heap == NULL) 
		Fail ("Not enough space for the heap.\n");
}

void HeapDestroy ()
{
	(void) free (heap);
}

static void SwapEvents (e1, e2)
struct event *e1, *e2;
{
	struct event e3;

	e3 = *e1;
	*e1 = *e2;
	*e2 = e3;
}

/* We need this so that commands are well ordered. */
static int Compare (e1, e2)
struct event *e1, *e2;
{
	return (e1->time < e2->time ) || (e1->time == e2->time 
		&& e1->mutime < e2->mutime);
}

void HeapPut(evn) 
struct event *evn;
{
	int i;

	heap_num++;
	if (heap_num > heap_size) {
		fprintf(stderr, "Not enough space in heap.\n");
		exit(1);
	}
	heap[heap_num] = *evn;
	i = heap_num + 1;
	while ( (i > 1) && Compare ( &heap[i-1], &heap[i/2-1]) ) {
		SwapEvents ( &heap[i-1], &heap[i/2-1] );
		i = i/2;
	}
}

/* Gets next event whith time <= t, returns 0 if no more. */
int HeapGet(evn, t)
struct event *evn;
unsigned long t;
{
	int i, j;

	if ( (heap_num < 0) || (heap[0].time > t) )
		return STAT_EOF;
	
	*evn = heap[0];
	heap[0] = heap[heap_num];
	heap_num--;
	for (i = 1; i <= (heap_num + 1) / 2; ) {
		if ( (2*i == heap_num + 1) || Compare (&heap[2*i-1], &heap[2*i]) ) 
			j = 2*i;
		else 
			j = 2*i+1;
		if (Compare (&heap[j-1], &heap[i-1]) ) {
			SwapEvents(&heap[i-1], &heap[j-1]);
			i = j;
		} else
			return STAT_OK;
	}
	return STAT_OK;
}
