/* decode.c : decodes the lines in the input file. */
/* Copyright 1993 Stefane Fermigier. */

#include <stdio.h>
#include <string.h>
#include <strings.h>

#include "midifile.h"
#include "tabul.h"

/* from main.c */
extern FILE *input_file;
extern char *file_name;
extern int debug;

/* from config.c */
extern struct measure *measures;
extern int tab_length;
extern char **def_command;

/* from hash.c */
char *HashGet();

/* from heap.c */
void HeapPut();
int HeapGet();


/* this file */
int pos_to_duration[LINE_SIZE];
int pos_to_accent[LINE_SIZE]; 
int current_measure = 0; /* Measure # at beginnig of the block. */
unsigned long current_time = 0; /* MIDI time at begining of current block. */

struct funct_info {
	char *name;
	int (*func)();
	int type;
};

/* from commands.c */
int Time(), Sig(), Accents(), Chords(), Tempo();
int Tab(), Chord(), Perc(), ShortChords();

/* from control.c */
int Repeat(), Goto(), End(), Label();

/* Built in functions. */
struct funct_info functions[] = {
	{ "sig", Sig, MEASURE_FUNC },
	{ "chords", Chords, MEASURE_FUNC },
	{ "tempo", Tempo, MEASURE_FUNC },
	{ "time", Time, POS_FUNC },
	{ "t", Time, POS_FUNC },
	{ "accents", Accents, POS_FUNC },
	{ "sc", ShortChords, POS_FUNC },
	{ "repeat", Repeat, CONTROL_FUNC },
	{ "goto", Goto, CONTROL_FUNC },
	{ "end", End, CONTROL_FUNC },
	{ "label", Label, CONTROL_FUNC },
	{ NULL, NULL, 0 }
};

static int DecodePosLine(), DecodeNormLine(), DecodeControl();

int DecodeBlock()
{
	char buff[LINE_SIZE];
	char command[LINE_SIZE];
	int n, i, status, ret_val, line_num;
	int mes_processed;
	char *descr, *s;
	void FlushHeap();

	/* Init things. */
	for (i = 0; i < LINE_SIZE; i++)
		pos_to_duration[i] = -1;
	line_num = -1;
	mes_processed = 0;

	/* Main loop. */
	while (1) {
		if (NULL == fgets(buff, LINE_SIZE, input_file)) {
			status = STAT_EOF;
			break;
		}  
		status = STAT_OK;
		if (debug) 
			fprintf(stderr, "Decoding > %s", buff);
	
		/* Skip comments */
		if (buff[0] == ';') 
			continue;
		if (strspn (buff, "\n \t") == strlen (buff)) /* Blank line. */
			break;
		line_num++; /* Commented lines are not counted. */

		/* Erase comments at the end of a line. */
		if ((s = index (buff, ';'))) 
			*s = '\0';

		/* First token. */
		if (buff[0] == '|') {
			if ( (def_command != NULL) && (def_command[line_num] != NULL) )
				strcpy (command, def_command[line_num]);
			else {
				fprintf (stderr, "Default command not defined for line # = %d\n", 
					line_num);
				exit (1);
			}
		} else if (buff[0] == ' ') 
			strcpy (command, "time");
		else {
			n = strcspn (buff, " \t\n|");
			command[0] = '\0';
			strncat (command, buff, n);
		}

		/* Look for a built-in function. */
		for (i = 0; functions[i].name != NULL; i++) {
			if (strcmp (command, functions[i].name) == 0) {
				if (functions[i].type == MEASURE_FUNC) {
					ret_val = DecodeNormLine (buff, functions[i].func);
				} else if (functions[i].type == POS_FUNC) {
					ret_val = DecodePosLine (buff, functions[i].func, NULL);
				} else if (functions[i].type == CONTROL_FUNC) {
					ret_val = DecodeControl (buff, functions[i].func);
				}
				break;
			}
		}
		if (functions[i].type == CONTROL_FUNC) {
			break;
		}
		if (functions[i].name != NULL) {
			mes_processed = MAX (mes_processed, ret_val);
			continue;
		}

		/* Look for config-file defined function. */
		descr = HashGet (command, TYPE_INSTRUMENT);
		if (descr) {
			struct instrument *instr;
			instr = (struct instrument *) descr;
			if (instr->type == TYPE_TAB)
				ret_val = DecodePosLine (buff, Tab, descr);
			else if (instr->type == TYPE_PERCUSSION)
				ret_val = DecodePosLine (buff, Perc, descr);
			else 
				Fail ("Unknown instrument type.\n");
			mes_processed = MAX (mes_processed, ret_val);
			continue;
		}
		descr = HashGet (command, TYPE_CHORD);
		if (descr) {
			ret_val = DecodePosLine (buff, Chord, descr);
			mes_processed = MAX (mes_processed, ret_val);
			continue;
		}
	}

	/* Update for next block. */
	if (debug)
		fprintf (stderr, "decode : mes_processed = %d\n", mes_processed);
	for (i = current_measure; i < current_measure + mes_processed; i++) {
		current_time += measures[i].duration;
	}
	current_measure += mes_processed;

	/* Flush heap... */
	if (debug) 
		fprintf (stderr, "Flushing heap. (current time = %d, current mes = %d)\n",
			current_time, current_measure);
	FlushHeap (current_time);

	return status;
}

/* Flush heap... */
void FlushHeap (t)
unsigned long t;
{
	struct event evn;
	static void WriteEvent();

	if (debug) 
		fprintf (stderr, "Flushing heap. (time = %d)\n", t);
	while (1) {
		if ( HeapGet (&evn, current_time) == STAT_EOF ) 
			break;
		
		WriteEvent (&evn);
	}
}

static void WriteEvent(evn)
struct event *evn;
{
	static unsigned long time = 0;
	int delta_time;

	delta_time = evn->time - time;
	time = evn->time;

	if (debug)
		fprintf (stderr, "Writing event (time = %d)\n", time);

	switch (evn->type) {
	case MIDI_EVENT :
		(void) mf_w_midi_event (delta_time, evn->op, evn->chan - 1, 
			evn->data.midi_codes, evn->length);
		break;
	case TEMPO_EVENT :
		if (debug)
			fprintf (stderr, "writing tempo = %d\n", evn->data.tempo);
		(void) mf_w_tempo (delta_time, evn->data.tempo);
		break;
	default : 
		fprintf (stderr, "Event type not implemented\n");
	}
}

/* We look for the measure separator ('|'), and send the token to the function
``func''. There is supposed to be only one token per measure. */

static int DecodeNormLine (line, func)
char *line;
int (*func)();
{
	int i, n, mes_processed;
	int time;
	char *s;
	char param[LINE_SIZE];

	/* We parse the line. */
	s = line;
	time = current_time;
	mes_processed = 0;
	for (i = 0; ; i++) {
		s = index(s, '|');
		if (s==NULL) 
			break;
		s++;
		/* Skip leading blanks. */
		while(s[0] == ' ' || s[0] == '\t') 
			s++;
		if (s[0] == '|' || s[0] == '\0') 
			param[0] = '\0';
		else {
			n = strcspn (s, " \n\t");
			strncpy(param, s, n);
			param[n] = '\0';
		}
		if (NULL == index(s, '|')) 
			break;

		mes_processed++;

		/* Don't call the function if the argument is "". */
		if (param[0]) { 
			if (func(current_measure + i, param, time) == STAT_ERROR) {
				fprintf(stderr, "Something wrong in file %s, line\n> %s\n", 
					file_name, line);
				exit(1);
			}
		}
		time += measures[current_measure + i].duration;
	}
	if (debug)
		fprintf (stderr, "i=%d\n", i);
	return mes_processed;
}

/* Decode the line in buffer ``line'', using actions specified by function
``func''. Note that here we can't use standard library functions because
we have to keep track of the position (tab expansion). */
/* Euh... Maybe we could first expand the tabs then use standard library... */

static int DecodePosLine (line, func, descr)
char *line;
int (*func)();
char *descr;
{
	char *s;
	int mes, i, true_pos, time, begin_mes_time;
	int begin_token_time, begin_token_pos;
	int flag, mes_processed;
	char token[NAME_LENGTH];

	if (func == Time) {
		for (i=0; i<LINE_SIZE; i++) 
			pos_to_duration[i] = -1;
	}

	/* We tokenize the line using ' ', '\t' and '|' separators. */
	mes = current_measure;
	time = begin_mes_time = current_time;
	if (debug)
		fprintf (stderr, "current mes = %d, time = %d\n", mes, time);
	s = line;
	true_pos = 0;
	mes_processed = 0;

	/* First, skip first token. */
	for (i = 0; i < strlen (s); i++) {
		if (s[i] == ' ' || s[i] == '|')
			break;
		if (s[i] == '\t') {
			break;
		}
		if (s[i] == '\0' || s[i] == '\n') 
			return 0;
		true_pos++;
	}
	for ( ; i < strlen (s); i++) {
		if (s[i] == ' ') {
			true_pos++;
			continue;
		}
		if (s[i] == '\t') {
			true_pos = (true_pos / tab_length + 1) * tab_length;
			continue;
		}
		break;
	}

	/* Determine the first state of the automata. */
	if (s[i] == '|') { /* We don't want to change measure number. */
	 	flag = 0;
		i++;
		true_pos++;
		token[0] = '\0';
	} else if (s[i]=='-') {
		flag = 0;
		token[0] = '\0';
	} else {
		flag = 1;
		token[0] = '\0';
	}
	begin_token_pos = true_pos;
	begin_token_time = time;

	/* Now this is a state automaton with 2 states. The state is in ``flag'',
	1 means reading a token, 0 means skipping blanks. */

	for ( ; i < strlen (s); i++) {
		if (flag == 1) { /* Reading token. */
			if (s[i]=='\n' || s[i]==' ' || s[i]=='\t' || s[i]=='|' || s[i]=='-') {
				flag = 0;
				mes_processed = mes - current_measure + 1;
				if (STAT_OK != func (mes, token, begin_token_time, begin_token_pos,
					descr)) {
					fprintf(stderr, "Something wrong in file %s, line\n> %s\n", 
						file_name, line);
					exit(1);
				}
			} else {
				char str[2];

				str[0] = s[i];
				str[1] = '\0';
				strcat (token, str);
			}
		} else { /* flag == 0 : skipping blanks. */
			if (s[i]!='\n' && s[i]!=' ' && s[i]!='|' && s[i]!='\t' && s[i]!='-') {
				token[0] = s[i];
				token[1] = '\0';
				flag = 1;
				begin_token_pos = true_pos;
				begin_token_time = time;
			}
		}
		if (s[i] == '|') {
			begin_mes_time += measures[mes].duration;
			time = begin_mes_time;
			mes++;
			mes_processed = mes - current_measure;
			if (debug) 
				fprintf (stderr, "current mes = %d, time = %d\n", mes, time);
		}

		if (s[i] == '\t') {
			int j, old_true_pos;

			old_true_pos = true_pos;
			true_pos = (true_pos / tab_length + 1) * tab_length;
			for (j = old_true_pos; j < true_pos; j++) 
				time += MAX (0, pos_to_duration[j]);
		} else {
			time += MAX (0, pos_to_duration[true_pos]);
			if (0 && debug)
				fprintf (stderr, "time = %d, true_pos = %d\n", time, true_pos);
			true_pos++;
		}
	}

	if (debug)
		fprintf (stderr, "mes_processed = %d\n", mes_processed);
	return mes_processed;
}

static int DecodeControl (char *line, int (*func)())
{
	int Tokenize (); /* from misc.c */
	char list[MAX_TOK][NAME_LENGTH];

	Tokenize(line,list);
	func(list);
}
