/* config.c : decodes the lines in the input file, initialize things. */

#include <stdio.h>
#include <string.h>

#ifndef VMS
#include <malloc.h>
#include <values.h>
#else
#include "values_vms.h"
#endif

#include "midifile.h"
#include "tabul.h"

/* from main.c */
void Fail();
extern int debug, verbose;
extern FILE *output_file;

/* from hash.c */
void HashInit();
char *HashGet();
void HashPut();
void HashDestroy();
void HashStat();

/* from heap.c */
void HeapInit();
void HeapPut();

/* exported definitions */
char *def_signature = "4/4";
int tab_length = 8;
unsigned long def_tempo = DEF_TEMPO;
int tempo_set_from_args = 0;
int division = 96; /* Number of ``ticks'' per quarter note. */
int measures_size = 500; /* Number of measures in memory. */
struct measure *measures;
char **def_command;

struct funct_info {
	char *name;
	int (*func)();
	int nargs; /* argument number for the function, function name included.
			0 means variable number */
};

static int TabLength(), Signature(), Perc(), Tempo();
static int Chord(), Duration(), Division(), DefSignature(), DefComm();
static int Patch(), Tab(), Symbol(), Measures(), Heap();

struct funct_info function_map[] = {
	{ "tablength", TabLength, 2 },
	{ "signature", Signature, 3 },
	{ "perc", Perc, 5 },
	{ "duration", Duration, 3 },
	{ "tempo", Tempo, 2 },
	{ "chord", Chord, 0 },
	{ "division", Division, 2 },
	{ "def_signature", DefSignature, 2 },
	{ "def_comm", DefComm, 0 },
	{ "patch", Patch, 3 },
	{ "tab", Tab, 5 },
	{ "symbol", Symbol, 2 },
	{ "measures", Measures, 2 },
	{ "heap", Heap, 2 },
	{ NULL, NULL, 0 }
};

/* Read the input file during config phase. 
Return STAT_EOF on EOF, STAT_END_CONFIG at end of config. */
int ReadConfig(in_file, file_name)
FILE *in_file;
char *file_name;
{
	char buff [LINE_SIZE], buff2 [LINE_SIZE];
	int Tokenize(); /* from misc.c */
	char tok_list [MAX_TOK] [NAME_LENGTH];
	int n_tok, i;
	char *s;

	while (1) {
		s = fgets(buff, LINE_SIZE, in_file);
		if (s == NULL)
			break;
		if (strlen(buff) == LINE_SIZE - 1) {
			fprintf(stderr, "Line too long in file %s, could not process.\n", file_name);
			break;
		}

		strncpy(buff2, buff, LINE_SIZE);
		if (debug) 
			fprintf(stderr, "> %s", buff2);

		n_tok = Tokenize(buff, tok_list);

		if (n_tok == 0) 
			continue; /* Skip comments. */
		if (n_tok == STAT_TOK_TOO_LONG) {
			fprintf(stderr, "Token too long (max %d characters) in file %s, in line %s\n", NAME_LENGTH, file_name, buff2);
			continue;
		}
		if (n_tok == STAT_TOO_MUCH_TOK) {
			fprintf(stderr, "Too much tokens (max %d) in file %s, in line %s\n", MAX_TOK, file_name, buff2);
			exit(1);
		}

		if (strcmp(tok_list[0], "beginsong") == 0) {
			return STAT_END_CONFIG;
		}

		for (i = 0; function_map[i].name != NULL; i++) {
			if (strcmp(tok_list[0], function_map[i].name) == 0) {
				if ( (function_map[i].nargs != 0) && (function_map[i].nargs != n_tok)) {
					fprintf(stderr, "Wrong parameters number in file %s, line : %s\n",
						file_name, buff2);
					exit(1);
				}
				function_map[i].func(tok_list);
				break;
			}
		}
		if (function_map[i].name != NULL)
			continue;

		fprintf(stderr, "Command not recognized in file %s : %s\n",
			file_name, buff2);
	}
	return STAT_EOF;
}

void BeginConfig()
{
	HashInit (HASH_SIZE);
	HeapInit (HEAP_SIZE);
}

void FinishConfig()
{
	int i;
	struct signature *sign;
	struct event evn;
	void MidiInit ();

	/* Allocate memory and init measures. */
	measures = (struct measure *) malloc (sizeof(struct measure) 
		* measures_size);
	sign = (struct signature *) HashGet (def_signature, TYPE_SIGNATURE);
	if (sign == NULL) {
		Fail ("Default signature not defined\n");
	}
	for (i = 0; i < measures_size; i++) {
		measures[i].duration = sign->duration;
		measures[i].signature = sign;
	}

	/* Write headers in output file. */
	MidiInit ();

	/* Write tempo. */
	evn.type = TEMPO_EVENT;
	evn.time = 0;
	evn.mutime = 0;
	evn.data.tempo = def_tempo;
	HeapPut (&evn);

	if (verbose)
		HashStat();
}

/* Initialisation commands. */

/* Set the tablength. */
static int TabLength (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	tab_length = atoi (list[1]);
	return STAT_OK;
}

/* Create a signature entry in symbol table. */
static int Signature (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	struct signature *sign;

	sign = (struct signature *) malloc(sizeof(struct signature));
	if (sign == NULL) 
		Fail ("Not enough memory\n");

	sign->name = strdup (list[1]);
	sign->duration = atoi (list[2]);
	HashPut(sign->name, TYPE_SIGNATURE, (char *) sign);
	return STAT_OK;
}

/* Create a percussion entry in symbol table. */
static int Perc(list)
char list [MAX_TOK] [NAME_LENGTH];
{
	struct instrument *instr;

	instr = (struct instrument *) malloc(sizeof(struct instrument));
	instr->name = strdup (list[1]);
	instr->type = TYPE_PERCUSSION;
	instr->channel = atoi (list[2]);
	instr->pitch = atoi (list[3]);
	instr->volume = atoi (list[4]);
	if (debug) 
		fprintf (stderr, "Perc : name = %s, chan = %d, pitch = %d, vol = %d\n",
			instr->name, instr->channel, instr->pitch, instr->volume);

	HashPut (instr->name, TYPE_INSTRUMENT, (char *) instr);
	return STAT_OK;
}

long DecodeTempo (param)
char *param;
{
	double x;
	double atof();
	long t;

	x = atof (param);
	if (debug)
		fprintf (stderr, "Tempo = %f (%s)\n", x, param);
	if (x > 1000)
		t = atoi (param);
	else
		t = (long) (60.0 / x * 1e6); /* attention : conversion ? */
	return t;
}

/* Set default tempo. */
static int Tempo(list)
char list [MAX_TOK] [NAME_LENGTH];
{
	if (! tempo_set_from_args)
		def_tempo = DecodeTempo (list[1]);

	return STAT_OK;
}

/* Default commands in order. */
static int DefComm (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	int i;

	def_command = (char **) malloc (MAX_TOK * sizeof(char *));
	if (def_command == NULL)
		Fail ("Could npt allocate memory for default commands.\n");

	if (list[1] == NULL) {
		fprintf (stderr, "Wrong number of arguments.\n");
		return STAT_ERROR;
	}
	for (i = 1; list[i][0] != '\0'; i++) 
		def_command[i-1] = strdup (list[i]);
	
	def_command[i] = NULL;
	return STAT_OK;
}

/* Create a chord entry in symbol table. */
static int Chord (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	struct chord *chord;
	int i;
	int Note2Pitch(); /* from misc.c */

	if ( (list[1] == NULL) || (list[2] == NULL) ) {
		fprintf (stderr, "Wrong number of arguments.\n");
		return STAT_ERROR;
	}

	chord = (struct chord *) malloc (sizeof(struct chord));
	chord->name = strdup (list[1]);
	chord->type = 0; /* to be modified */
	if (chord->name[0] == 'X') { /* ``generic chord'' */
		for (i = 2; list[i][0] != '\0'; i++) {
			chord->notes[i-2] = atoi (list[i]);
		}
		chord->notes[i-2] = -1;
	} else {
		for (i = 2; list[i][0] != '\0'; i++) {
			chord->notes[i-2] = Note2Pitch (list[i]);
		}
		chord->notes[i-2] = -1;
	}
	HashPut (chord->name, TYPE_CHORD, (char *) chord);
	return STAT_OK;
}

/* Create a duration entry in symbol table. */
static int Duration (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	struct duration *dur;

	dur = (struct duration *) malloc(sizeof(struct duration));
	if (dur == NULL) 
		Fail ("Not enough memory\n");

	dur->name = strdup (list[1]);
	dur->duration = atoi (list[2]);
	HashPut (dur->name, TYPE_DURATION, (char *) dur);
	return STAT_OK;
}

static int Tab (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	struct instrument *instr;

	instr = (struct instrument *) malloc(sizeof(struct instrument));
	if (instr == NULL)
		Fail ("Could alloc memory for instrument.\n");
	instr->name = strdup (list[1]);
	instr->type = TYPE_TAB;
	instr->channel = atoi (list[2]);
	instr->volume = atoi (list[3]);
	instr->open_note = atoi (list[4]);

	HashPut (instr->name, TYPE_INSTRUMENT, (char *) instr);
	return STAT_OK;
}

static int Patch (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	static int count = 0;
	struct event evn;

	evn.type = MIDI_EVENT;
	evn.time = 0;
	evn.mutime = count++;
	evn.op = program_chng;
	evn.chan = (unsigned char) atoi (list [1]);
	evn.length = 1;
	evn.data.midi_codes[0] = (unsigned char) atoi (list [2])-1;

	HeapPut (&evn);

	return STAT_OK;
}

/* Destroy old table of symbol and create a (bigger) one. */
static int Symbol (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	HashDestroy ();
	HashInit (atoi (list[1])); 

	return STAT_OK;
}

/* Destroy old heap and create a (bigger) one. */
static int Heap (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	void HeapDestroy (); /* from heap.c */

	HeapDestroy ();
	HeapInit (atoi (list[1])); 

	return STAT_OK;
}

static int Measures (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	measures_size = atoi (list[1]);

	return STAT_OK;
}

static int Division (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	division = atoi (list[1]);

	return STAT_OK;
}

static int DefSignature (list)
char list [MAX_TOK] [NAME_LENGTH];
{
	def_signature = strdup (list[1]);

	return STAT_OK;
}

/* Fonctions to write to the midifile. */

static int OutputC (c)
unsigned char c;
{
	return fputc (c, output_file);
}

static trk_length_offset;
extern int laststat, lastmeta, Mf_numbyteswritten;

/* Write the headers of the midifiles. */
void MidiInit ()
{
	void write16bit(), write32bit(); /* from midifile.c */

	Mf_putc = OutputC;

	/* Write header. */
	write32bit (MThd);
	write32bit (6); /* length */
	write16bit (0); /* format */
	write16bit (1); /* track # */
	write16bit (division);

	/* Write track header. */
	trk_length_offset = ftell (output_file);
	write32bit (MTrk);
	write32bit (0); /* temporary length */

	Mf_numbyteswritten = 0;
	laststat = 0;
}

/* Write track end and close midifile. */
void MidiClose ()
{
	int n;
	void FlushHeap(); /* from decode.c */

	FlushHeap (MAXINT);

  if (laststat != meta_event || lastmeta != end_of_track) {
      /* mf_write End of track meta event */
      eputc (0);
      eputc (meta_event);
      eputc (end_of_track);
      eputc (0);
  }
  laststat = 0;

  if (fseek (output_file, trk_length_offset, 0) < 0)
    Fail ("Error seeking during final stage of write.\n");

  /* Re-mf_write the track chunk header with right length */
	n = Mf_numbyteswritten;
  write32bit (MTrk);
  write32bit (n);
	if (debug)
		fprintf (stderr, "Mf_numbyteswritten = %d\n", Mf_numbyteswritten);

	fclose (output_file);
}
