/* commands.c : commands interpreted during decoding phase */
/* Copyright 1993 Stefane Fermigier. */

#include <stdio.h>
#include <string.h>
#include <strings.h>

#ifndef VMS
#include <values.h>
#else
#include "values_vms.h"
#endif

#include "midifile.h"
#include "tabul.h"

/* from main.c */
extern int debug;

/* from config.c */
extern struct measure *measures;

/* from hash.c */
char *HashGet();

/* from heap.c */
void HeapPut();
int HeapGet();

/* from decode.c */
extern int pos_to_duration[LINE_SIZE];
extern int pos_to_accent[LINE_SIZE]; 

int Sig (mes, param, time)
int mes, time;
char *param;
{
	struct signature *sig;

	sig = (struct signature *) HashGet (param, TYPE_SIGNATURE);
	if (sig == NULL) {
		fprintf (stderr, "Signature '%s' not defined.\n", param);
		return STAT_ERROR;
	} else {
		measures[mes].duration = sig->duration;
		return STAT_OK;
	}
}

int chords_velocity = 50;

int PlayChord (char *name, int begin_time, int end_time)
{
	struct chord *chord;
	char buff[NAME_LENGTH];
	int i, status;

	/* Look for the chord in symbol table. */
	chord = (struct chord *) HashGet (name, TYPE_CHORD);

	status = STAT_OK;
	if (chord != NULL) {
		for (i=0; chord->notes[i] >= 0; i++) {
			struct event evn;

			evn.type = MIDI_EVENT;
			evn.time = begin_time;
			evn.mutime = MAXINT;
			evn.length = 2;
			evn.op = note_on;
			evn.chan = 1; /* channel */
			evn.data.midi_codes[0] = chord->notes[i];
			evn.data.midi_codes[1] = chords_velocity; /* velocity */;

			HeapPut(&evn);

			evn.op = note_off;
			evn.time = end_time;
			evn.mutime = 0;

			HeapPut(&evn);
		}
	} 
	
	else { /* Generic chord */
		if (name[1] == 'b' || name[1] == '#') 
			strcpy(buff, name+1);
		else
			strcpy(buff, name);
		buff[0] = 'X';
		chord = (struct chord *) HashGet (buff, TYPE_CHORD);
		if (chord == NULL) {
			fprintf (stderr, "Chord '%s' not defined.\n", name);
			status = STAT_ERROR;
		} else {
			int root;
			char root_name[4];

			root_name[0] = name[0];
			if (name[1] == 'b' || name[1] == '#') {
				root_name[1] = name[1];
				root_name[2] = '4';
				root_name[3] = 0;
			} else {
				root_name[1] = '4';
				root_name[2] = 0;
			}
			root = Note2Pitch(root_name);
			for (i=0; chord->notes[i] >= 0; i++) {
				struct event evn;

				evn.type = MIDI_EVENT;
				evn.time = begin_time;
				evn.mutime = MAXINT;
				evn.length = 2;
				evn.op = note_on;
				evn.chan = 1; /* channel */
				evn.data.midi_codes[0] = root + chord->notes[i];
				evn.data.midi_codes[1] = chords_velocity; 
	
				HeapPut(&evn);
	
				evn.op = note_off;
				evn.time = end_time;
				evn.mutime = 0;
	
				HeapPut(&evn);
			}
		}
	}
	return status;
}

int ShortChords (int mes, char *param, int time, int pos, char *descr)
{
	if (debug) 
		fprintf (stderr, "ShortChords called with parameters : %d %s %d %d.\n", 
			mes, param, time, pos);
	return PlayChord(param, time, time+pos_to_duration[pos]);
}

int Chords (int mes, char *param, int time)
{
	return PlayChord(param, time, time + measures[mes].duration);
}

int Tempo (mes, param, time)
int mes, time;
char *param;
{
	struct event evn;
	long DecodeTempo(); /* from config.c */

	evn.type = TEMPO_EVENT;
	evn.time = time;
	evn.mutime = 0;
	evn.data.tempo = DecodeTempo (param);;

	HeapPut(&evn);

	return STAT_OK;
}

int Accents (mes, param, time, pos, instr)
int mes, pos, time; 
char *param, *instr;
{
	printf ("Accents not implemented in this version.\n");
	return STAT_OK;
}

int Time (mes, param, time, pos, descr)
int mes, pos, time; 
char *param, *descr;
{
	struct duration *dur;
	int total_dur;
	char buff[NAME_LENGTH], *s, *t;

	strcpy (buff, param);
	s = buff;
	total_dur = 0;
	do {
		t = strchr (s, '+');
		if (t) 
			t[0] = '\0';
		dur = (struct duration *) HashGet (s, TYPE_DURATION);
		if (dur == NULL) {
			fprintf (stderr, "Duration '%s' not defined.\n", s);
			return STAT_ERROR;
		}
		total_dur += dur->duration;
		s = t+1;
	} while (t != NULL);

	pos_to_duration[pos] = total_dur;

	return STAT_OK;
}

int Tab (mes, param, time, pos, descr)
int mes, pos, time; 
char *param, *descr;
{
	struct instrument *instr;
	struct event evn;
	int dur;

	if (debug) 
		fprintf (stderr, "Tab called with parameters : %d %s %d %d.\n", 
			mes, param, time, pos);
	instr = (struct instrument *) descr;
	dur = pos_to_duration[pos];

	if (debug) 
		fprintf (stderr, "Tab : dur = %d\n", dur);

	if (dur < 0) {
		fprintf (stderr, "Wrong position of note in tab line.\n");
		return STAT_ERROR;
	}

	evn.type = MIDI_EVENT;
	evn.time = time;
	evn.mutime = MAXINT;
	evn.length = 2;
	evn.op = note_on;
	evn.chan = instr->channel;
	evn.data.midi_codes[0] = instr->open_note + atoi (param);
	evn.data.midi_codes[1] = instr->volume;

	HeapPut(&evn);

	evn.op = note_off;
	evn.time = time + dur;
	evn.mutime = 0;

	HeapPut(&evn);

	return STAT_OK;
}

int Perc (mes, param, time, pos, descr)
int mes, pos, time; 
char *param, *descr;
{
	struct instrument *instr;
	struct event evn;
	int dur;

	if (debug) 
		fprintf (stderr, "Perc called with parameters : %d %s %d %d.\n", 
			mes, param, time, pos);
	instr = (struct instrument *) descr;
	dur = pos_to_duration[pos];
	if (dur < 0) {
		fprintf (stderr, "Wrong position of note in perc line.\n");
		return STAT_ERROR;
	}

	evn.type = MIDI_EVENT;
	evn.time = time;
	evn.mutime = MAXINT;
	evn.length = 2;
	evn.op = note_on;
	evn.chan = instr->channel;
	evn.data.midi_codes[0] = instr->pitch;
	evn.data.midi_codes[1] = instr->volume;

	HeapPut(&evn);

	return STAT_OK;
}

int Chord (mes, param, time, pos, descr)
int mes, pos, time; 
char *param, *descr;
{
	struct chord *chord;
	struct event evn;
	int i, dur;

	chord = (struct chord *) descr;

	if (param[0] = 'x')  {
		dur = pos_to_duration[pos];
		if (dur < 0) {
			fprintf (stderr, "Wrong position of note in perc line.\n");
			return STAT_ERROR;
		}
	} else {
		struct duration *duration;
		duration = (struct duration *) HashGet (param, TYPE_DURATION);
		if (duration == NULL) {
			fprintf (stderr, "Unknow duration in chord line.\n");
			return STAT_ERROR;
		}
		dur = duration->duration;
	}

	for (i=0; chord->notes[i] >= 0; i++) {
		evn.type = MIDI_EVENT;
		evn.time = time;
		evn.mutime = MAXINT;
		evn.length = 2;
		evn.op = note_on;
		evn.chan = 0; /* channel */
		evn.data.midi_codes[0] = chord->notes[i];
		evn.data.midi_codes[1] = 100; /* velocity */;

		HeapPut(&evn);

		evn.op = note_off;
		evn.time = time + dur;
		evn.mutime = 0;

		HeapPut(&evn);
	}

	return STAT_OK;
}
