/*LINTLIBRARY*/

/*  @(#)motif.c 1.3 94/01/27
 *
 *  X11 (Motif) dependent graphics routines used by reve.
 *
 *  Copyright (c) 1990 - 1994 - Rich Burridge & Yves Gallot.
 *  All rights reserved.
 *
 *  Permission is granted to copy this source, for redistribution
 *  in source form only, provided the news headers in "substantially
 *  unaltered format" are retained, the introductory messages are not
 *  removed, and no monies are exchanged.
 *
 *  Permission is also granted to copy this source, without the
 *  news headers, for the purposes of making an executable copy by
 *  means of compilation, provided that such copy will not be used
 *  for the purposes of competition in any othello tournaments, without
 *  prior permission from the authors.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file), then an attempt will be made to fix them.
 */

#include "reve.h"
#include <sys/stat.h>

#ifndef VMS
#include <sys/param.h>
#endif

#include "widgetindex.h"
#include "xdefs.h"

#include <X11/keysym.h>
#include <X11/cursorfont.h>
#include <Xm/Xm.h>
#include <Xm/AtomMgr.h>
#include <Xm/Protocols.h>
#include <Xm/Text.h>

#include "move.xbm"
#include "suggest.xbm"

#include "reve.xbm"
#include "reve.xpm"

#define  MAXICONCOLORS      8      /* Maximum colors in reve icon. */
#define  TITLE_LINE_HEIGHT  25

CVars c ;                   /* REVE/reve_proc global variables */
Vars v ;                    /* Reve variables and options. */
XVars X ;                   /* X11/Motif variables. */

static int get_strwidth            (char *) ;

static Pixel get_color             (char *) ;
static Pixmap create_mp_pixmap     (char *) ;

static void add_button_callbacks   () ;
static void dismiss_popup          (Widget, XtPointer, XmAnyCallbackStruct *) ;
static void draw_text              (int, int, int, int, char *) ;
static void force_popup_on_screen  (Widget, int *, int *) ;
static void event_proc             (Widget, XtPointer, XEvent *, Boolean *) ;
static void get_font               () ;
static void get_screen_size        (int *, int *) ;
static void make_icon              () ;
static void make_pieces            () ;
static void position_help          (enum help_type) ;
static void position_popup         (Widget, Widget) ;
static void redraw_square          (Widget w, XtPointer,
                                    XmDrawnButtonCallbackStruct *) ;
static void set_clocks_active      () ;
static void x_error_proc           (Display *, XErrorEvent *) ;

static Widget get_board_widget     (int, int) ;


void
add_delete_callback(Widget widget)
{
  Atom atom ;

  atom = XmInternAtom(XtDisplay(widget), "WM_DELETE_WINDOW", False) ;
  XmAddWMProtocolCallback(widget, atom, (XtCallbackProc) dismiss_popup, NULL) ;
}


void
beep()
{
  XBell(X->dpy, 56) ;
  XFlush(X->dpy) ;
}


void
close_reve()
{
  XEvent event ;

  event.xclient.type         = ClientMessage ;
  event.xclient.display      = X->dpy ;
  event.xclient.window       = XtWindow(X->warr[WI_MAIN_FRAME]) ;
  event.xclient.message_type = XInternAtom(X->dpy, "WM_CHANGE_STATE", 0) ;
  event.xclient.format       = 32 ;
  event.xclient.data.l[0]    = IconicState ;
  XSendEvent(X->dpy, X->root, 0,
             SubstructureRedirectMask | SubstructureNotifyMask, &event) ;
  XFlush(X->dpy) ;
}


void
connect_io()     /* Connect to possible computer process and remote human. */
{
#ifdef HASREVEPROC
  if (v->pipe_io[1][0] > 0)             /* Separate computer program. */
    XtAppAddInput(tu_global_app_context, v->pipe_io[1][0], XtInputReadMask,
                  (XtInputCallbackProc) read_from_reve, NULL) ;
#endif /*HASREVEPROC*/
#ifndef NO_NETWORK
  if (v->socketfd > 0)                  /* Remote human (user@host). */
    XtAppAddInput(tu_global_app_context, v->socketfd, XtInputReadMask,
                  (XtInputCallbackProc) read_from_sock, NULL) ;
#endif /*!NO_NETWORK*/
}


/*  Minimal routine to create a pixmap for the color reve icon. This
 *  routine assumes that the data file is in the correct format.
 */
 
static Pixmap
create_mp_pixmap(char *buf)
{
  char c_ch[MAXICONCOLORS], c_str[MAXICONCOLORS][MAXLINE] ;
  char *p ;
  Colormap cmap ;
  GC gc ;
  int height, i, j, ncolors, nchars, width ;
  Pixmap pixmap = 0 ;
  unsigned char *data, *dp ;
  unsigned char lkup_tbl[256] ;
  XColor ccol ;
  XImage *image ;

  p = buf ;

/* Extract width, height, number of colors and number of characters/pixel. */

  SSCANF(p, "%d %d %d %d", &width, &height, &ncolors, &nchars) ;
  while (*p != '\n') p++ ;
  p++ ;

  for (i = 0; i < ncolors; i++)
    {
      c_ch[i] = *p ;
      SSCANF(p+4, "%s", c_str[i]) ;
      while (*p != '\n') p++ ;
      p++ ;
    }

  for (i = 0; i < 256; i++) lkup_tbl[i] = 0 ;
  cmap = DefaultColormap(X->dpy, X->screen) ;
  for (i = 0; i < ncolors; i++)
    if (XParseColor(X->dpy, cmap, c_str[i], &ccol))
      if (XAllocColor(X->dpy, cmap, &ccol) == True)
        lkup_tbl[(unsigned int) c_ch[i]] = (unsigned char) ccol.pixel ;

  data = (unsigned char *) malloc((unsigned)(width * height)) ;
  if (data == NULL) return(pixmap) ;

  dp = data ;
  for (i = 0; i < height; i++)
    {
      for (j = 0; j < width; j++)
        *dp++ = lkup_tbl[(unsigned int) *p++] ;
      while (*p != '\n') p++ ;
      p++ ;
    }
  gc = DefaultGC(X->dpy, X->screen) ;
  image = XCreateImage(X->dpy, DefaultVisual(X->dpy, X->screen),
                       X->depth, ZPixmap, 0, (char *) data,
                       width, height, 8, 64) ;
  pixmap = XCreatePixmap(X->dpy, X->root, width, height, X->depth) ;
  XPutImage(X->dpy, pixmap, gc, image, 0, 0, 0, 0, width, height) ;
  return(pixmap) ;
}


static void
dismiss_popup(Widget widget, XtPointer data, XmAnyCallbackStruct *cbs)
{
  XtUnmapWidget(widget) ;
  XtUnmanageChild(widget) ;
}


void
do_show_help(enum help_type htype)
{
  if (!X->hwarr)
    {
      tu_help_frame_widget("help", X->warr[WI_MAIN_FRAME], &X->hwarr) ;
      add_delete_callback(X->hwarr[WI_HELP_FRAME]) ;
      load_help_file(v->helpfile) ;
    }
  position_help(htype) ;
  show_popup(X->hwarr[WI_HELP_FRAME]) ;
}


void
do_startup()
{
  char **argv ;
  Dimension cellwidth, cellheight ;
  int argc, i ;
  Pixmap pixmap ;

  X          = (XVars) LINT_CAST(calloc(1, sizeof(XObject))) ;
  X->dpy     = tu_disp_get_dpy(NULL) ;
  X->screen  = DefaultScreen(X->dpy) ;
  X->root    = RootWindow(X->dpy, X->screen) ;
  X->depth   = DefaultDepth(X->dpy, X->screen) ;
  X->foregnd = BlackPixel(X->dpy, X->screen) ;
  X->backgnd = WhitePixel(X->dpy, X->screen) ;

  XSetErrorHandler((XErrorHandler) x_error_proc) ;
  pixmap = XCreatePixmap(X->dpy, X->root, 64, 64, 1) ;
  X->gc_mask = GCForeground | GCBackground | GCGraphicsExposures ;
  X->gc_val.foreground = X->foregnd ;
  X->gc_val.background = X->backgnd ;
  X->gc_val.graphics_exposures = False ;
  X->igc = XCreateGC(X->dpy, pixmap, X->gc_mask, &X->gc_val) ;

  X->move_bm    = XCreatePixmapFromBitmapData(X->dpy, X->root,
                    (char *) move_bits, move_width, move_height, 1, 0, 1) ;
  X->suggest_bm = XCreatePixmapFromBitmapData(X->dpy, X->root,
                    (char *) suggest_bits, suggest_width, suggest_height,
                    1, 0, 1) ;

  X->normal_cur = XCreateFontCursor(X->dpy, XC_top_left_arrow) ;
  X->hour_cur   = XCreateFontCursor(X->dpy, XC_watch) ;

  X->gc    = XCreateGC(X->dpy, X->root, X->gc_mask, &X->gc_val) ;
  X->ropgc = XCreateGC(X->dpy, X->root, X->gc_mask, &X->gc_val) ;
  XSetFillStyle(X->dpy, X->ropgc, FillStippled) ;

  argc = tu_get_cmd_argc() ;
  argv = (char **) calloc(argc, sizeof(char *)) ;
  for (i = 0; i < argc; i++)
    argv[i] = tu_get_cmd_argv(i) ;

  c = (CVars) LINT_CAST(calloc(1, sizeof(CompVars))) ;
  c->max_depth = 2 ;            /* Computer strategy - maximum depth. */

  v = (Vars) LINT_CAST(calloc(1, sizeof(ReveVars))) ;
  init_data(argv[0]) ;

  initialise() ;                /* Initialise variables used by reve. */
  load_resources() ;            /* Get resources from various places. */
  read_resources() ;            /* Read resources from merged database. */
  get_options(argc, argv) ;     /* Read and process command line options. */
  init_clocks() ;               /* Setup timer clocks. */
  set_display_types() ;         /* Work out what displays to initialise. */

  (void) tu_main_frame_widget("reve", NULL, &X->warr) ;
  add_delete_callback(X->warr[WI_MAIN_FRAME]) ;
  get_font() ;
  XtVaSetValues(X->warr[WI_MAIN_FRAME], XmNtitle, v->line, 0) ;
  make_icon() ;
  XtRealizeWidget(X->warr[WI_MAIN_FRAME]) ;
  v->pid = fork_child() ;

  X->white = get_color("#ffffff") ;
  X->black = get_color("#000000") ;
  X->brown = get_color("#c79665") ;
  make_pieces() ;
  initboard() ;
  set_clocks_visible(v->do_clock) ;
  paint_board() ;
  paint_panel() ;
  paint_prop_sheet() ;
  XtAddEventHandler(X->warr[WI_MAIN_FRAME], KeyPressMask | KeyReleaseMask,
                    FALSE, event_proc, NULL) ;
  XtAddGrab(X->warr[WI_MAIN_FRAME], 1, 1) ;
  XtMapWidget(X->warr[WI_MAIN_FRAME]) ;

#ifndef NO_NETWORK
  if (v->isremote)                 /* Do we need to get remote connection? */
    {
      get_addrs(myname, opponent) ;
      open_ctl() ;
      open_socket() ;
      start_msgs() ;
      if (is_local()) set_config(dtype) ;       /* Reconfigure. */
      else            invite_remote() ;
      end_msgs() ;
      message(M_PANEL, "Connection established") ;
      beep() ;
    }
#endif /*!NO_NETWORK*/
  connect_io() ;
  if (!v->isremote && v->dtype != XBOTH) load_and_move() ;
  reset_clock(BLACK) ;
  reset_clock(WHITE) ;
  v->started = 1 ;
}


void
draw_image(enum image_type itype, int n, int state)
{
  int col, row, x, y ;
  Pixmap image ;
  Widget button ;

       if (itype == I_MOVE)    image = X->move_bm ;
  else if (itype == I_SUGGEST) image = X->suggest_bm ;
  row = n / BOARD_SIZE ;
  col = n % BOARD_SIZE ;
  button = get_board_widget(row, col) ;
  x = (v->squarew - SWIDTH)  / 2 ;
  y = (v->squareh - SHEIGHT) / 2 ;

  X->gc_mask            = GCForeground | GCStipple |
                          GCTileStipXOrigin | GCTileStipYOrigin ;
  X->gc_val.foreground  = X->black ;
  X->gc_val.stipple     = image ;
  X->gc_val.ts_x_origin = x ;
  X->gc_val.ts_y_origin = y ;
  XChangeGC(X->dpy, X->ropgc, X->gc_mask, &X->gc_val) ;
  XFillRectangle(X->dpy, XtWindow(button), X->ropgc, x, y, SWIDTH, SHEIGHT) ;
  set_image(row, col) ;
}


void
draw_line(int move, int state, int x1, int y1, int x2, int y2)
{
  if (state) X->gc_val.foreground = X->black ;
  else       X->gc_val.foreground = X->brown ;
  X->gc_val.function = GXcopy ;
  XChangeGC(X->dpy, X->gc, GCForeground | GCFunction, &X->gc_val) ;
  XDrawLine(X->dpy, X->squares[move], X->gc, x1, y1, x2, y2) ;
}


void
draw_piece(int piece, int row, int col, int width, int height)
{
  int n, x, xrad, y, yrad ;
  Pixel color ;
  Widget widget ;

  widget = get_board_widget(row, col) ;
  n = row * BOARD_SIZE + col ;
  xrad = width / 2 ;
  yrad = height / 2 ;
  X->gc_val.foreground = X->brown ;
  X->gc_val.function   = GXcopy ;
  XChangeGC(X->dpy, X->gc, GCForeground | GCFunction, &X->gc_val) ;
  XFillRectangle(X->dpy, X->squares[n], X->gc, 0, 0,
                 v->squarew, v->squareh) ;
  if (piece != FREE)
    {
      color = (piece == BLACK) ? X->black : X->white ;
      x = (v->squarew - width) / 2 ;
      y = (v->squareh - height) / 2 ;
      X->gc_val.foreground = color ;
      X->gc_val.function   = GXcopy ;
      XChangeGC(X->dpy, X->gc, GCForeground | GCFunction, &X->gc_val) ;
      XFillArc(X->dpy, X->squares[n], X->gc, x, y,
               2 * xrad - 1, 2 * yrad - 1, 0, 360 * 64) ;
      if (piece == WHITE)
        {
          X->gc_val.foreground = X->black ;
          X->gc_val.function   = GXcopy ;
          XChangeGC(X->dpy, X->gc, GCForeground | GCFunction, &X->gc_val) ;
          XDrawArc(X->dpy, X->squares[n], X->gc, x, y,
                   2 * xrad - 1, 2 * yrad - 1, 0, 360 * 64) ;
        }
    }
  set_image(row, col) ;
}


static void
draw_text(int move, int x, int y, int state, char *str)
{
       if (v->board.square[move] == BLACK)
    X->gc_val.foreground = (state) ? X->white : X->black ;
  else if (v->board.square[move] == WHITE)
    X->gc_val.foreground = (state) ? X->black : X->white ;
  else
    X->gc_val.foreground = (state) ? X->black : X->brown ;
  X->gc_val.font     = X->font->fid ;
  X->gc_val.function = GXcopy ;
  XChangeGC(X->dpy, X->gc, GCFont | GCForeground | GCFunction, &X->gc_val) ;
  XDrawString(X->dpy, X->squares[move], X->gc, x, y, str, strlen(str)) ;
}


static void
event_proc(Widget widget, XtPointer client_data, XEvent *event,
           Boolean *continue_to_dispatch)
{
  char chs[2] ;
  int down, up ;
  KeySym ksym ;
  XKeyPressedEvent *key_event ;

  switch (event->type)
    {
      case KeyPress   :
      case KeyRelease : key_event = (XKeyPressedEvent *) event ;
                        XLookupString(key_event, chs, 1, &ksym,
                                      (XComposeStatus *) NULL) ;
                        if (ksym == XK_Shift_L || ksym == XK_Shift_R) return ;
                        v->cur_ch = chs[0] ;
                        if (event->type == KeyRelease) handle_key() ;
                        break ;
    }
}


static void
force_popup_on_screen(Widget popup, int *popup_x_p, int *popup_y_p)
{
  Dimension popup_width, popup_height ;
  Position left, top ;
  int popup_x, popup_y ;
  int n, screen_width, screen_height ;

  popup_x = *popup_x_p ;
  popup_y = *popup_y_p ;

/* Get the screen size. */

  get_screen_size(&screen_width, &screen_height) ;

  XtVaGetValues(popup,
                XmNwidth,  &popup_width,
                XmNheight, &popup_height,
                0) ;
 
/* Make sure frame does not go off side of screen. */
 
  n = popup_x + (int) popup_width ;
  if (n > screen_width) popup_x -= (n - screen_width) ;
  else if (popup_x < 0) popup_x = 0 ;
 
/* Make sure frame doen't go off top or bottom. */
 
  n = popup_y + (int) popup_height ;
  if (n > screen_height) popup_y -= n - screen_height ;
  else if (popup_y < 0) popup_y = 0 ;
 
/* Set location and return. */

  left = (Position) popup_x ;
  top  = (Position) popup_y ;
  XtVaSetValues(popup,
                XmNx, left,
                XmNy, top,
                0) ;

  *popup_x_p = popup_x ;
  *popup_y_p = popup_y ;
}


Widget
get_board_widget(int row, int col)
{
  char name[MAXLINE] ;
  int n ;

  n = row*BOARD_SIZE + col ;
  SPRINTF(name, "%c%c", 'a' + col, '1' + row) ;
  return(XtNameToWidget(X->warr[WI_BOARD_FORM], name)) ;
}


int
get_bool_value(Widget w)
{
  Boolean value ;

  XtVaGetValues(w, XmNset, &value, 0) ;
  return(value) ;
}


static Pixel
get_color(char *cstr)
{
  Colormap cmap = DefaultColormap(X->dpy, X->screen) ;
  XColor ccol ;
  Pixel pixel = 0 ;

  if (XParseColor(X->dpy, cmap, cstr, &ccol))
    if (XAllocColor(X->dpy, cmap, &ccol) == True) pixel = ccol.pixel ;
    else FPRINTF(stderr, "Unable to allocate %s color.\n") ;
  return(pixel) ;
}


int
get_excl_value(Widget w)
{
  Boolean value ;
  Cardinal count ;
  int i ;
  WidgetList kids ;

  XtVaGetValues(w,
                XmNnumChildren, &count,
                XmNchildren,    &kids,
                0) ;
  for (i = 0; i < count; i++)
    {
      XtVaGetValues(kids[i], XmNset, &value, 0) ;
      if (value) return(i) ;
    }
/*NOTREACHED*/
}


static void
get_font()
{
  XFontStruct *font ;
  XmFontContext context ;
  XmFontList fontlist ;
  XmStringCharSet charset ;

  XtVaGetValues(X->warr[WI_BLACK_LABEL], XmNfontList, &fontlist, 0) ;
  XmFontListInitFontContext(&context, fontlist) ;
  if (XmFontListGetNextFont(context, &charset, &font) == True)
    {
      X->font = font ;
      X->font_height = font->max_bounds.ascent + font->max_bounds.descent ;
    }
  else
    {
      FPRINTF(stderr, "Can't get font for labelling pieces.\n") ;
      exit(1) ;
    }
}


int
get_int_value(Widget w)
{
  return(atoi(XmTextGetString(w))) ;
}


char *
get_resource(enum res_type rtype)      /* Get Reve X resources. */
{
  char cstr[MAXLINE], nstr[MAXLINE], str[MAXLINE] ;
  char *str_type[20] ;
  XrmValue value ;

  STRCPY(str, v->resources[(int) rtype]) ;
  SPRINTF(nstr,  "reve.%s", str) ;

#ifdef VMS
  if (tolower(str[0])) str[0] = toupper(str[0]) ;
#else
  if (islower(str[0])) str[0] = toupper(str[0]) ;
#endif

  SPRINTF(cstr, "Reve.%s", str) ;
  if (XrmGetResource(X->reve_DB, nstr, cstr, str_type, &value) == 0)
    return((char *) NULL) ;
  else return(value.addr) ;
}


static void
get_screen_size(int *width_p, int *height_p)
{
  *width_p  = DisplayWidth(X->dpy,  X->screen) ;
  *height_p = DisplayHeight(X->dpy, X->screen) ;
}


static int
get_strwidth(char *str)        /* Get width in pixels of string value. */
{
  return(XTextWidth(X->font, str, strlen(str))) ;
}


int
is_window_showing(Widget widget)
{
  return(XtIsManaged(widget)) ;
}


void
load_help_file(char *helpfile)
{
  char *text ;
  FILE *fp ;
  struct stat statb ;

  if ((fp = find_file(helpfile)) == NULL)
    {
      FPRINTF(stderr, "Cannot open online help file\n") ;
      return ;
    }
  FSTAT(fileno(fp), &statb) ;
  if (!(text = XtMalloc((unsigned) (statb.st_size+1))))
    {
      FPRINTF(stderr, "Can't allocate space for %s\n", helpfile) ;
      FCLOSE(fp) ;
      return ;
    }
  if (!fread(text, sizeof(char), statb.st_size+1, fp))
    FPRINTF(stderr, "Warning: may not have read the entire help file.\n") ;
  text[statb.st_size] = 0 ;
  XmTextSetString(X->hwarr[WI_HELP_TEXT], text) ;

/* Save offsets to the various help categories. */

  v->help_offsets[(int) H_DESC] = find_offset(text, "\nDESCRIPTION") ;
  v->help_offsets[(int) H_OPT]  = find_offset(text, "\nOPTIONS") ;
  v->help_offsets[(int) H_RES]  = find_offset(text, "\nRESOURCES") ;
  v->help_offsets[(int) H_FORM] = find_offset(text, "\nREVE GAMES") ;
  v->help_offsets[(int) H_FILE] = find_offset(text, "\nFILES") ;

  XtFree(text) ;
  FCLOSE(fp) ;
}


/*  Get the resource databases. These are looked for in the following ways:
 *
 *  Classname file in the app-defaults directory. In this case, Classname
 *  is Reve.
 *
 *  Classname file in the directory specified by the XUSERFILESEARCHPATH
 *  or XAPPLRESDIR environment variable.
 *
 *  Property set using xrdb, accessible through the XResourceManagerString
 *  macro or, if that is empty, the ~/.Xdefaults file.
 *
 *  XENVIRONMENT environment variable or, if not set, .Xdefaults-hostname
 *  file.
 *
 *  REVEDEFAULTS environment variable or, if not set, the ~/.reverc file.
 */

void
load_resources()
{
  XrmDatabase db ;
  char *home, name[MAXPATHLEN], *ptr ;
  int len ;

  home = getenv("HOME") ;
  XrmInitialize() ;
#ifdef VMS
  STRCPY(name, "DECW$SYSTEM_DEFAULTS:Reve.dat") ;
/* Get applications defaults file, if any. */

  db = XrmGetFileDatabase(name) ;
  XrmMergeDatabases(db, &X->reve_DB) ;

/* Merge server defaults, created by xrdb. If nor defined, use 
   SYS$LOGIN:DECW$Xdefaults.DAT */

  if (XResourceManagerString(X->dpy) != NULL)
    db = XrmGetStringDatabase(XResourceManagerString(X->dpy)) ;
  else
    { 
      SPRINTF(name, "%sDECW$Xdefaults.DAT", home) ;
      db = XrmGetFileDatabase(name) ;
    }
  XrmMergeDatabases(db, &X->reve_DB) ;
/*  Finally merge in Reve defaults via REVEDEFAULTS or, if not defined, the
 *  ~/.reverc file.
 */

  if ((ptr = getenv("REVEDEFAULTS")) == NULL)
    {
      SPRINTF(name, "%s/.reverc", home) ;
      db = XrmGetFileDatabase(name) ;
    }
  else db = XrmGetFileDatabase(ptr) ;
  XrmMergeDatabases(db, &X->reve_DB) ;
}
#else
  STRCPY(name, "/usr/lib/X11/app-defaults/Reve") ;


/* Get applications defaults file, if any. */

  db = XrmGetFileDatabase(name) ;
  XrmMergeDatabases(db, &X->reve_DB) ;

/* Merge server defaults, created by xrdb. If nor defined, use ~/.Xdefaults. */

  if (XResourceManagerString(X->dpy) != NULL)
    db = XrmGetStringDatabase(XResourceManagerString(X->dpy)) ;
  else
    { 
      SPRINTF(name, "%s/.Xdefaults", home) ;
      db = XrmGetFileDatabase(name) ;
    }
  XrmMergeDatabases(db, &X->reve_DB) ;

/*  Open XENVIRONMENT file or, if not defined, the .Xdefaults, and merge
 *  into existing database.
 */

  if ((ptr = getenv("XENVIRONMENT")) == NULL)
    {
      SPRINTF(name, "%s/.Xdefaults-", home) ;
      len = strlen(name) ;
      GETHOSTNAME(name+len, MAXPATHLEN-len) ;
      db = XrmGetFileDatabase(name) ;
    }
  else db = XrmGetFileDatabase(ptr) ;
  XrmMergeDatabases(db, &X->reve_DB) ;

/*  Finally merge in Reve defaults via REVEDEFAULTS or, if not defined, the
 *  ~/.reverc file.
 */

  if ((ptr = getenv("REVEDEFAULTS")) == NULL)
    {
      SPRINTF(name, "%s/.reverc", home) ;
      db = XrmGetFileDatabase(name) ;
    }
  else db = XrmGetFileDatabase(ptr) ;
  XrmMergeDatabases(db, &X->reve_DB) ;
}
#endif

static void
make_icon()
{
  if (DisplayCells(X->dpy, X->screen) > 2)
    X->icon = create_mp_pixmap(concat_strs(reve_xpm_bits)) ;
  else
    X->icon = XCreatePixmapFromBitmapData(X->dpy,
                  RootWindow(X->dpy, X->screen), (char *) reve_xbm_bits,
                  reve_xbm_width, reve_xbm_height, 1, 0, 1) ;
  XtVaSetValues(X->warr[WI_MAIN_FRAME],
                XmNiconName,   "reve",
                XmNiconPixmap, X->icon,
                0) ;
}


static void
make_pieces()
{
  char name[MAXLINE] ;
  Dimension height, width ;
  int col, n, row ;
  Widget button ;

  XtVaGetValues(X->warr[WI_A1],
                XmNwidth,  &width,
                XmNheight, &height,
                0) ;
  v->squarew = (int) width  - (2 * IMAGE_DIFF) ;
  v->squareh = (int) height - (2 * IMAGE_DIFF) ;
  v->cell_height = v->squareh - (2 * PIECE_MARGIN) ;
  v->cell_width  = v->squarew - (2 * PIECE_MARGIN) ;
  for (row = 0; row < BOARD_SIZE; row++)
    for (col = 0; col < BOARD_SIZE; col++)
      {
        button = get_board_widget(row, col) ;
        XtAddCallback(button, XmNexposeCallback,
                      (XtCallbackProc) redraw_square, NULL) ;
        n = row*BOARD_SIZE + col ;
        if (X->squares[n]) XFreePixmap(X->dpy, X->squares[n]) ;
        X->squares[n] = XCreatePixmap(X->dpy, X->root,
                                      v->squarew, v->squareh, X->depth) ;
      }
}


void
message(enum mes_type mtype, char *str)
{
  Widget w ;

       if (mtype == M_PANEL)  w = X->warr[WI_MESSAGES] ;
  else if (mtype == M_EVAL)   w = X->warr[WI_EVAL_INFO] ;
  else if (mtype == M_BSCORE) w = X->warr[WI_BLACK_COUNT] ;
  else if (mtype == M_WSCORE) w = X->warr[WI_WHITE_COUNT] ;
  else if (mtype == M_TURN)   w = X->warr[WI_MOVE_INFO] ;
  else if (mtype == M_BCLOCK) w = X->warr[WI_BLACK_CLOCK] ;
  else if (mtype == M_WCLOCK) w = X->warr[WI_WHITE_CLOCK] ;
  else if (mtype == M_BPLAYS) w = X->warr[WI_BLACK_PLAYS] ;
  else if (mtype == M_WPLAYS) w = X->warr[WI_WHITE_PLAYS] ;
  XmTextSetString(w, str) ;
}


void
open_reve()
{
  XMapWindow(X->dpy, XtWindow(X->warr[WI_MAIN_FRAME])) ;
}


void
opt_set(enum opt_type otype, int n)
{
  Boolean value = n ;
  Widget w ;

       if (otype == O_BEST) w = X->warr[WI_CURRENT] ;
  else if (otype == O_LAST) w = X->warr[WI_LAST] ;
  else if (otype == O_EVAL) w = X->warr[WI_EVALUATION] ;
  else if (otype == O_NUM)  w = X->warr[WI_NUMBER] ;
  else if (otype == O_MOVE) w = X->warr[WI_INVALID] ;
  else if (otype == O_CLK)  w = X->warr[WI_CLOCKS] ;
  XtVaSetValues(w, XmNset, value, 0) ;
}


static void
position_help(enum help_type htype)
{
  XmTextSetTopCharacter(X->hwarr[WI_HELP_TEXT],
                        (XmTextPosition) v->help_offsets[(int) htype]) ;
}


static void
position_popup(Widget base, Widget popup)
{
  int bh, bw, bx, by, px, py ;
  int screen_width, screen_height ;
  Position base_x, base_y, popup_x, popup_y ;
  Dimension base_width, base_height, popup_width, popup_height ;

  XtVaGetValues(base,
                XmNx,      &base_x,
                XmNy,      &base_y,
                XmNwidth,  &base_width,
                XmNheight, &base_height,
                0) ;
  bx = (int) base_x ;
  by = (int) base_y ;
  bw = (int) base_width ;
  bh = (int) base_height ;

  XtVaGetValues(popup,
                XmNx,      &popup_x,
                XmNy,      &popup_y,
                XmNwidth,  &popup_width,
                XmNheight, &popup_height,
                0) ;
 
  px = (int) popup_x ;
  py = (int) popup_y ;
 
  get_screen_size(&screen_width, &screen_height) ;
  px = bx + bw + 5 ;
  py = by - TITLE_LINE_HEIGHT ;
  force_popup_on_screen(popup, &px, &py) ;
}


void
raise_reve()
{
  XRaiseWindow(X->dpy, XtWindow(X->warr[WI_MAIN_FRAME])) ;
}


static void
redraw_square(Widget widget, XtPointer client_data,
              XmDrawnButtonCallbackStruct *cbs)
{
  char col, name[MAXLINE], row ;
  int n ;

  STRCPY(name, XtName(widget)) ;
  SSCANF(name, "%c%c", &col, &row) ;
  n = (row - '1')*BOARD_SIZE + (col - 'a') ;
  XCopyArea(X->dpy, X->squares[n], XtWindow(widget), X->gc,
            0, 0, v->squarew, v->squareh, IMAGE_DIFF, IMAGE_DIFF) ;
  XFlush(X->dpy) ;
}


void
set_clocks_visible(int state)
{
  if (state)
    {
      XtMapWidget(X->warr[WI_BLACK_CLOCK_LABEL]) ;
      XtMapWidget(X->warr[WI_BLACK_CLOCK]) ;
      XtMapWidget(X->warr[WI_WHITE_CLOCK_LABEL]) ;
      XtMapWidget(X->warr[WI_WHITE_CLOCK]) ;
    }
  else
    {
      XtUnmapWidget(X->warr[WI_BLACK_CLOCK_LABEL]) ;
      XtUnmapWidget(X->warr[WI_BLACK_CLOCK]) ;
      XtUnmapWidget(X->warr[WI_WHITE_CLOCK_LABEL]) ;
      XtUnmapWidget(X->warr[WI_WHITE_CLOCK]) ;
    }
}


void
set_cursor(ctype)
enum cur_type ctype ;
{
  Cursor cursor ;

       if (ctype == C_NORMAL)    cursor = X->normal_cur ;
  else if (ctype == C_HOURGLASS) cursor = X->hour_cur ;
  XDefineCursor(X->dpy, XtWindow(X->warr[WI_MAIN_FRAME]), cursor) ;
}


void
set_excl_value(Widget w)
{
  Cardinal count ;
  int i ;
  Widget parent = XtParent(w) ;
  WidgetList kids ;

  XtVaGetValues(parent,
                XmNnumChildren, &count,
                XmNchildren,    &kids,
                0) ;
  for (i = 0; i < count; i++)                   /* Clear all values first. */
    XtVaSetValues(kids[i], XmNset, False, 0) ;
  XtVaSetValues(w, XmNset, True, 0) ;
}


void
set_image(int row, int col)
{
  char name[MAXLINE] ;
  int n ;
  Widget widget ;

  n = row*BOARD_SIZE + col ;
  widget = get_board_widget(row, col) ;
  XCopyArea(X->dpy, X->squares[n], XtWindow(widget), X->gc,
            0, 0, v->squarew, v->squareh, IMAGE_DIFF, IMAGE_DIFF) ;
  XFlush(X->dpy) ;
}


void
show_number(int move, int value, int state)
{
  char num[3] ;
  int strw, x, y ;

  if (move == -1) return ;
  SPRINTF(num, "%2d", value) ;
  strw = get_strwidth(num) ;
  x = (v->squarew / 2) - (strw / 2) ;
  y = (v->squareh / 2) + (X->font_height / 2) ;
  draw_text(move, x, y, state, num) ;
  set_image(move / BOARD_SIZE, move % BOARD_SIZE) ;
}


void
show_popup(Widget widget)
{
  if (!is_window_showing(widget))
    position_popup(X->warr[WI_MAIN_FRAME], widget) ;
  XtMapWidget(widget) ;
  XtManageChild(widget) ;
  XRaiseWindow(X->dpy, XtWindow(widget)) ;
}


static void
x_error_proc(display, error)
Display *display ;
XErrorEvent *error ;
{
  char msg[80] ;

  XGetErrorText(display, error->error_code, msg, 80) ;
  FPRINTF(stderr, "\nX Error (intercepted): %s\n", msg) ;
  FPRINTF(stderr, "Major Request Code   : %d\n", error->request_code) ;
  FPRINTF(stderr, "Minor Request Code   : %d\n", error->minor_code) ;
  FPRINTF(stderr, "Resource ID (XID)    : %u\n", error->resourceid) ;
  FPRINTF(stderr, "Error Serial Number  : %u\n", error->serial) ;
}
