/*LINTLIBRARY*/

/*  @(#)procs.c 1.3 94/01/27
 *
 *  Procedures associated with the reve game panel items.
 *
 *  Copyright (C) 1990 - 1994 - Rich Burridge & Yves Gallot.
 *  All rights reserved.
 *
 *  Permission is granted to copy this source, for redistribution
 *  in source form only, provided the news headers in "substantially
 *  unaltered format" are retained, the introductory messages are not
 *  removed, and no monies are exchanged.
 *
 *  Permission is also granted to copy this source, without the
 *  news headers, for the purposes of making an executable copy by
 *  means of compilation, provided that such copy will not be used
 *  for the purposes of competition in any othello tournaments, without
 *  prior permission from the authors.
 *
 *  No responsibility is taken for any errors on inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file), then an attempt will be made to fix them.
 */

#include "reve.h"

#ifndef VMS
#include <unistd.h>
#endif

#include <signal.h>

extern CVars c ;
extern Vars v ;                 /* Reve variables and options. */


void
do_edit()
{
  message(M_PANEL, "Not currently implemented.") ;
}


void
do_stop()
{
  message(M_PANEL, "Not currently implemented.") ;
}


void
do_suggestion(int player, int move, int note, int state)
{
  int count ;

  set_cursor(C_NORMAL) ;

  if (move == -1) return ;
  if (v->best_cmove != -1 && v->do_bestmove)
    {
      draw_square(v->best_cmove, 0, 2) ;
      if (v->do_number)
        show_number(v->best_cmove, v->cmove_depth, 0) ;
    }

  draw_suggest(move, state) ;
  if (v->show_moves && !state)
    draw_move(move, 1) ;
  if (v->show_moves || state)
    {
      count = legal(move, player, &v->board) ;
      show_number(move, count, 1) ;
    }
  if (v->show_notes && state)
    set_eval(player, move, note) ;
}


void
make_move()
{
  if (legal(v->move, v->next_player, &v->board) == 0)
    {
      v->invalid = 1 ;
      message(M_PANEL, "Invalid move") ;
      if (v->show_legal) show_all(1) ;
      return ;
    }
  do_move(v->next_player) ;
  v->invalid = 0 ;

  switch (v->dtype)
    {
      case XBLACK :
      case XWHITE : opponent_move(OPPONENT(v->next_player)) ;
                    break ;
      case XBOTH  : if (check(OPPONENT(v->next_player))) break ;
                    (void) check(v->next_player) ;
                    return ;
    }
  v->next_player = OPPONENT(v->next_player) ;
}


void
redo()
{
  int p ;

 /*  The amount we redo is dependant upon two things:
  *
  *  1/ Whether the opponent is the computer, in which case we redo
  *     two "moves". Otherwise we just redo one "move".
  *
  *  2/ A "move" is considered to be all consecutive moves by the opponent.
  */

  redo_move(v->next_player) ;
  p = (v->next_player == BLACK) ? v->black_plays : v->white_plays ;
  if (p == COMPUTER) redo_move(v->next_player) ;

  set_score() ;
  set_turn(v->next_player) ;
}


void
redo_move(int player)
{
  int i, m, n, x, y ;
 
  m = n = 64 - v->board.moves_left ;
  while (v->moves[n].move != -1 && v->moves[n].player == player) n++ ;
  n-- ;
 
  if (n < m || n > 63) message(M_PANEL, "No moves to redo.") ;
  else
    {
      if (v->do_last)
        show_last(v->last_move, 0) ;
      if (v->do_number)
        show_number(v->last_move, 60 - v->board.moves_left, 0) ;
      if (v->show_moves) show_all(0) ;
      do_suggestion(player, v->suggestion, c->note, 0) ;
      FOR_BOARD(i)
        {
          if (v->moves[n].square[i] != v->board.square[i])
            draw_piece(v->moves[n].square[i], i / BOARD_SIZE, i % BOARD_SIZE,
                       v->cell_width, v->cell_height) ;
          v->board.square[i] = v->moves[n].square[i] ;
        }
      v->board.moves_left = v->moves[n].moves_left ;
      v->board.player     = v->moves[n].player ;
      v->board.move       = v->moves[n].move ;
      v->board.note       = v->moves[n].note ;
      v->board.timeleft   = v->moves[n].timeleft ;
      v->last_move     = v->board.move ;

      FOR_BOARD(i) v->old_board.square[i] = v->moves[n-1].square[i] ;
      v->old_board.moves_left = v->moves[n-1].moves_left ;
      v->old_board.player     = v->moves[n-1].player ;
      v->old_board.move       = v->moves[n-1].move ;
      v->old_board.note       = v->moves[n-1].note ;
      v->old_board.timeleft   = v->moves[n-1].timeleft ;

      c->timeleft = v->board.timeleft ;
      reset_time(c->timeleft) ;
      message(M_PANEL, "") ;

      if (v->do_last)
        show_last(v->last_move, 1) ;
      if (v->do_number)
        show_number(v->last_move, 60 - v->board.moves_left, 1) ;
      if (v->show_notes)
        set_eval(v->board.player, v->board.move, v->board.note) ;
      else message(M_EVAL, "") ;
      v->next_player = OPPONENT(v->board.player) ;
    }
}


void
set_computer(int val)
{
  int curi ;

  if (val == CP_WHITE)
    {
      v->dtype = XBLACK ;
      v->play_computer = 1 ;
      v->black_plays = HUMAN ;
      read_str(&v->black_plays_text, v->player_values[HUMAN]) ;
      v->white_plays = COMPUTER ;
      read_str(&v->white_plays_text, v->player_values[COMPUTER]) ;
    }
  else if (val == CP_BLACK)
    {
      v->dtype = XWHITE ;
      v->play_computer = 1 ;
      v->black_plays = COMPUTER ;
      read_str(&v->black_plays_text, v->player_values[COMPUTER]) ;
      v->white_plays = HUMAN ;
      read_str(&v->white_plays_text, v->player_values[HUMAN]) ;
    }
  else if (val == CP_NEITHER)
    {
      v->dtype = XBOTH ;
      v->play_computer = 0 ;
      v->black_plays = HUMAN ;
      read_str(&v->black_plays_text, v->player_values[HUMAN]) ;
      v->white_plays = HUMAN ;
      read_str(&v->white_plays_text, v->player_values[HUMAN]) ; 
    }

  if (v->next_player == BLACK) curi = v->black_plays ;
  else                         curi = v->white_plays ;
  if (curi == COMPUTER)
    if (v->next_player == BLACK || v->next_player == WHITE)
      opponent_move(v->next_player) ;
}


void
show_all_moves()
{
  if (v->show_moves) show_all(0) ;
  else            show_all(1) ;
}


void
suggest()
{
  int player ;

  if (v->gameover) return ;
  player = OPPONENT(v->next_player) ;
  if (v->suggestion != -1)
    {
      v->sstate = !v->sstate ;
      do_suggestion(player, v->suggestion, v->snote, v->sstate) ;
    }
  else
    {
      set_cursor(C_HOURGLASS) ;
      v->restore_moves = v->show_moves ;
      if (v->show_moves) show_all(0) ;
      write_to_reve(M_SUGGESTION, v->board.square, player, v->level) ;
    }
}


void
undo()
{
  int p ;

 /*  The amount we undo is dependant upon two things:
  *
  *  1/ Whether the opponent is the computer, in which case we undo
  *     two "moves". Otherwise we just undo one "move".
  *
  *  2/ A "move" is considered to be all consecutive moves by the opponent.
  */

  undo_move(v->next_player) ;
  p = (v->next_player == BLACK) ? v->black_plays : v->white_plays ;
  if (p == COMPUTER) undo_move(v->next_player) ;

  set_score() ;
  set_turn(v->next_player) ;
}


void
undo_move(int player)
{
  int i, limit, n, x, y ;

  n = 63 - v->board.moves_left ;
  while (v->moves[n].player != player) n-- ;

  limit = (v->black_plays == COMPUTER) ? 4 : 3 ;
  if (n >= limit)
    {
      if (v->do_last)
        show_last(v->last_move, 0) ;
      if (v->do_number)
        show_number(v->last_move, 60 - v->board.moves_left, 0) ;
      if (v->show_moves) show_all(0) ;
      do_suggestion(player, v->suggestion, c->note, 0) ;
      FOR_BOARD(i)
        {
          if (v->moves[n].square[i] != v->board.square[i])
            draw_piece(v->moves[n].square[i], i / BOARD_SIZE, i % BOARD_SIZE,
                       v->cell_width, v->cell_height) ;
          v->board.square[i] = v->moves[n].square[i] ;
        }
      v->board.moves_left = v->moves[n].moves_left ;
      v->board.player     = v->moves[n].player ;
      v->board.move       = v->moves[n].move ;
      v->board.note       = v->moves[n].note ;
      v->board.timeleft   = v->moves[n].timeleft ;
      v->last_move     = v->board.move ;

      FOR_BOARD(i) v->old_board.square[i] = v->moves[n-1].square[i] ;
      v->old_board.moves_left = v->moves[n-1].moves_left ;
      v->old_board.player     = v->moves[n-1].player ;
      v->old_board.move       = v->moves[n-1].move ;
      v->old_board.note       = v->moves[n-1].note ;
      v->old_board.timeleft   = v->moves[n-1].timeleft ;

      c->timeleft = v->board.timeleft ;
      reset_time(c->timeleft) ;
      message(M_PANEL, "") ;

      if (n > 3)
        {
          if (v->do_last)
            show_last(v->last_move, 1) ;
          if (v->do_number)
            show_number(v->last_move, 60 - v->board.moves_left, 1) ;
          if (v->show_notes)
            set_eval(v->board.player, v->board.move, v->board.note) ;
        }
      else message(M_EVAL, "") ;
      v->next_player = OPPONENT(v->board.player) ;
    }
  else message(M_PANEL, "No moves to undo.") ;
}


/* Display time left in mm:ss format. */

void
display_time(int player, int timeleft)
{
  char text[MAXLINE] ;
  enum mes_type mtype ;
  int mins, secs ;

  mins = timeleft / 60 ;
  secs = timeleft % 60 ;
  mtype = (player == BLACK) ? M_BCLOCK : M_WCLOCK ;
  SPRINTF(text, "%1d:%02d", mins, secs) ;
  if (v->do_clock) message(mtype, text) ;
}


void
domove(BOARD *pos, int mv, BOARD *nextpos, int player)
{
  int i ;

  if (pos != nextpos) FOR_BOARD(i) nextpos->square[i] = pos->square[i] ;

  v->s_move = mv ;
  v->s_row = mv >> 3 ;
  v->s_col = mv & 7 ;
  v->s_player = player ;
  v->s_opponent = -player ;
  v->s_flip = 1 ;
  v->s_pos = nextpos ;

  nextpos->square[v->s_move] = player ;

  (void) sandwich(-9) ;
  (void) sandwich(-8) ;
  (void) sandwich(-7) ;
  (void) sandwich(-1) ;
  (void) sandwich(1) ;
  (void) sandwich(7) ;
  (void) sandwich(8) ;
  (void) sandwich(9) ;

  nextpos->moves_left = (pos->moves_left) - 1 ;
}


void
do_read_from_reve(struct reve_out out)
{
  c->profmax = out.depth ;
       if (out.type == M_BEST) show_best(out.move, out.note) ;
  else if (out.type == M_MOVE)
    {
      set_cursor(C_NORMAL) ;
      v->move = out.move ;
      c->note = out.note ;
      opponent_move(v->next_player) ;
    }
  else if (out.type == M_SUGGESTION)
    {
      if (legal(out.move, v->next_player, &v->board) == 0)
        FOR_BOARD(out.move)
          if (v->board.square[out.move] == FREE &&
              legal(out.move, v->next_player, &v->board)) break ;
      do_suggestion(v->next_player, out.move, out.note, 1) ;
      if (v->restore_moves) show_all(1) ;
      v->sstate     = 1 ;
      v->suggestion = out.move ;
      v->snote      = out.note ;
      v->processing = 0 ;
    }
}


int
fork_child()
{
#ifdef HASREVEPROC
  char *argv[6] ;         /* Reve_proc command line arguments. */
  int i, tabsiz ;

  PIPE(v->pipe_io[0]) ;      /* Setup input pipe. */
  PIPE(v->pipe_io[1]) ;      /* Setup output pipe. */
  switch (v->pid = fork())
    {
      case -1 : CLOSE(v->pipe_io[0][0]) ;
                CLOSE(v->pipe_io[0][1]) ;
                CLOSE(v->pipe_io[1][0]) ;
                CLOSE(v->pipe_io[1][1]) ;
                perror("reve fork failed") ;
                exit(1) ;
      case  0 : DUP2(v->pipe_io[0][0], 0) ;         /* Child. */
                DUP2(v->pipe_io[1][1], 1) ;
                DUP2(v->pipe_io[1][1], 2) ;

#ifndef HAS_GETDTAB
#ifdef hpux
                tabsiz = _NFILE ;
#else
                tabsiz = sysconf(_SC_OPEN_MAX) ;
#endif /*hpux*/
#else
                tabsiz = getdtablesize() ;
#endif /*!HAS_GETDTAB*/

                for (i = tabsiz; i > 2; i--) CLOSE(i) ;
                for (i = 0; i < NSIG; i++) SIGNAL(i, SIG_DFL) ;
                i = 0 ;
                argv[i++] = "v->reve_proc" ;
                argv[i++] = "-e" ;
                argv[i++] = c->edgefile ;
                if (c->saveres) argv[i++] = "-log" ;
                if (c->debug)   argv[i++] = "-debug" ;
                argv[i] = (char *) NULL ;
                execvp(v->reveproc, argv) ;
                perror("reve child exec") ;
                _exit(-1) ;
      default : CLOSE(v->pipe_io[0][0]) ;           /* Parent. */
                CLOSE(v->pipe_io[1][1]) ;
    }
#else
  init_edge_table(c->edgefile) ;
#endif /*HASREVEPROC*/
  return(v->pid) ;
}


void
init_clocks()               /* Setup timer clocks. */
{
  v->bclock_value = v->timevals[v->level] * 60 ;
  v->wclock_value = v->timevals[v->level] * 60 ;
}


int
legal(int mv, int player, BOARD *pos)
{
  if (pos->square[mv]) return(0) ;     /* Already occupied */

  v->s_move     = mv ;
  v->s_row      = mv >> 3 ;
  v->s_col      = mv & 7 ;
  v->s_player   = player ;
  v->s_opponent = -player ;
  v->s_flip     = 0 ;
  v->s_pos      = pos ;

  return(sandwich(-9) + sandwich(-8) + sandwich(-7) + sandwich(-1) +
         sandwich(1)  + sandwich(7)  + sandwich(8)  + sandwich(9)) ;
}


#ifdef HASREVEPROC
void
read_from_reve(int fd)
{
  struct reve_out out ;
  int reply, sout ;

  if (!v->started) return ;
  sout = sizeof(struct reve_out) ;
  if ((reply = read(fd, (char *) &out, sout)) > 0) do_read_from_reve(out) ;
  else
    {
      FPRINTF(stderr, "Couldn't get connection to reve_proc. Exiting.\n",
              v->progname) ;
      if (v->pid) KILL(v->pid, SIGKILL) ;
      exit(1) ;
    }
}
#endif /*HASREVEPROC*/


void
reset_clock(int player)
{
  time_t tval ;

  v->start_time = tval = time((time_t *) 0) ;
  if (player == BLACK) v->last_btime = tval ;
  else                 v->last_wtime = tval ;
}


void
reset_time(time_t timeleft)
{
  struct reve_in in ;

  in.type = M_TIME ;
  in.timeleft = timeleft ;
  WRITE(v->pipe_io[0][1], (char *) &in, sizeof(struct reve_in)) ;
}


/*  Test whether the square move sandwiches a line
 *  of enemy pieces in the direction [row_inc, col_inc];
 *  If (s_flip) then update position by capturing such pieces
 *  Returns the number of pieces captured.
 */

int
sandwich(int increment)
{
  int square, offset ;
  int row_offset, col_offset, piece, piece_count ;
  int pcount = 0 ;

  if ((v->s_move+increment) < 0 || (v->s_move+increment) > 63) return(pcount) ;
  if (v->s_pos->square[v->s_move+increment] != v->s_opponent)  return(pcount) ;
 
/*  Quick test to catch most failures -
 *  note that the tested square may not even
 *  be on the board, but the condition is a
 *  sufficient one for failure.
 */
 
  row_offset = (increment < -1 ? v->s_row  :      /* inc -1: -9, -8, -7 */
    increment >  1 ? 7-v->s_row : 8) ;            /* inc  1:  7,  8,  9 */
  col_offset = (increment & 4 ? v->s_col   :      /* inc -1: -9, -1,  7 */
    increment & 1 ? 7-v->s_col : 8) ;             /* inc  1: -7,  1,  9 */
 
  offset = (row_offset > col_offset ? col_offset : row_offset) ;

/* offset = shortest distance to an edge in the direction of search */

  if (2 > offset) return(pcount) ;
 
  piece_count = 1 ;
  square = v->s_move+increment ;
 
  while (--offset)
    {
      if (!(piece = v->s_pos->square[square += increment]))
        return(pcount) ;        /* If empty square, give up */
 
      if (piece == v->s_player) break ;
      else piece_count++ ;      /* Count opponent's pieces encountered */
    }

  if (!offset) return(pcount) ;

  pcount = piece_count ;
  if (v->s_flip)
    while (piece_count--)
      v->s_pos->square[square -= increment] = v->s_player ;

  return(pcount) ;
}


/* Decrement the clock for the current user. */

void
update_clock(int player, int doinc)
{
  char str[MAXLINE] ;
  int diff, *n, timeleft ;
  time_t cur_time, last_time ;

  diff = 0 ;
  cur_time  = time((time_t *) 0) ;
  last_time = (player == BLACK) ? v->last_btime : v->last_wtime ;
  if (cur_time != last_time) diff = (int) (cur_time - last_time) ;
  if (doinc && diff <= 0) diff = 1 ;
  else if (diff <= 0) return ;
  if (player == BLACK)
    {
      v->last_btime = cur_time ;
      n = &v->bclock_value ;
    }
  else
    {
      v->last_wtime = cur_time ;
      n = &v->wclock_value ;
    }
  *n -= diff ;
  if (*n < 0) *n = 0 ;
  display_time(player, *n) ;
  if (v->do_clock && !*n)
    {
      v->gameover = 1 ;
      SPRINTF(str, "Timer expired for %s. ***GAME OVER***",
              (player == BLACK) ? v->bstone_name : v->wstone_name) ;
      message(M_PANEL, str) ;
      beep() ;
    }
}


/* Check if valid move for this player. */

int
valid_move(BOARD *board, int player)
{
  int mv ;
  int valid ;      /* Set if there is a valid move. */

  for (mv = 0; mv < 64; mv++)
    if (board->square[mv] == FREE)
      {
        valid = legal(mv, player, board) ;
        if (valid > 0) return(1) ;
      }
  return(0) ;
}


/*ARGSUSED*/
void
write_to_reve(enum move_type mtype, int *reve_board, int player, int level)
{
  int i ;
  struct reve_in in ;
  struct reve_out out ;
  time_t savetime ;          /* Restore correct time after a suggestion. */

  for (i = 0; i < 64; i++) in.board[i] = reve_board[i] ;
  in.player      = player ;
  in.level       = level ;
  in.type        = mtype ;
  v->processing  = 1 ; 
#ifdef HASREVEPROC
  WRITE(v->pipe_io[0][1], (char *) &in, sizeof(struct reve_in)) ; 
#else
  if (v->opp_iconise) close_reve() ;
       if (in.type == M_TIME)    c->timeleft  = in.timeleft ;
  else if (in.type == M_PROFMAX) c->max_depth = in.level ;
  else
    { 
      if (in.type == M_SUGGESTION) savetime = c->timeleft ;
      play_reve(in.board, in.player, in.level, &out.move, (long *) &out.note) ;
      out.type = in.type ;

      v->next_player = OPPONENT(v->next_player) ;
      do_read_from_reve(out) ;
      v->next_player = OPPONENT(v->next_player) ;

      if (in.type == M_SUGGESTION) c->timeleft = savetime ;
    }
#endif /*HASREVEPROC*/
}
