
/*  @(#)get.c 1.3 94/01/27
 *
 *  Various extraction routines used by reve.
 *
 *  Copyright (C) 1990 - 1994 - Rich Burridge & Yves Gallot.
 *  All rights reserved.
 *
 *  Permission is granted to copy this source, for redistribution
 *  in source form only, provided the news headers in "substantially
 *  unaltered format" are retained, the introductory messages are not
 *  removed, and no monies are exchanged.
 *
 *  Permission is also granted to copy this source, without the
 *  news headers, for the purposes of making an executable copy by
 *  means of compilation, provided that such copy will not be used
 *  for the purposes of competition in any othello tournaments, without
 *  prior permission from the authors.
 *
 *  No responsibility is taken for any errors on inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file), then an attempt will be made to fix them.
 */

#include "reve.h"
#include "patchlevel.h"
#include <ctype.h>

#ifdef X11
#include <X11/Xos.h>
#endif /*X11*/

#ifdef NO_TIMEVAL
struct timeval {
  long tv_sec ;         /* Seconds */
  long tv_usec ;        /* Microseconds */
} ;
#endif /*NO_TIMEVAL*/

#ifdef VMS
  float seconds;
#endif

extern CVars c ;
extern Vars v ;                 /* Reve variables and options. */


/*  Does a string compare between a1 and a2.
 *  To return '0', a1 and a2 must match to the length of a2, and that
 *  length has to be at least 'minlen'. Otherwise, return non-zero.
 */

int
argcmp(char *a1, char *a2, int minlen)
{
  int len_a1 = strlen(a1) ;
  int len_a2 = strlen(a2) ;

  if (len_a1 < minlen || len_a2 < minlen) return(1) ;
  if (len_a1 > len_a2) return(1) ;
  return(strncmp(a1, a2, len_a1)) ;
}


char *
concat_strs(char **text)
{
  char buf[BUFFERSIZE], *p ;
  int i ;
 
  p = buf ;
  for (i = 0; text[i]; i++)
    {
      p += strlen(strcpy(p, text[i])) ;
      *p++ = '\n' ;
      *p = 0 ;
    }
  return(buf) ;
}


/* Get boolean resource from the server. */

int
get_bool_resource(enum res_type rtype, int *boolval)
{
  char *val, tempstr[MAXLINE] ;
  int len, n ;

  if ((val = get_resource(rtype)) == NULL) return(0) ;
  STRCPY(tempstr, val) ;
  len = strlen(tempstr) ;
  for (n = 0; n < len; n++)
    if (isupper(tempstr[n])) tempstr[n] = tolower(tempstr[n]) ;
  if (EQUAL(tempstr, "true")) *boolval = 1 ;
  else                        *boolval = 0 ;
  return(1) ;
}


/* Get integer resource from the server. */

int
get_int_resource(enum res_type rtype, int *intval)
{
  char *val ;
 
  if ((val = get_resource(rtype)) == NULL) return(0) ;
  *intval = atoi(val) ;
  return(1) ;
}
 
 
/* Read and process command line options. */

void
get_options(int argc, char **argv)
{
  int i ;

  for (i = 1; i < argc; i++)
    {
      if (ARG_EQUAL("-bestmove", 3))
        v->do_bestmove = 1 ;
      else if (ARG_EQUAL("-black", 3))
        v->isblack = 1 ;
      else if (ARG_EQUAL("-clock", 2))
        v->do_clock = 1 ;
      else if (ARG_EQUAL("-debug", 3))
        c->debug = 1 ;
      else if (ARG_EQUAL("-difficulty", 3))
        { 
          if (++i < argc)
            {
              v->level = atoi(argv[i]) ;
              if (v->level < 1 || v->level > MAXDIFF)
              v->level = INIT_DEPTH ;
              v->old_diffval = v->level-1 ;
            }
          else FPRINTF(stderr, "-difficulty needs difficulty value\n") ;
        }
      else if (ARG_EQUAL("-edgefile", 2))
        {
          if (++i < argc) STRCPY(c->edgefile, argv[i]) ;
          else FPRINTF(stderr, "-edgefile needs an edgetable filename\n") ;
        }
      else if (ARG_EQUAL("-help", 6))
        usage() ;
      else if (ARG_EQUAL("-helpfile", 2))
        {
          if (++i < argc) STRCPY(v->helpfile, argv[i]) ;
          else FPRINTF(stderr, "-helpfile needs a help filename") ;
        }
      else if (ARG_EQUAL("-inverse", 2))    /* Display in inverse video. */
        v->inv_video = 1 ;
      else if (ARG_EQUAL("-last", 3))
        v->do_last = 1 ;
      else if (ARG_EQUAL("-load", 4))
        {
          if (++i < argc)
            {
              STRCPY(v->gamefile, argv[i]) ;
              v->loadgame = 1 ;        /* Game file to load. */
            }
          else FPRINTF(stderr, "-load needs a game filename to load\n") ;
        }
      else if (ARG_EQUAL("-log", 4))
        c->saveres = 1 ;
      else if (ARG_EQUAL("-notes", 3))
        v->show_notes = 1 ;
      else if (ARG_EQUAL("-number", 3))
        v->do_number = 1 ;
      else if (ARG_EQUAL("-opponent", 2))
        {
          if (++i < argc && argv[i] && argv[i][0] != '-')
            {
#ifndef NO_NETWORK
              opponent = read_str(&argv[i]) ;
              v->play_computer = 0 ;
#else
              FPRINTF(stderr, "%s: -opponent option disabled.\n", v->progname) ;
#endif /*!NO_NETWORK*/
            }
        }
      else if (ARG_EQUAL("-props", 2))
        v->props_showing = 1 ;
      else if (ARG_EQUAL("-reve_proc", 2))
        {
          if (++i < argc) STRCPY(v->reveproc, argv[i]) ;
          else FPRINTF(stderr, "-reve_proc needs a reve_proc pathname\n") ;
        }
      else if (ARG_EQUAL("-version", 2))
        {
          FPRINTF(stderr, "%s version 2.0.%1d\n\n", v->progname, PATCHLEVEL) ;
          exit(1) ;
        }
      else if (ARG_EQUAL("-white", 2))
        v->iswhite = 1 ;
    }
}


/* Get a string resource from the server. */

int
get_str_resource(enum res_type rtype, char *strval)
{
  char *val ;
   
  if ((val = get_resource(rtype)) == NULL) return(0) ;
  STRCPY(strval, val) ;
  return(1) ;
}


char *
getuserhost()
{
  char namehost[MAXLINE], *name, host[MAXLINE] ;
  int len ;

#ifdef VMS
  SPRINTF(namehost,"%s@%s", getenv("USER"), getenv("SYS$NODE"));
#else
  name = getlogin() ;
  if (name == NULL) name = "unknown" ;
  len = MAXLINE ;
  gethostname(host, len) ;
  SPRINTF(namehost, "%s@%s", name, host) ;
#endif
  return(namehost) ;
}


int
find_offset(char *text, char *str)
{
  char *p ;
  int len, pos ;

  pos = 1 ;
  len = strlen(str) ;
  for (p = text; p = strchr(p, *str); p++)
    if (!strncmp(p, str, len))
      {
        pos = (p - text) ;
        break ;
      }
  return(pos) ;
}


void
load_and_move()
{
  if (v->loadgame)
    {
      v->loadgame = 0 ;
      load_game() ;
    }

  if (v->first_move)
    {
      v->first_move = 0 ;
      v->next_player = BLACK ;
      opponent_move(v->next_player) ;
    }
}


void
paint_all()
{
  paint_board() ;
  paint_panel() ;
  load_and_move() ;
}


void
paint_board()
{
  int i, j, x, y ;

  for (i = 0; i < BOARD_SIZE; i++)
    for (j = 0; j < BOARD_SIZE; j++)
      draw_piece(v->board.square[i*BOARD_SIZE+j], i, j,
                 v->cell_width, v->cell_height) ;

  if (v->do_last)
    show_last(v->last_move, 1) ;
  if (v->do_number)
    show_number(v->last_move, 60 - v->board.moves_left, 1) ;

  if (v->show_moves || (v->invalid && v->show_legal))
    show_all(1) ;
  if (v->suggestion != -1)
    draw_suggest(v->suggestion, 1) ;
}


void
paint_panel()
{
  char mes[MAXLINE] ;

  message(M_BPLAYS, v->player_values[0]) ;
  message(M_WPLAYS, v->player_values[1]) ;
  set_score() ;
  SPRINTF(mes, "%s to move.",
          (v->next_player == BLACK) ? v->bstone_name : v->wstone_name) ;
  message(M_TURN, mes) ;
}


void
paint_prop_sheet()
{
  opt_set(O_BEST, v->do_bestmove) ;
  opt_set(O_LAST, v->do_last) ;
  opt_set(O_EVAL, v->show_notes) ;
  opt_set(O_NUM,  v->do_number) ;
  opt_set(O_MOVE, v->show_moves) ;
  opt_set(O_CLK,  v->do_clock) ;
}


void
do_key_move(int n1, int n2)
{
  v->move = (n2 - '1') * BOARD_SIZE + (n1 - 'a') ;
  update_clock(v->next_player, 1) ;
  make_move() ;
  reset_clock(v->next_player) ;
  v->validkey = 0 ;
}


void
handle_key()     /* Process the latest key that the user has pressed. */
{
  char str[9] ;  /* To display half move position. */
  int nextc ;

  if (v->cur_ch == ESCAPE) v->validkey = 0 ;
  if (v->validkey)
    {
      nextc = v->cur_ch ;
      v->cur_ch = v->validkey ;
    }
  switch (v->cur_ch)
    {
      case '1'      :
      case '2'      :
      case '3'      :
      case '4'      :
      case '5'      :
      case '6'      :
      case '7'      :
      case '8'      : if (!v->validkey)
                        {
                          v->validkey = v->cur_ch ;
                          SPRINTF(str, "Move: %c", v->cur_ch) ;
                          message(M_PANEL, str) ;
                        }
                      else if (nextc >= 'a' && nextc <= 'h')
                        {
                          SPRINTF(str, "Move: %c%c", v->cur_ch, nextc) ;
                          message(M_PANEL, str) ;
                          do_key_move(nextc, v->cur_ch) ;
                        }
                      else v->validkey = 0 ;
                      break ;
      case 'a'      :
      case 'b'      :
      case 'c'      :
      case 'd'      :
      case 'e'      :
      case 'f'      :
      case 'g'      :
      case 'h'      : if (!v->validkey)
                        {
                          v->validkey = v->cur_ch ;
                          SPRINTF(str, "Move: %c", v->cur_ch) ;
                          message(M_PANEL, str) ;
                        }
                      else if (nextc >= '1' && nextc <= '8')
                        {
                          SPRINTF(str, "Move: %c%c", v->cur_ch, nextc) ;
                          message(M_PANEL, str) ;
                          do_key_move(v->cur_ch, nextc) ;
                        }
                      else v->validkey = 0 ;
                      break ;
      default       : message(M_PANEL, "") ;
                      v->validkey = 0 ;
    }
}


void
init_data(char *name)
{
  int i ;
  int t[MAXDIFF] = { 0, 1, 3, 5, 10, 15, 20, 30, 60 } ;

  v->suggestion = -1 ;             /* Positive if a suggested move. */

  i = 0 ;
  for (i = 0; i < MAXDIFF; i++) v->timevals[i] = t[i] ;

  read_str(&v->player_values[0], "human") ;
  read_str(&v->player_values[1], "computer") ;

  read_str(&v->comp_plays[2], "neither") ;
  read_str(&v->comp_plays[3], "both") ;

  i = 0 ;
  read_str(&v->resources[i++], "bestmove") ;
  read_str(&v->resources[i++], "showClocks") ;
  read_str(&v->resources[i++], "difficulty") ;
  read_str(&v->resources[i++], "last") ;
  read_str(&v->resources[i++], "showLegalMoves") ;
  read_str(&v->resources[i++], "log") ;
  read_str(&v->resources[i++], "notes") ;
  read_str(&v->resources[i++], "number") ;
  read_str(&v->resources[i++], "quick") ;
  read_str(&v->resources[i++], "blackStoneName") ;
  read_str(&v->resources[i++], "whiteStoneName") ;
  read_str(&v->resources[i++], "printCommand") ;
  read_str(&v->resources[i++], "iconiseForOpponentMove") ;
  read_str(&v->resources[i++], "bellAfterOpponentMove") ;
  read_str(&v->resources[i++], "raiseAfterOpponentMove") ;

  read_str(&v->progname, name) ;       /* Save program name for later use. */
  STRCPY(v->gamefile, "reve.game") ;
  SPRINTF(v->line, "Reve  -  Version 2.0.%1d", PATCHLEVEL) ;
}


void
do_opponent_move(int player)
{
  set_cursor(C_NORMAL) ;
  if (v->best_cmove != -1 && v->do_bestmove)
    draw_square(v->best_cmove, 0, 2) ;
  do_move(player) ;
  v->last_move = v->move ;
}


void
initialise()        /* Initialise various variable used by reve. */
{
  int i ;

  v->play_computer = 1 ;                 /* Default is human vs computer. */
  v->iconic        = 0 ;                 /* Initially an open window. */
  v->invalid       = 0 ;
  v->last_move     = -1 ;
  v->last_outline  = -1 ;
  v->next_player   = BLACK ;
  v->best_cmove    = -1 ;
  c->debug         = 0 ;
  v->gameover      = 0 ;    /* Set true when the game is over. */
  v->opp_bell      = 0 ;    /* Don't sound bell after opponents move. */
  v->opp_iconise   = 0 ;    /* Don't close reve window for opponents move. */
  v->opp_raise     = 0 ;    /* Don't auto raise window after opponent move. */
  v->processing    = 0 ;    /* No computer move initially. */
  v->props_showing = 0 ;    /* Properties window not visible. */
  v->started       = 0 ;
  v->validkey      = 0 ;
  v->show_moves    = 0 ;
  v->sstate        = 0 ;
  v->isblack       = 0 ;
  v->isremote      = 0 ;
  v->iswhite       = 0 ;
  v->level         = INIT_DEPTH ;

  read_str(&v->myname, getuserhost()) ;      /* user@host for this person. */
  v->opponent  = NULL ;                      /* No networked game by default. */

  v->pipe_io[0][0] = v->pipe_io[0][1] = v->pipe_io[1][0] = v->pipe_io[1][1] = -1 ;
  v->socketfd      = -1 ;

  read_str(&v->bstone_name,   "Black") ;   /* Default names for messages. */
  read_str(&v->wstone_name,   "White") ;
  read_str(&v->comp_plays[0], "white") ;   /* Computer plays: property item. */
  read_str(&v->comp_plays[1], "black") ;
  read_str(&v->printcommand,  "trans | lpr") ;   /* Default print command. */

  STRCPY(c->edgefile, EDGENAME) ;
  STRCPY(v->helpfile, HELPNAME) ;
  STRCPY(v->reveproc, REVEPROC) ;
}


void
read_resources()            /* Read all possible resources from database. */
{
  int boolval, intval, len, n ;
  char str[MAXLINE] ;

  if (get_bool_resource(R_BESTMOVE, &boolval)) v->do_bestmove = boolval ;

  if (get_int_resource(R_DIFFICULTY, &intval))
    {
      v->level = intval ;
      if (v->level < 1 || v->level > MAXDIFF) v->level = INIT_DEPTH ;
    }

  if (get_bool_resource(R_CLOCK,     &boolval)) v->do_clock    = boolval ;
  if (get_bool_resource(R_LAST,      &boolval)) v->do_last     = boolval ;
  if (get_bool_resource(R_LOG,       &boolval)) c->saveres     = boolval ;
  if (get_bool_resource(R_LEGALMOVE, &boolval)) v->show_legal  = boolval ;
  if (get_bool_resource(R_NOTES,     &boolval)) v->show_notes  = boolval ;
  if (get_bool_resource(R_NUMBER,    &boolval)) v->do_number   = boolval ;
  if (get_bool_resource(R_OPPICON,   &boolval)) v->opp_iconise = boolval ;
  if (get_bool_resource(R_OPPBELL,   &boolval)) v->opp_bell    = boolval ;
  if (get_bool_resource(R_OPPRAISE,  &boolval)) v->opp_raise   = boolval ;

  if (get_str_resource(R_PCMD,    str)) read_str(&v->printcommand, str) ;

  if (get_str_resource(R_BSTONEN, str))
    {
      read_str(&v->bstone_name, str) ;
      len = strlen(str) ;
      for (n = 0; n < len; n++)
        if (isupper(str[n])) str[n] = tolower(str[n]) ; 
      read_str(&v->comp_plays[1], str) ;
    }
  if (get_str_resource(R_WSTONEN, str))
    {
      read_str(&v->wstone_name, str) ;
      len = strlen(str) ;
      for (n = 0; n < len; n++)
        if (isupper(str[n])) str[n] = tolower(str[n]) ; 
      read_str(&v->comp_plays[0], str) ;
    }
}


void
read_str(char **str, char *value)
{
  if (*str != NULL) (void) free(*str) ;
  if (value != NULL && strlen(value))
    {
      *str = (char *) malloc((unsigned) (strlen(value) + 1)) ;
      STRCPY(*str, value) ;
    }
  else *str = NULL ;
}


void
set_config(enum disp_type dtype)
{
  int val ;

  if (v->isremote)
    if (dtype == XWHITE)
      {
        read_str(&v->black_plays_text, v->opponent) ;
        read_str(&v->white_plays_text, v->myname) ;
        v->processing = 1 ;
        set_cursor(C_HOURGLASS) ;
      }
    else
      { 
        read_str(&v->black_plays_text, v->myname) ;
        read_str(&v->white_plays_text, v->opponent) ;
      }
     
  v->isblack = v->iswhite = 0 ;
       if (dtype == XBLACK) v->isblack = 1 ;
  else if (dtype == XWHITE) v->iswhite = v->first_move = 1 ;
  else if (dtype == XBOTH)  v->isblack = v->iswhite    = 1 ;
     
  val = (v->isblack == 0) ;
  v->bp_value = val ;
  if (!v->isremote) read_str(&v->black_plays_text, v->player_values[val]) ;

  val = (v->iswhite == 0) ;
  v->wp_value = val ;
  if (!v->isremote) read_str(&v->white_plays_text, v->player_values[val]) ;

       if (v->isremote)     v->cp_value = CP_NEITHER ;
  else if (dtype == XBLACK) v->cp_value = CP_WHITE ;
  else if (dtype == XWHITE) v->cp_value = CP_BLACK ;
  else if (dtype == XBOTH)  v->cp_value = CP_NEITHER ;
}


/*  Work out, who plays what. There are a fair number of combinations.
 *  This table hopefully gives the setup information for each combination.
 *
 *  NOTE: This could easily be done with less tests; instead the routine is
 *        written for clarity.
 *
 *  | BLACK  WHITE   OPPONENT
 *  |---------------------------
 *  |A  -      -        -        black on local, white is computer.
 *  |B  -      -     user@host   black on local, white on remote.
 *  |C  -      -     computer    black on local, white is computer.
 *  |D  -     SET       -        white on local, black is computer.
 *  |E  -     SET    user@host   black on remote, white on local.
 *  |F  -     SET    computer    black is computer, white is local.
 *  |G SET     -        -        black on local, white is computer.
 *  |H SET     -     user@host   black on local, white on remote.
 *  |I SET     -     computer    black on local, white is computer.
 *  |J SET    SET       -        both on local display.
 *  |K SET    SET    user@host   black on local, white on remote.
 *  |L SET    SET    computer    black on local, white is computer.
 */

void
set_display_types()
{
       if (!v->isblack && !v->iswhite && !v->opponent)                 /* A */
    {
      v->dtype    = XBLACK ;
      v->isremote = 0 ;
    }
  else if (!v->isblack && !v->iswhite &&  v->opponent &&
           !v->play_computer)                                          /* B */
    {
      v->dtype    = XBLACK ;
      v->isremote = 1 ;
    }
  else if (!v->isblack && !v->iswhite &&  v->opponent &&
            v->play_computer)                                          /* C */
    {
      v->dtype    = XBLACK ;
      v->isremote = 0 ;
    }
  else if (!v->isblack &&  v->iswhite && !v->opponent)                 /* D */
    {
      v->dtype    = XWHITE ;
      v->isremote = 0 ;
    }
  else if (!v->isblack &&  v->iswhite &&  v->opponent &&
           !v->play_computer)                                          /* E */
    {
      v->dtype    = XWHITE ;
      v->isremote = 1 ;
    }
  else if (!v->isblack &&  v->iswhite &&  v->opponent &&
            v->play_computer)                                          /* F */
    {
      v->dtype    = XWHITE ;
      v->isremote = 0 ;
    }
  else if ( v->isblack && !v->iswhite && !v->opponent)                 /* G */
    {
      v->dtype    = XBLACK ;
      v->isremote = 0 ;
    }
  else if ( v->isblack && !v->iswhite &&  v->opponent &&
           !v->play_computer)                                          /* H */
    {
      v->dtype    = XBLACK ;
      v->isremote = 1 ;
    }
  else if ( v->isblack && !v->iswhite &&  v->opponent &&
            v->play_computer)                                          /* I */
    {
      v->dtype    = XBLACK ;
      v->isremote = 0 ;
    }
  else if ( v->isblack &&  v->iswhite && !v->opponent)                 /* J */
    {
      v->dtype    = XBOTH ;
      v->isremote = 0 ;
      v->play_computer = 1 ;
    }
  else if ( v->isblack &&  v->iswhite &&  v->opponent &&
           !v->play_computer)                                          /* K */
    {
      v->dtype    = XBLACK ;
      v->isremote = 1 ;
    }
  else if ( v->isblack &&  v->iswhite &&  v->opponent &&
            v->play_computer)                                          /* L */
    {
      v->dtype    = XBLACK ;
      v->isremote = 0 ;
    }
  set_config(v->dtype) ;
}


void
usage()
{
  FPRINTF(stderr, "Usage:\n") ;
  usageoption(v->progname) ;
  usageoption("[-bestmove]") ;
  usageoption("[-black]") ;
  usageoption("[-clock]") ;
  usageoption("[-debug]") ;
  usageoption("[-dificulty value]") ;
  usageoption("[-edgefile filename]") ;
  usageoption("[-help]") ;
  usageoption("[-helpfile filename]") ;
  usageoption("[-inverse]") ;
  usageoption("[-last]") ;
  usageoption("[-load gamefile]") ;
  usageoption("[-log]") ;
  usageoption("[-notes]") ;
  usageoption("[-number]") ;
  usageoption("[-opponent]") ;
  usageoption("[-reve_proc pathname]") ;
  usageoption("[-version]") ;
  usageoption("[-white]") ;
}


void
usageoption(char *st)         /* Output usage option. */
{
  int len = strlen(st) ;

  if (len + v->cpos > 78)
    {
      FPRINTF(stderr, "\n   ") ;
      v->cpos = 3 ;
    }
   
  FPRINTF(stderr, "%s ", st) ;
  v->cpos += strlen(st) + 1 ;
}


reve_sleep(unsigned x)   /* Suspend execution for interval in microseconds. */
{
  struct timeval st_delay ;

  st_delay.tv_usec = x ;
  st_delay.tv_sec = 0 ;

#ifdef VMS
  seconds = st_delay.tv_usec * 1000000;
  (void) LIB$WAIT(&seconds) ;
#else
  SELECT(32, NULL, NULL, NULL, &st_delay) ;
#endif
}	
