/*LINTLIBRARY*/

/*  @(#)boardstuff.c 1.3 94/01/27
 *
 *  Various board routines used by reve.
 *
 *  Copyright (C) 1990 - 1994 - Rich Burridge & Yves Gallot.
 *  All rights reserved.
 *
 *  Permission is granted to copy this source, for redistribution
 *  in source form only, provided the news headers in "substantially
 *  unaltered format" are retained, the introductory messages are not
 *  removed, and no monies are exchanged.
 *
 *  Permission is also granted to copy this source, without the
 *  news headers, for the purposes of making an executable copy by
 *  means of compilation, provided that such copy will not be used
 *  for the purposes of competition in any othello tournaments, without
 *  prior permission from the authors.
 *
 *  No responsibility is taken for any errors on inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file), then an attempt will be made to fix them.
 */

#include "reve.h"
#include <ctype.h>

#ifdef X11
#include <X11/Xos.h>
#endif /*X11*/

extern CVars c ;
extern Vars v ;                 /* Reve variables and options. */


/*  This routine checks to see if a move can be made for this player.
 *  If not, various checks are made to see if the game is finished.
 *  Return value indicates if a move can be made.
 */

int
check(int player)
{
  if ((!count(&v->board, BLACK)) || (!count(&v->board, WHITE)) ||
      ((count(&v->board, BLACK) + count(&v->board, WHITE)) == 64))
    {
      who_wins() ;
      v->gameover = 1 ;
      message(M_PANEL, "***GAME OVER***") ;
      return(0) ;
    }
  if (!(v->move = valid_move(&v->board, player)))
    {
      SPRINTF(v->line, "%s is forced to pass",
                    (player == BLACK) ? v->bstone_name : v->wstone_name) ;
      message(M_PANEL, v->line) ;
      set_turn(OPPONENT(player)) ;
      if (!(v->move = valid_move(&v->board, OPPONENT(player))))
        {
          who_wins() ;
          v->gameover ;
          message(M_PANEL, "***GAME OVER***") ;
        }
      return(0) ;
    }
  return(1) ;
}


/* Count the number of player pieces on the board. */

int
count(BOARD *board, int player)
{
  int i, n ;

  n = 0 ;
  FOR_BOARD(i)
    if (board->square[i] == player) n++ ;
  return(n) ;
}


void
do_move(int player)
{
  int taken ;                /* Number of pieces flipped this go. */

  taken = formfliplist(v->move, player) ;
  update_board_image(player, taken) ;
}


void
draw_move(int n, int state)
{
  draw_image(I_MOVE, n, state) ;
}


void
draw_outline(int move, int state)
{
  if (v->last_outline == -1) return ;
  draw_square(move, state, 1) ;
  set_image(move / BOARD_SIZE, move % BOARD_SIZE) ;
}


void
draw_rect(int move, int state, int x1, int y1, int x2, int y2)
{
  draw_line(move, state, x1, y1, x2, y1) ;
  draw_line(move, state, x1, y1, x1, y2) ;
  draw_line(move, state, x2, y1, x2, y2) ;
  draw_line(move, state, x1, y2, x2, y2) ;
}


void
draw_square(int move, int state, int offset)
{
  draw_rect(move, state, offset, offset,
            v->squarew - offset, v->squareh - offset) ;
  set_image(move / BOARD_SIZE, move % BOARD_SIZE) ;
}


void
draw_suggest(int n, int state)
{
  draw_image(I_SUGGEST, n, state) ;
}


int
formfliplist(int move, int player)
{        
  int cnt, i, n, old_cnt ;

  old_cnt = count(&v->board, player) ;
  FOR_BOARD(i) v->old_board.square[i] = v->board.square[i] ;
  v->old_board.moves_left = v->board.moves_left ;
  domove(&v->old_board, move, &v->board, player) ;

  n = 63 - v->board.moves_left ;
  FOR_BOARD(i) v->moves[n].square[i] = v->board.square[i] ;
  v->moves[n].moves_left = v->board.moves_left ;
  v->moves[n].move = move ;
  v->moves[n].note = c->note ;
  v->moves[n].player = player ;
  v->moves[n].timeleft = c->timeleft ;

  cnt = count(&v->board, player) ;
  return(cnt - old_cnt - 1) ;
}


void
initboard()    /* Initialise the othello board. */
{
  static int ivals[4]   = { 27,    28,    35,    36 } ;
  static int icolors[4] = { WHITE, BLACK, BLACK, WHITE } ;
  int i, j , n ;

  v->suggestion = -1 ;
  v->sstate     = 0 ;
  v->show_moves = 0 ;
  for (i = 0; i < 64; i++) v->moves[i].move = -1 ;
  for (n = 0; n < 4; n++)
    {
      FOR_BOARD(i) v->moves[n].square[i] = FREE ;
      for (j = 0; j <= n; j++) v->moves[n].square[ivals[j]] = icolors[j] ;
      v->moves[n].player     = icolors[n] ;
      v->moves[n].move       = ivals[n] ;
      v->moves[n].moves_left = 63 - n ;
    }

  FOR_BOARD(i) v->old_board.square[i] = v->board.square[i] = FREE ;
  v->board.square[27] = WHITE ;
  v->board.square[28] = BLACK ;
  v->board.square[35] = BLACK ;
  v->board.square[36] = WHITE ;
  v->board.moves_left = 60 ;
}


void
load_game()
{

/*  Read in a game from file, and setup internal variables accordingly.
 *
 *  Notation in the games file is of the format:
 *  1, <C-4>   -     [ remarks field ]
 *  2,   -   <E-3>   [ remarks field ]
 *
 *  The move number is the field before the comma. This is just used as a
 *  consistency check. Next is the black move or the white move (surronded
 *  by '<' and '>' characters. Obviously only one piece moves per go, and
 *  the '-' character is used to signify which piece doesn't move (in case
 *  one player had to miss a turn.
 *
 *  As each position is added, the board is redisplayed, and if the game is
 *  not over, the next player can commence.
 */

  char buf[MAXLINE], col, *lptr, *mptr, row ;
  int moveno, n ;
  FILE *fp ;

  if ((fp = fopen(v->gamefile, "r")) == NULL)
    {
      SPRINTF(buf, "Couldn't open game file: %s", v->gamefile) ;
      message(M_PANEL, buf) ;
      return ;
    }

  moveno = 0 ;
  initboard() ;
  v->last_move = -1 ;
  paint_board() ;
  message(M_EVAL, "") ;
  while (fgets(buf, MAXLINE, fp) != NULL)
    {
      if (buf[0] == '\n' || buf[0] == '#') continue ;
      moveno++ ;
      SSCANF(buf, "%d", &n) ;
      if (n != moveno || n > 60)
        {
          SPRINTF(buf, "Load error: incorrect move [%d] on line %d",
                  n, v->move) ;
          message(M_PANEL, buf) ;
          return ;
        }
      lptr = (char *) strchr(buf, '<') ;
      mptr = (char *) strchr(buf, '-') ;
      if (lptr == NULL || mptr == NULL)
        {
          SPRINTF(buf, "Load error: missing < or - on line %d", v->move) ;
          message(M_PANEL, buf) ;
          return ;
        }
      if (lptr < mptr) v->next_player = BLACK ;    /* Black move? */
      else             v->next_player = WHITE ;
      SSCANF(lptr, "<%c-%c>", &row, &col) ;
      if (load_move(row, col) == 0)
        {
          SPRINTF(buf, "Load error: invalid move <%c-%c> on line %d",
                                    row, col, v->move) ;
          message(M_PANEL, buf) ;
          return ;
        }
      SPRINTF(buf, "Loaded move %d for %s", moveno,
              (v->next_player == BLACK) ? v->bstone_name : v->wstone_name) ;
      message(M_PANEL, buf) ;
    }
  FCLOSE(fp) ;

  set_score() ;
  set_turn(OPPONENT(v->next_player)) ;

  v->next_player = OPPONENT(v->next_player) ;

  if (check(v->next_player)) return ;
  v->next_player = OPPONENT(v->next_player) ;
  if (!check(v->next_player)) return ;

  if (v->next_player == BLACK || v->next_player == WHITE)
    {
      opponent_move(v->next_player) ;
      v->next_player = OPPONENT(v->next_player) ;
    }
}


int
load_move(int col, int row)
{
  int i, x, y ;

  if (row < '1' || row > '8') return(0) ;
  if (isupper(col))           col = tolower(col) ;
  if (col < 'a' || col > 'h') return(0) ;
  v->move = (row - '1') * BOARD_SIZE + (col - 'a') ;
  if (legal(v->move, v->next_player, &v->board) == 0) return(0) ;
  (void) formfliplist(v->move, v->next_player) ;
  if (v->do_last)
    show_last(v->last_move, 0) ;
  if (v->do_number)
    show_number(v->last_move, 59 - v->board.moves_left, 0) ;
  FOR_BOARD(i)
    if (v->board.square[i] != v->old_board.square[i])
      draw_piece(v->next_player, i / BOARD_SIZE, i % BOARD_SIZE,
                 v->cell_width, v->cell_height) ;
  if (v->do_last)
    show_last(v->move, 1) ;
  if (v->do_number)
    show_number(v->move, 60 - v->board.moves_left, 1) ;
  v->last_move = v->move ;
  return(1) ;
}


void
opponent_move(int player)
{
  for (;;)
    {
      if (!v->processing)
        {
          if (check(player))
            {
              set_cursor(C_HOURGLASS) ;
              v->best_cmove = -1 ;
              message(M_EVAL, "") ;
              v->move = v->moves[63 - v->board.moves_left].move ;
              reset_clock(player) ;
#ifndef NO_NETWORK
              if (v->isremote)
                {
                  write_to_sock(v->socketfd, v->move) ;
                  if (v->opp_iconise) close_reve() ;
                }
              else
#endif /*!NO_NETWORK*/
                write_to_reve(M_MOVE, v->board.square, player, v->level) ;
#ifndef HASREVEPROC
              if (v->opp_iconise) close_reve() ;
#endif /*HASREVEPROC*/
              return ;
            }
          else
            {
              v->next_player = OPPONENT(v->next_player) ;
              return ;
            }
        }
      else
        {
          v->processing = 0 ;
          update_clock(v->next_player, 1) ;
          do_opponent_move(player) ;
          reset_clock(OPPONENT(player)) ;
          if (check(OPPONENT(player)) || v->gameover)
            {
              v->next_player = OPPONENT(v->next_player) ;
              if (v->opp_iconise) open_reve() ;
              if (v->opp_bell)    beep() ;
              if (v->opp_raise)   raise_reve() ;
              return ;
            }
        }    
    }    
}


void
save_game_to(FILE *fp)
{
  int n ;

  for (n = 4; n < 64; n++)
    {
      if (v->moves[n].move == -1) break ;
      if (v->moves[n].player == BLACK)
        FPRINTF(fp, "%d,  <%c-%c>   -  [ note : %ld ]\n",
                n-3, (v->moves[n].move % 8)  + 'a',
                     (v->moves[n].move >> 3) + '1', v->moves[n].note) ;
      else
        FPRINTF(fp, "%d,  -   <%c-%c>  [ note : %ld ]\n",
                n-3, (v->moves[n].move % 8)  + 'a',
                     (v->moves[n].move >> 3) + '1', v->moves[n].note) ;
    }
}


void
save_game()
{

/*  Save the current game status to file.
 *
 *  The notation in the saved games file is of the format:
 *  1, <C-4>   -    [ note : 12345 ]
 *  2,   -   <E-3>  [ note : 0 ]
 *
 *  The move number is the field before the comma. This is just used as a
 *  consistency check. Next is the black move or the white move (surronded
 *  by '<' and '>' characters. Obviously only one piece moves per go, and
 *  the '-' character is used to signify which piece doesn't move (in case
 *  one player had to miss a turn.
 */

  char buf[MAXLINE] ;
  FILE *fp ;

  if ((fp = fopen(v->gamefile, "w")) == NULL)
    {
      SPRINTF(buf, "Couldn't open game file: %s", v->gamefile) ;
      message(M_PANEL, buf) ;
      return ;
    }
  save_game_to(fp) ;
  FCLOSE(fp) ;
  SPRINTF(buf, "Current game saved in %s", v->gamefile) ;
  message(M_PANEL, buf) ;
}


void
set_eval(int player, int move, long note)
{
  SPRINTF(v->line, "%s: <%c-%c> eval : %ld depth: %d",
          (player == BLACK) ? v->bstone_name : v->wstone_name,
          (move & 7) + 'a', (move >> 3) + '1', note, c->profmax) ;
  message(M_EVAL, v->line) ;
}


void
set_score()
{
  SPRINTF(v->line, "%2d", count(&v->board, BLACK)) ;
  message(M_BSCORE, v->line) ;
  SPRINTF(v->line, "%2d", count(&v->board, WHITE)) ;
  message(M_WSCORE, v->line) ;
}


void
set_turn(int player)
{
  SPRINTF(v->line, "%s to move",
          (player == BLACK) ? v->bstone_name : v->wstone_name) ;
  message(M_TURN, v->line) ;
}


void
show_all(int state)
{
  int count, i, piece, player ;

  piece = v->next_player ;
  if (state)
    {
      FOR_BOARD(i) v->s_all.square[i] = FREE ;
      FOR_BOARD(i)
        if (v->board.square[i] == FREE &&
            (count = legal(i, v->next_player, &v->board)) != 0)
          {
            v->s_all.square[i] = v->next_player ;
            draw_piece(piece, i / BOARD_SIZE, i % BOARD_SIZE,
                       v->cell_width / 2, v->cell_height / 2) ;
            show_number(i, count, 1) ;
          }
    }
  else
    {
      FOR_BOARD(i)
        if (v->s_all.square[i] != FREE)
          {
            v->s_all.square[i] = FREE ;
            draw_piece(FREE, i / BOARD_SIZE, i % BOARD_SIZE,
                       v->cell_width, v->cell_height) ;
          }
      if (v->suggestion != -1 && v->sstate)
        {
          player = OPPONENT(v->next_player) ;
          do_suggestion(player, v->suggestion, v->snote, 1) ;
        }
    }    
  v->show_moves = state ;
}


void
show_best(int move, long note)
{
  if (move == -1 || !v->do_bestmove) return ;
  if (v->best_cmove != -1)
    {
      draw_square(v->best_cmove, 0, 2) ;
      if (v->do_number)
        show_number(v->best_cmove, v->cmove_depth, 0) ;
    }
  v->best_cmove  = move ;
  v->cmove_depth = c->profmax ;
  draw_square(v->best_cmove, 1, 2) ;
  if (v->do_number)
    show_number(v->best_cmove, v->cmove_depth, 1) ;
  if (v->show_notes)
    set_eval(v->next_player, move, note) ;
}


void
show_last(int move, int state)
{
  if (move == -1) return ;
  if (v->board.moves_left < 60) draw_square(move, state, 2) ;
}


void
update_board_image(int player, int taken)
{
  update_pieces(player, taken) ;
  v->last_move = v->move ;
  if (v->show_notes)
    if ((player == BLACK && v->black_plays == COMPUTER) ||
        (player == WHITE && v->white_plays == COMPUTER))
      set_eval(player, v->move, c->note) ;
}


void
update_pieces(int player, int taken)
{
  int col, flips, i, piece, row, total_flips, x, y ;

  total_flips = 4 ;

  if (v->do_last)
    show_last(v->last_move, 0) ;
  if (v->do_number)
    show_number(v->last_move, 59 - v->board.moves_left, 0) ;
  if (v->show_moves || (v->invalid && v->show_legal))
    show_all(0) ;
  if (v->suggestion != -1)
    draw_suggest(v->suggestion, 0) ;

  v->suggestion = -1 ;
  v->sstate     = 0 ;
  for (flips = 0; flips < total_flips; flips++)
    {
      FOR_BOARD(i)
        {
          if (v->board.square[i] != v->old_board.square[i])
            {
              if (i == v->move) piece = v->board.square[i] ;
              else
                piece = (flips % 2) ? v->board.square[i]
                                    : v->board.square[i] * -1 ;
              draw_piece(piece, i / BOARD_SIZE, i % BOARD_SIZE,
                         v->cell_width, v->cell_height) ;
            } 
        }    
      PAUSE ;
    } 
  if (v->do_last)
    show_last(v->move, 1) ;
  if (v->do_number)
    show_number(v->move, 60 - v->board.moves_left, 1) ;
  set_score() ;
  set_turn(OPPONENT(player)) ;
  if (!v->do_last && !v->do_number)
    SPRINTF(v->line, "%s moved at <%c-%c> and took %d %s",
                  (player == BLACK) ? v->bstone_name : v->wstone_name,
                  (v->move % 8)  + 'a', (v->move >> 3) + '1',
                  taken, (taken  == 1) ? "stone" : "stones") ;
  else
    SPRINTF(v->line, "%s took %d %s",
                  (player == BLACK) ? v->bstone_name : v->wstone_name,
                  taken, (taken  == 1) ? "stone" : "stones") ;
  message(M_PANEL, v->line) ;
}


void
who_wins()
{
  int cs, ps ;

  ps = count(&v->board, WHITE) ;
  cs = count(&v->board, BLACK) ;
  if (ps > cs)
    {
      SPRINTF(v->line, "%s wins %d-%d", v->wstone_name, ps, cs) ;
      message(M_PANEL, v->line) ;
    }
  else if (ps == cs)
    {
      SPRINTF(v->line,"A tie %d-%d", ps, cs) ;
      message(M_PANEL, v->line) ;
    }
  else
    {
      SPRINTF(v->line, "%s wins %d-%d", v->bstone_name, cs, ps) ;
      message(M_PANEL, v->line) ;
    }
  message(M_TURN, "") ;
  message(M_EVAL, "") ;
}
