/*
** 
**  $Header: /usr/home/hellmann/nedit_add_ons/RCS/xcalltre.c,v 1.36 96/02/23 11:29:59 hellmann Exp $
** 
**  FILENAME:  xcalltre.c
** 
**   PURPOSE:  Provide functions for displaying CallTree using X windows.
** 
**    AUTHOR:  Doug Hellmann
** 
**      DATE:  6 September 1995
** 
**  COMMENTS: 
** 
**    RCSLOG:   $Log:	xcalltre.c,v $
** Revision 1.36  96/02/23  11:29:59  hellmann
** Removed tag code, since we can't guarantee that we can use it.
** 
** Revision 1.35  96/02/21  17:39:41  hellmann
** Added/updated header.
** 
** 
** 
*/

#include "client.h"

#include "calltreP.h"
#include "calltree.h"
#include "xcalltre.h"
#include "Tree.h"
#include "ComboBox.h"
#include "Aldora.h"

Widget callTreeTree;
Widget callTreeTreeRoot = NULL;
Widget callTreeSW;
Widget callTreeDlg = NULL;
Widget callTreeFile;
Widget callTreeCommand;
Widget callTreeRoot;
Widget callTreeDepth;
Widget callTreeShowUnknowns;
Widget callTreeDirection;
Widget callTreeExpandMode;
Widget callTreeSearchFor;
extern CallTree * call_tree;


/*
 *
 * PURPOSE: Go to the tag displayed on the button.
 * 
 * CALLED AS RESULT OF: Button press from child of call tree display
 *
*/
void CallTree_TagCB(Widget w, XtPointer xtp1, XtPointer xtp2)
{
  CallTree * ct = (CallTree*) xtp1;
  char buf[1024];
  char *dummy1, *dummy2;
  int success;
  char wd[MAXPATHLEN];
  char filename[MAXPATHLEN], path[MAXPATHLEN];

  
  if (!ct)
    return;
    
  if (!ct->fcn_name)
    return;
    
  if (!*(ct->fcn_name))
    return;
    
    getcwd(wd, sizeof(wd));
    ParseFilename(ct->file_name, filename, path);
    
    RemoteEditFile(path, filename, ct->line_num);
   
}

/*
 *
 * PURPOSE: Collapse the branch of the display specified.
 *
 * PARAMETERS: Root of branch (is left alone), parent of tree.
 *
 * RETURNS: none
 *
*/
static void CallTree_CollapseVisual(Widget root_node)
{
  WidgetList children;
  Cardinal i;
#ifdef  DEBUG_Xcalltre
  static int depth = 0;
#endif

  
  if (!root_node)
  {
#ifdef DEBUG_Xcalltre  
    fprintf(stderr, "Can not collapse null root\n");
#endif
    return;
  }
  
  XtVaGetValues(root_node, XmNuserData, &children, NULL);
  
  if (!children)
  {
#ifdef DEBUG_Xcalltre  
    fprintf(stderr, "No children\n");
#endif
    return;
  }
  
  for (i = 0; children[i] != NULL; i++)
  {
#ifdef DEBUG_Xcalltre  
     fprintf(stderr, "...collapsing %d:%d\n", depth, i);
     depth++;
#endif

     CallTree_CollapseVisual(children[i]);

#ifdef  DEBUG_Xcalltre
     depth--;
#endif

     XtDestroyWidget(children[i]);
  }
  
  XtFree( (char*) children);
  XtVaSetValues(root_node, XmNuserData, NULL, NULL);  
}


/*
 *
 * PURPOSE: Collapse a branch of the display.
 * 
 * CALLED AS RESULT OF: Press of any collapse button causes recursive call.
 *
*/
void CallTree_CollapseCB(Widget w, XtPointer xtp1, XtPointer xtp2)
{
  Widget parent = (Widget) xtp1;
  XtPointer userData;
  
  XtVaGetValues(XtParent(w), XmNuserData, &userData, NULL);

  if (!userData)
  {
#ifdef  DEBUG_Xcalltre
    fprintf(stderr, "Got no user data\n");
#endif

    return;
  }
  
  XtUnmanageChild( parent );
  BeginWait(callTreeDlg);
  CallTree_CollapseVisual(XtParent(w));
  XtManageChild( parent );
  EndWait(callTreeDlg);
}

/*
 *
 * PURPOSE: Expand the display visual beginning with this root node.
 *
 * PARAMETERS: Widget which is child of tree widget.
 *
 * RETURNS: None
 *
*/
void CallTree_ExpandVisual(Widget root, int how_far, int use_search)
{
  Widget parent = (Widget) XtParent(root);
  CallTree * ct;
  CallTree * sub_node;
  List * sub_node_list;
  WidgetList subnodes;
  WidgetList children;
  Cardinal num_children;
  Widget tag_btn;
  int num_subnodes, i;
  Position x, y;
  int mode = OptionMenuGetValue(callTreeDirection);
  int unknown = XmToggleButtonGetState(callTreeShowUnknowns);
  
  if ( (how_far < 0) && !use_search )
  {
#ifdef DEBUG_Xcalltre
    fprintf(stderr, "NOT expanding %s because how_far < 0 (%d) or !use_search (%d)\n",
    	how_far, use_search);
#endif
    return;
  }
  
  XtVaGetValues(root, 
  	XmNchildren, &children, 
  	XmNnumChildren, &num_children, 
  	XmNuserData, &subnodes, 
  	XmNx, &x,
  	XmNy, &y,
  	NULL);
  	
  tag_btn = NULL;
  for (i = 0; i < num_children; i++)
    if (!strcmp(XtName(children[i]), "tag_btn"))
    {
      tag_btn = children[i];
      break;
    }
  
  XtVaGetValues(tag_btn, XmNuserData, &ct, NULL);
    
  if (use_search && ct->search_result)
    return;
  		
#ifdef  DEBUG_Xcalltre
  fprintf(stderr, "Expanding node at %d, %d\n", x, y);
#endif
  
  if (!ct)
  {
#ifdef  DEBUG_Xcalltre
  fprintf(stderr, "NOT expanding because !ct\n");
#endif
    return;
  }
    
  if (subnodes)
  {
#ifdef  DEBUG_Xcalltre
  fprintf(stderr, "NOT expanding because already have subnodes\n");
#endif
    return;
  }
    
  num_subnodes = ListSize(ct->calls);
  
  if (!num_subnodes)
  {
#ifdef  DEBUG_Xcalltre
  fprintf(stderr, "NOT expanding because there will be no subnodes\n");
#endif
    return;
  }
    
  subnodes = (WidgetList) XtMalloc ( sizeof(Widget) * (num_subnodes + 1) );
  
  if (mode == SHOW_FUNCTIONS_CALLED)
    sub_node_list = ct->calls;
  else
  if (mode == SHOW_FUNCTIONS_WHICH_CALL)
    sub_node_list = ct->called_by;
  else
    sub_node_list = NULL;
    
  for (sub_node = ListReset(sub_node_list), i = 0;
       sub_node != NULL;
       sub_node = ListNext(sub_node_list))
  {
    if ((sub_node == ct) && (use_search))
    {
#ifdef DEBUG_Xcalltre
      fprintf(stderr, "\t skipped recursive subnode %d\n", i);
#endif
      continue;
    }
      
    subnodes[i] = CallTree_Display(sub_node, parent, root, how_far, mode, unknown, use_search);

    if (subnodes[i])
    {
#ifdef DEBUG_Xcalltre
      fprintf(stderr, "\t expanded subnode %d\n", i);
#endif
      i++;
    }
  }
  subnodes[i] = NULL;
  
  if (i > 0)
    XtVaSetValues(root, XmNuserData, subnodes, NULL);
  else
    XtFree( (XtPointer) subnodes);
  
}


/*
 *
 * PURPOSE: Expand a branch of the display tree.
 * 
 * CALLED AS RESULT OF: User press of any expand button.
 *
*/
void CallTree_ExpandCB(Widget w, XtPointer xtp1, XtPointer xtp2)
{ 
  BeginWait(callTreeDlg);    
  XtUnmanageChild(XtParent(w));
  CallTree_ExpandVisual(XtParent(w), GetPrefCflowExpandDepth() - 1, False);
  EndWait(callTreeDlg);  
  XtManageChild(XtParent(w));
}

/*
 *
 * PURPOSE: Force the specified node to be the root of the tree parent.
 *
 * PARAMETERS: wicget which should be the root of the tree
 *
 * RETURNS: none
 *
*/
void ReRoot(Widget new_root)
{
  Widget super_node;
  
  	/* get the super node and siblings of the new root */
  	
  XtVaGetValues(new_root, XtNsuperNode, &super_node, NULL);
  
  if (!super_node)
  {
  	/* this node is the root node and can not be rerooted */
  	
    return;
  }
  else
  {
    WidgetList siblings;
    WidgetList children;
    WidgetList new_siblings;
    int c, i, j;
    Cardinal num_children;
    Widget old_root = callTreeTreeRoot;
    
  	/* reset super node of new node to NULL, 
  	   meaning this node will be the root node  */
  	
    XtVaSetValues(new_root, XtNsuperNode, NULL, NULL);
    callTreeTreeRoot = new_root;

  	/* get the siblings of this node and remove new_root */
  	
    XtVaGetValues(super_node, XmNuserData, &siblings, NULL);
    
    for (c = 0; siblings[c] != NULL; c++);
    new_siblings = (WidgetList) XtMalloc( sizeof(Widget) * (c+1) );
    
#ifdef  DEBUG_Xcalltre
    fprintf(stderr, "got %d siblings\n", c);
#endif

    
    for (i = 0, j = 0; i < c+1; i++)
    {
      if (siblings[i] == NULL)
        break;
        
      if (siblings[i] == new_root)
      {
#ifdef  DEBUG_Xcalltre
        fprintf(stderr, "skipping %s at %d\n",
        	XtName(siblings[i]), i);
#endif

        continue;
      }
      else
      {
#ifdef  DEBUG_Xcalltre
        fprintf(stderr, "copying %s at %d to %d\n", 
        	XtName(siblings[i]), i, j);
#endif

	new_siblings[j] = siblings[i];
	j++;
      }
    }
    new_siblings[c] = NULL;
    
    XtFree( (char*)siblings);
    
    fprintf(stderr, "Setting user data\n");
    
    XtVaSetValues(super_node, XmNuserData, new_siblings, NULL);

  	/* delete the nodes from other branches or higher in the tree */

    fprintf(stderr, "collapsing old node\n");
    
    CallTree_CollapseVisual(old_root);
/*
    fprintf(stderr, "collapsing super node\n");
    
    CallTree_CollapseVisual(super_node);

    fprintf(stderr, "destroying super node\n");

    XtDestroyWidget(super_node);
    
    XtVaGetValues(new_root, XmNchildren, &children, 
    			    XmNnumChildren, &num_children, 
    			    NULL);
    
    fprintf(stderr, "got %d children\n", num_children);
    for (c = 0; c < num_children; c++)
    {
#ifdef  DEBUG_Xcalltre
      fprintf(stderr, "%d: %s\n", c, XtName(children[c]));
#endif

      if (!strcmp("reroot_btn", XtName(children[c])))
      {
        XtDestroyWidget(children[c]);
#ifndef  DEBUG_Xcalltre
        break;
#endif

      }
    } 
*/

    
  }
  
}


/*
 *
 * PURPOSE: Make the selected function the root of the tree.
 * 
 * CALLED AS RESULT OF: pressing reroot button on a node.
 *
*/
void CallTree_ReRootCB(Widget w, XtPointer xtp1, XtPointer xtp2)
{
  Widget parent = (Widget) xtp1;
  CallTree * ct;
  
  BeginWait(callTreeDlg);
  
  XtVaGetValues(w, XmNuserData, &ct, NULL);

#ifndef  COMPLEX_REROOT  
  XmComboBoxSetString(callTreeRoot, ct->fcn_name);
  CallTree_UpdateDisplayCB(w, xtp1, xtp2);
#else

/*
  XtUnmanageChild(callTreeTree);
*/

  
  ReRoot(XtParent(w));
  
  XtVaSetValues(callTreeRoot, XmNvalue, ct->fcn_name, NULL); 

/*
  XtManageChild(callTreeTree);
*/


#endif

  EndWait(callTreeDlg);

}


/*
 *
 * PURPOSE: Build a tree display hierarchy based on the specified call tree.
 *
 * PARAMETERS: Call tree, parent widget, root widget (NULL if tree should be root).
 *		show_undefined (assumes that if there are no calls from this
 *		function, we don't have a definition for it).
 * RETURNS: node created
 *
*/
Widget CallTree_Display(CallTree * tree, 
		      Widget parent, 
		      Widget root, 
		      int depth,
		      int mode,
		      int show_unknown,
		      int use_search)
{
  CallTree * sub_tree;
  List * sub_node_list;
  Widget rc = NULL;
  Widget tag_btn, reroot_btn, expand_btn, collapse_btn;
  WidgetList subnodes;
  int num_subnodes;
  int num_local_subnodes;
  Boolean unknown;
  char * dummy1;
  char * dummy2;
  Boolean have_tag;
    
  if (!tree)
  {
    DialogF(DF_ERR, parent, 1, "Can not display an empty tree.", "Ok");
    return NULL;
  }
  
  if (use_search && !tree->search_result && (tree->search_subnode <= 0))
  {
#ifdef DEBUG_Xcalltre_search
    fprintf(stderr, "\t%d: not displaying %s because use_search but not this node or subnode\n",
    		__LINE__, tree->fcn_name);
      fprintf(stderr, "\tsearch_result %d search_subnode %d\n", 
      		tree->search_result, tree->search_subnode);
#endif
    return;
  }
/* 
  else
  if (!tree->search_result)  
    tree->search_subnode--;
 */
  
  if ( (depth < 0) && !use_search )
  {
#ifdef DEBUG_Xcalltre
    fprintf(stderr, "\tnot displaying %s because depth < 0 or not use_search\n",
    		tree->fcn_name);
#endif
    return NULL;
  }
  
  if ( use_search && (tree->search_result <= 0) && (tree->search_subnode <= 0) )
  {
#ifdef DEBUG_Xcalltre_search
    fprintf(stderr, "\t%d: not displaying %s because using search and not found\n",
    		__LINE__, tree->fcn_name);
    fprintf(stderr, "\t search_result = %d search_subnode = %d\n", 
    		tree->search_result, tree->search_subnode);
#endif
    return NULL;
  }

  if (mode == SHOW_FUNCTIONS_CALLED)
    sub_node_list = tree->calls;
  else
  if (mode == SHOW_FUNCTIONS_WHICH_CALL)
    sub_node_list = tree->called_by;
  else
    sub_node_list = NULL;
  
  num_local_subnodes = CallTree_CountLocalChildren(tree, mode);
  num_subnodes = ListSize(sub_node_list);
  
  	/*
  	** Display the node even if it is unknown if if would be the root node.
  	** Otherwise nothing would be displayed.
  	*/
  
  unknown = (tree->line_num < 0) && (root != NULL);
  
  if (unknown && !show_unknown)
    return NULL;
    
  	/*
  	** Reset unknown to properly control the method of display later.
  	*/

  unknown = tree->line_num < 0;
     
  if (tree->fcn_name)
  {
      
#ifdef  DEBUG_Xcalltre
      fprintf(stderr, "adding node for %s at depth %d\n", tree->fcn_name, depth);
      fprintf(stderr, "\tsearch_result %d search_subnode %d\n", 
      		tree->search_result, tree->search_subnode);
#endif
    
      rc = XtVaCreateManagedWidget("tree_node", xmRowColumnWidgetClass, parent,
  		    XtNsuperNode, root,
  		    XmNorientation, XmHORIZONTAL,
  		    XmNspacing, 0,
  		    XmNentryBorder, 0,
  		    XmNpacking, XmPACK_TIGHT,
  		    NULL);
  		    
      if (root == NULL)
        callTreeTreeRoot = rc;
      
      if ((num_local_subnodes > 0) && root)
      {
		reroot_btn = XtVaCreateManagedWidget("reroot_btn", xmArrowButtonWidgetClass,
      				  rc,
      				  XmNuserData, tree,
      				  XmNarrowDirection, XmARROW_LEFT,
      				  NULL);
		XtAddCallback(reroot_btn, XmNactivateCallback, CallTree_ReRootCB, (XtPointer) parent);
		AldoraHelp_Add(reroot_btn, "Make this function the root of the tree.");
      }
      
      if (!unknown)
      {
        XmString label;
        label = XmStringCreateSimple(tree->fcn_name);
        tag_btn = XtVaCreateManagedWidget("tag_btn", xmPushButtonWidgetClass,
  		    rc,
  		    XmNuserData, tree,
  		    XmNlabelString, label,
  		    NULL);
        XtAddCallback(tag_btn, XmNactivateCallback, CallTree_TagCB, (XtPointer) tree);
        AldoraHelp_Add(tag_btn, "Jump to the definition of this function.");
        XmStringFree(label);
      }
      else
      {
        XmString label;
        label = XmStringCreateSimple(tree->fcn_name);
        tag_btn = XtVaCreateManagedWidget("tag_label", xmLabelWidgetClass,
        	    rc,
  		    XmNlabelString, label,
        	    NULL);
        XmStringFree(label);
      }
      
      if (num_local_subnodes > 0)
      {
		 collapse_btn = XtVaCreateManagedWidget("collapse_btn", xmArrowButtonWidgetClass, 
				  rc,
				  XmNuserData, tree,
				  XmNarrowDirection, XmARROW_LEFT,
      				  NULL);
      		 XtAddCallback(collapse_btn, XmNactivateCallback, CallTree_CollapseCB, (XtPointer) parent);
        	 AldoraHelp_Add(collapse_btn, "Collapse all nodes rooted at this node.");

		 expand_btn = XtVaCreateManagedWidget("expand_btn", xmArrowButtonWidgetClass, 
		    	rc,
		    	XmNuserData, tree,
		    	XmNarrowDirection, XmARROW_RIGHT,
  		    	NULL);
        	 XtAddCallback(expand_btn, XmNactivateCallback, CallTree_ExpandCB, (XtPointer) parent);
        	 AldoraHelp_Add(expand_btn, "Expand tree at this node.");
      } 

  }
  
  if (!(use_search && tree->search_result))
  {    		  
    if ( ((depth > 0) && (num_subnodes > 0)) || (use_search && tree->search_subnode) )
    {
      CallTree_ExpandVisual(rc, depth - 1, use_search);
    }
  }
  
  if (!tree->search_result)  
    tree->search_subnode--;
    
  
  return rc;
}

/*
 *
 * PURPOSE: Create the interface for the call tree display.
 *
 * PARAMETERS: Parent dialog.
 *
 * RETURNS: none
 *
*/
void CallTree_InterfaceCreate(Widget parent)
{
  int i, j;
  Widget form, sep, update, cancel, redisplay;
  Widget rc;
  Widget frame, frame_title;
  Widget sub_form;

#define COLUMN_POSITION (25)
#define MAIN_COLUMN_POSITION (50)

  if (!callTreeDlg)
    callTreeDlg = parent;
  
  form = XtVaCreateManagedWidget("nedit_call_tree_form", xmFormWidgetClass,
  		parent,
  		NULL);
  		
  	/* create action area */
  	
  redisplay = XtVaCreateManagedWidget("Refresh", xmPushButtonWidgetClass,
  		form,
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNbottomAttachment, XmATTACH_FORM,
  		XmNbottomOffset, 4,
  		XmNleftOffset, 4,
  		NULL);
  XtAddCallback(redisplay, XmNactivateCallback, CallTree_UpdateDisplayCB, NULL);
  XtVaCreateWidget("AldoraHelp", aldoraHelpObjectClass, redisplay, NULL);
  AldoraHelp_SetActive((int)GetPrefBalloonHelp());
       	
  cancel = XtVaCreateManagedWidget("Exit", xmPushButtonWidgetClass,
  		form,
  		XmNrightAttachment, XmATTACH_FORM,
  		XmNbottomAttachment, XmATTACH_FORM,
  		XmNbottomOffset, 4,
  		XmNrightOffset, 4,
  		NULL);
  XtAddCallback(cancel, XmNactivateCallback, ClientExitCB, NULL);
  
  sep = XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, form,
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNrightAttachment, XmATTACH_FORM,
  		XmNbottomAttachment, XmATTACH_WIDGET,
  		XmNbottomWidget, cancel,
  		NULL);

  rc = XtVaCreateManagedWidget("options_rc", xmRowColumnWidgetClass, form,
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNtopAttachment, XmATTACH_FORM,
  		XmNbottomAttachment, XmATTACH_WIDGET,
  		XmNbottomWidget, sep,
  		XmNorientation, XmVERTICAL,
  		NULL);

  /* root of tree */
  	
  frame = XtVaCreateManagedWidget("root_frame", xmFrameWidgetClass,
  		rc,
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNtopAttachment, XmATTACH_FORM,
  		XmNrightAttachment, XmATTACH_POSITION,
  		XmNrightPosition, MAIN_COLUMN_POSITION,
  		XmNshadowType, XmSHADOW_IN,
  		NULL);
  	frame_title = XtVaCreateManagedWidget("Base Function", xmLabelGadgetClass, frame,
  		XmNchildType, XmFRAME_TITLE_CHILD,
  		NULL);
  	sub_form = XtVaCreateManagedWidget("root_form", /* xmFormWidgetClass, */ xmRowColumnWidgetClass,
  		frame,
  		XmNchildType, XmFRAME_WORKAREA_CHILD,
  		NULL);

		/*
		** What direction does the relationship flow?
		*/
  		
  	callTreeDirection = XmVaCreateSimpleOptionMenu( sub_form, "direction_menu", NULL,
  		NULL, 0,
  		CallTree_UpdateDisplayCB,
  		XmVaPUSHBUTTON, NULL, NULL, NULL, NULL, /* calls displayed functions */
  		XmVaPUSHBUTTON, NULL, NULL, NULL, NULL, /* called by displayed functions */
  		
  		XmNrightAttachment, XmATTACH_POSITION,
  		XmNrightPosition, COLUMN_POSITION,
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNbottomAttachment, XmATTACH_FORM,
  		XmNtopAttachment, XmATTACH_WIDGET,
  		XmNtopWidget, callTreeDepth,
  		NULL);
  	XtManageChild(callTreeDirection);
  	AldoraHelp_Add(callTreeDirection, "Display all calls made from root or\nall calls made to root function.");

  	callTreeRoot = XtVaCreateManagedWidget("call_tree_root", xmComboBoxWidgetClass,
  		sub_form,
  		
  			/* positioning */
  		
  		XmNbottomAttachment, XmATTACH_FORM,
  		XmNleftAttachment, XmATTACH_POSITION,
  		XmNleftPosition, COLUMN_POSITION,
  		XmNrightAttachment, XmATTACH_FORM,
  		XmNtopAttachment, XmATTACH_FORM,
  		
  			/* ComboBox resources */
  		
  		XmNeditable, True,
  		XmNvisibleItemCount, 16,
  		XmNshowLabel, False,
  		XmNsquareArrow, True,
  		
  		NULL);
  	XtAddCallback(callTreeRoot, XmNactivateCallback, CallTree_UpdateDisplayCB, NULL);
  	XtAddCallback(callTreeRoot, XmNselectionCallback, CallTree_UpdateDisplayCB, NULL);
  	AldoraHelp_Add(callTreeRoot, "Specify function name from which to start.");

  /* expansion method to use */
  
  frame = XtVaCreateManagedWidget("expansion_frame", xmFrameWidgetClass,
  		rc,
  		XmNtopAttachment, XmATTACH_WIDGET,
  		XmNtopWidget, frame,
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNrightAttachment, XmATTACH_POSITION,
  		XmNrightPosition, MAIN_COLUMN_POSITION,
  		XmNshadowType, XmSHADOW_IN,
  		NULL);
  	frame_title = XtVaCreateManagedWidget("Node Expansion", xmLabelGadgetClass,
  		frame,
  		XmNchildType, XmFRAME_TITLE_CHILD,
  		NULL);
  	sub_form = XtVaCreateManagedWidget("expansion_form", 
  		/* xmFormWidgetClass, */ xmRowColumnWidgetClass,
  		frame,
  		XmNchildType, XmFRAME_WORKAREA_CHILD,
  		XmNorientation, XmVERTICAL,
  		NULL);

  	callTreeExpandMode = XmVaCreateSimpleOptionMenu( sub_form, "mode_menu", NULL,
  		NULL, 0,
  		CallTree_UpdateDisplayCB,
  		XmVaPUSHBUTTON, NULL, NULL, NULL, NULL, /* simple */
  		XmVaPUSHBUTTON, NULL, NULL, NULL, NULL, /* search */
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNtopAttachment, XmATTACH_FORM,
  		XmNrightAttachment, XmATTACH_POSITION,
  		XmNrightPosition, COLUMN_POSITION,
  		NULL);
	  XtManageChild(callTreeExpandMode);
	  AldoraHelp_Add(callTreeExpandMode, "Expand in steps or search for a reference.");
 
		/* scale for setting depth of expansion */

  	callTreeDepth = XtVaCreateManagedWidget("call_tree_depth", xmScaleWidgetClass,
  		sub_form,
  		XmNtopAttachment, XmATTACH_FORM,
  		XmNleftAttachment, XmATTACH_POSITION,
  		XmNleftPosition, COLUMN_POSITION,
  		XmNrightAttachment, XmATTACH_FORM,
  		  		
  		XmNmaximum, 10,
  		XmNminimum,  0,
  		XmNorientation, XmHORIZONTAL,
  		XmNshowValue, True,
  		XmNvalue, GetPrefCflowExpandDepth(),
  		XmNtitleString, XmStringCreateSimple("Simple expansion depth"),
  		
  		NULL);
	  XtAddCallback(callTreeDepth, XmNvalueChangedCallback, ChangeExpansionDepthCB, NULL);
	  AldoraHelp_Add(callTreeDepth, "Each node will be automatically expanded to this depth.");

	  callTreeSearchFor = XtVaCreateManagedWidget("call_tree_search_for", xmComboBoxWidgetClass,
  		sub_form,
  		
  			/* positioning */
  		
  		XmNbottomAttachment, XmATTACH_FORM,
  		XmNleftAttachment, XmATTACH_POSITION,
  		XmNleftPosition, COLUMN_POSITION,
  		XmNrightAttachment, XmATTACH_FORM,
  		XmNtopAttachment, XmATTACH_WIDGET,
  		XmNtopWidget, callTreeDepth,
  		
  			/* ComboBox resources */
  		
  		XmNeditable, True,
  		XmNvisibleItemCount, 16,
  		XmNshowLabel, False,
  		XmNsquareArrow, True,
  		
  		XmNlabelString, XmStringCreateSimple("Search target"),
  		XmNshowLabel, True,
  		
  		NULL);
	  XtAddCallback(callTreeSearchFor, XmNactivateCallback, CallTree_UpdateDisplayCB, NULL);
	  XtAddCallback(callTreeSearchFor, XmNselectionCallback, CallTree_UpdateDisplayCB, NULL);
	  AldoraHelp_Add(callTreeSearchFor, "Specify function name for Search expansion mode.");
  
		/* 
		** should we show functions for which we don't have a 
		** definition? 
		*/
		
	  XtVaCreateManagedWidget("sep2", xmSeparatorWidgetClass, sub_form, NULL);
	
	  callTreeShowUnknowns = XtVaCreateManagedWidget("Show non-local functions",
  		xmToggleButtonWidgetClass,
  		sub_form,
  		XmNbottomAttachment, XmATTACH_FORM,
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNtopAttachment, XmATTACH_WIDGET,
  		XmNtopWidget, callTreeExpandMode,
  		XmNrightAttachment, XmATTACH_POSITION,
  		XmNrightPosition, COLUMN_POSITION,
  		XmNset, (int) GetPrefShowUnknown(),
  		NULL);
	  XtAddCallback(callTreeShowUnknowns, XmNvalueChangedCallback, CallTree_UpdateDisplayCB, NULL);
	  AldoraHelp_Add(callTreeShowUnknowns, "Control display of functions for\nwhich definitions are not found.");
  
  /* data source info */
  	
  frame = XtVaCreateManagedWidget("data_source_frame", xmFrameWidgetClass,
  		rc,
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNtopAttachment, XmATTACH_WIDGET,
  		XmNtopWidget, frame,
  		XmNrightAttachment, XmATTACH_POSITION,
  		XmNrightPosition, MAIN_COLUMN_POSITION,
  		XmNbottomAttachment, XmATTACH_WIDGET,
  		XmNbottomWidget, sep,
  		XmNshadowType, XmSHADOW_IN,
  		NULL);
  	frame_title = XtVaCreateManagedWidget("Data Source", xmLabelGadgetClass,
  		frame,
  		XmNchildType, XmFRAME_TITLE_CHILD,
  		NULL);
  	sub_form = XtVaCreateManagedWidget("data_sub_form", xmFormWidgetClass,
  		frame,
  		XmNchildType, XmFRAME_WORKAREA_CHILD,
  		NULL);

  		/* update by reading a file */
  	
  	update = XtVaCreateManagedWidget("File:", xmPushButtonWidgetClass,
  		sub_form,
  		XmNtopAttachment, XmATTACH_FORM,
  		XmNtopOffset, 4,
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNrightAttachment, XmATTACH_POSITION,
  		XmNrightPosition, COLUMN_POSITION,
  		XmNbottomOffset, 4,
  		NULL);
	  XtAddCallback(update, XmNactivateCallback, CallTree_UpdateDisplayFromFileCB, NULL);
	  AldoraHelp_Add(update, "Re-reads the cflow tree from the specified file.");

	  callTreeFile = XtVaCreateManagedWidget("call_tree_file", xmTextFieldWidgetClass,
  		sub_form,
  		XmNleftAttachment, XmATTACH_POSITION,
  		XmNleftPosition, COLUMN_POSITION,
  		XmNrightAttachment, XmATTACH_FORM,
  		XmNtopAttachment, XmATTACH_FORM,
  		NULL);
	  XtAddCallback(callTreeFile, XmNactivateCallback, CallTree_UpdateDisplayFromFileCB, NULL);
	  XmTextFieldSetString(callTreeFile, (char*) GetPrefCflowFile());
	  AldoraHelp_Add(callTreeFile, "Name of the file containing the cflow tree data.");

  		/* update from output of a command */

	  update = XtVaCreateManagedWidget("Command:", xmPushButtonWidgetClass,
  		sub_form,
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNtopAttachment, XmATTACH_WIDGET,
  		XmNtopWidget, callTreeFile,
  		XmNtopOffset, 4,
  		XmNrightAttachment, XmATTACH_POSITION,
  		XmNrightPosition, COLUMN_POSITION,
  		NULL);
  	XtAddCallback(update, XmNactivateCallback, CallTree_UpdateDisplayFromCommandCB, NULL);  
  	AldoraHelp_Add(update, "Executes shell command to rebuild the cflow file.\nThis assumes that the cflow tree is\nwritten to stdout.");		

  	callTreeCommand = XtVaCreateManagedWidget("call_tree_command", xmTextFieldWidgetClass,
  		sub_form,
  		XmNtopAttachment, XmATTACH_WIDGET,
  		XmNtopWidget, callTreeFile,
  		XmNrightAttachment, XmATTACH_FORM,
  		XmNleftAttachment, XmATTACH_POSITION,
  		XmNleftPosition, COLUMN_POSITION,
  		XmNbottomAttachment, XmATTACH_FORM,
  		NULL);
	  XtAddCallback(callTreeCommand, XmNactivateCallback, CallTree_UpdateDisplayFromCommandCB, NULL);
	  XmTextFieldSetString(callTreeCommand, (char*) GetPrefCflowCommand());
	  AldoraHelp_Add(callTreeCommand, "Enter shell command to rebuild the cflow file.\nCflow tree must be written to stdout.");		
  	

	/* the tree */

  callTreeSW = XmCreateScrolledWindow(form, "call_tree_SW", NULL, 0);
  XtVaSetValues(callTreeSW,
    		 XmNwidth, 500,
    		 XmNheight, 500,
    		 XmNleftAttachment, XmATTACH_WIDGET,
    		 XmNleftWidget, rc,
    		 XmNrightAttachment, XmATTACH_FORM,
    		 XmNbottomAttachment, XmATTACH_WIDGET,
    		 XmNbottomWidget, sep, 
    		 XmNtopAttachment, XmATTACH_FORM,
    		 NULL);
  XtManageChild(callTreeSW);

  callTreeTree = XtVaCreateManagedWidget("nedit_call_tree", XstreeWidgetClass,
    	      callTreeSW,
    	      XmNborderWidth, 0,
    	      XtNsquareLines, True,
    	      NULL);
    	      
}


/*
 *
 * PURPOSE: Create the dialog for the call tree display.
 *
 * PARAMETERS: None.
 *
 * RETURNS: Dialog created.
 *
*/
Widget CallTree_DialogCreate()
{
  int i, j;
  Widget form, sep, update, cancel;

  callTreeDlg = NeditCreateToplevelDialog("nedit_call_tree", "MCflow", "MCflow");
  CallTree_InterfaceCreate(callTreeDlg);
  return callTreeDlg;
}


/*
 *
 * PURPOSE: Update the call tree display by reading the interface widgets
 *          and building a new tree if needed.
 * 
 * CALLED AS RESULT OF: Explicitly from CallTreeCB and as callback of widgets
 *                      on the call tree dialog.
 *
*/
void CallTree_UpdateDisplayFromCommandCB(Widget w, XtPointer xtp1, XtPointer xtp2)
{
  char * command;
  
  BeginWait(callTreeDlg);
  
  if (call_tree)
      CallTree_Delete(call_tree);
  XtDestroyWidget(callTreeTree);
  callTreeTree = XtVaCreateManagedWidget("nedit_call_tree", XstreeWidgetClass,
    	      callTreeSW,
    	      XmNborderWidth, 0,
    	      XtNsquareLines, True,
    	      NULL);
  
  command = XmTextFieldGetString(callTreeCommand);
  call_tree = CallTree_BuildFromCommand(command);
  XtFree(command);
  CallTree_UpdateComboBox(call_tree, False);

  if (!call_tree || (ListSize(call_tree->all_fcns) == 0) )
  {
    DialogF(DF_WARN, callTreeDlg, 1, "No call tree was constructed.", "Ok");
  }
  else
  {  
    CallTree_UpdateDisplayCB(w, xtp1, xtp2);
  }
  
  EndWait(callTreeDlg);

}

/*
 *
 * PURPOSE: Update the call tree display by reading the interface widgets
 *          and building a new tree if needed.
 * 
 * CALLED AS RESULT OF: Explicitly from CallTreeCB and as callback of widgets
 *                      on the call tree dialog.
 *
*/
void CallTree_UpdateDisplayFromFileCB(Widget w, XtPointer xtp1, XtPointer xtp2)
{
  char * filename;
  
  BeginWait(callTreeDlg);
  
  if (call_tree)
      CallTree_Delete(call_tree);
  XtDestroyWidget(callTreeTree);
  callTreeTree = XtVaCreateManagedWidget("nedit_call_tree", XstreeWidgetClass,
		      callTreeSW,
		      XmNborderWidth, 0,
		      XtNsquareLines, True,
		      NULL);
  
  filename = XmTextFieldGetString(callTreeFile);
  
  call_tree = CallTree_ReadFromFile(filename);
  CallTree_UpdateComboBox(call_tree, False);
  XtFree(filename);
  
  if (!call_tree || (ListSize(call_tree->all_fcns) == 0) )
  {
    DialogF(DF_WARN, callTreeDlg, 1, "Error constructing call tree from file.", "Ok");
  }
  else
  { 
    CallTree_UpdateDisplayCB(w, xtp1, xtp2);
  }

  EndWait(callTreeDlg);
}

/*
 *
 * PURPOSE: Update the call tree display by reading the interface widgets
 *          and building a new tree if needed.
 * 
 * CALLED AS RESULT OF: Explicitly from CallTreeCB and as callback of widgets
 *                      on the call tree dialog.
 *
*/
void CallTree_UpdateDisplayCB(Widget w, XtPointer xtp1, XtPointer xtp2)
{
  char * filename;
  char * root_name;
  CallTree * root_of_tree = NULL;
  int mode = OptionMenuGetValue(callTreeDirection);
  int unknown = XmToggleButtonGetState(callTreeShowUnknowns);
  int use_search = OptionMenuGetValue(callTreeExpandMode);
  
  BeginWait(callTreeDlg);
  
  if (call_tree)
  {
	
	XtDestroyWidget(callTreeTree);
	callTreeTree = XtVaCreateManagedWidget("nedit_call_tree", XstreeWidgetClass,
    	    	callTreeSW,
    	    	XmNborderWidth, 0,
    	        XtNsquareLines, True,
    	    	NULL);

	root_name = XmComboBoxGetString(callTreeRoot);
	root_of_tree = CallTree_FindNode(call_tree, root_name);
	
	if (use_search)
	{
		char * search_target = XmComboBoxGetString(callTreeSearchFor);
		
		if (*search_target)
		{
	#ifdef  DEBUG_Xcalltre_search
		  fprintf(stderr, "Searching for '%s'\n", search_target);
	#endif

		  CallTree_ClearPath(root_of_tree);
		  CallTree_MarkPath(root_of_tree, search_target, mode);

	#ifdef  DEBUG_Xcalltre_search
		  fprintf(stderr, "Got %d hits from '%s'\n", 
		  	root_of_tree->search_subnode, root_of_tree->fcn_name);
	#endif
		}
		else
		{
	#ifdef  DEBUG_Xcalltre
		  fprintf(stderr, "MCflow: No search target\n");
	#endif
		  use_search = False;
		}

		XtFree(search_target);
	}

	if (root_of_tree) 
          CallTree_Display(root_of_tree, callTreeTree, NULL, GetPrefCflowExpandDepth(), mode, unknown, use_search);
	else
	if (*root_name)
	  DialogF(DF_WARN, callTreeDlg, 1, "The function %s was not found.", "Ok", root_name);
	
	XtFree(root_name);
  }
  else
  {
    DialogF(DF_ERR, callTreeDlg, 1, "No call tree information has been loaded.", "Ok");
  }
  
  EndWait(callTreeDlg);
}

/*
 *
 * PURPOSE: Try to set the ComboBox value to the specified string.
 *
 * PARAMETERS: ComboBox, value, notify_widget
 *
 * RETURNS: True if succeeds, False if fails.
 *
*/
Boolean ComboBoxTryToSetSelection(Widget cb, char * value, Boolean notify)
{
  XmString xstr;
  Boolean Return = False;
  
  if (!cb)
    return False;
  if (!value)
    return False;
  if (!*value)
    return False;
  
  xstr = XmStringCreateLtoR(value, XmSTRING_DEFAULT_CHARSET);
  
  if (XmComboBoxItemExists(cb, xstr))
  {
    Return = True;
    XmComboBoxSelectItem(cb, xstr, notify);
  }
  
  XmStringFree(xstr);
  return Return;
}


/*
 *
 * PURPOSE: Update the ComboBox of function names after loading a new call tree.
 *
 * PARAMETERS: CallTree to get the names.
 *
 * RETURNS: None.
 *
*/
void CallTree_UpdateComboBox(CallTree * call_Tree, Boolean notify)
{
  char * old_value;
  char * old_search_value;
  
#ifdef DEBUG_Xcalltre
  fprintf(stderr, "Updating combo box %s\n", XtName(callTreeRoot));
  fprintf(stderr, "and %s\n", XtName(callTreeSearchFor));
#endif

  XtVaGetValues(callTreeRoot, XmNvalue, &old_value, NULL);
  XtVaSetValues(callTreeRoot, XmNvalue, "", NULL);  
  XmComboBoxDeleteAllItems(callTreeRoot);

  XtVaGetValues(callTreeSearchFor, XmNvalue, &old_search_value, NULL);
  XtVaSetValues(callTreeSearchFor, XmNvalue, "", NULL);
  XmComboBoxDeleteAllItems(callTreeSearchFor);
  
  if (call_tree)
  {
    CallTree * ct;
    XmString xstr;
    
    for (ct = ListReset(call_tree->all_fcns);
    	 ct != NULL;
    	 ct = ListNext(call_tree->all_fcns))
    {
      if (ListSize(ct->calls) > 0)
      {
		xstr = XmStringCreateLtoR(ct->fcn_name, XmSTRING_DEFAULT_CHARSET);
		XmComboBoxAddItem(callTreeRoot, xstr, 0);
		XmComboBoxAddItem(callTreeSearchFor, xstr, 0);
		XmStringFree(xstr);
      }
    }
    
    if (!ComboBoxTryToSetSelection(callTreeRoot, old_value, False))
      ComboBoxTryToSetSelection(callTreeRoot, "main", False);

    if (!ComboBoxTryToSetSelection(callTreeSearchFor, old_search_value, False))
      ComboBoxTryToSetSelection(callTreeSearchFor, "", False);
      
    if (notify)
      CallTree_UpdateDisplayCB(NULL, NULL, NULL);
      
  }
  
  XtFree(old_value);
  XtFree(old_search_value);
}
