/*
 *  $Header: /usr/home/hellmann/nedit_add_ons/RCS/mctags.c,v 1.12 96/05/17 10:47:02 hellmann Exp $
 *
 *  FILENAME: mctags.c
 *
 *  PURPOSE: Interface to the tag file.
 *
 *  AUTHOR: Doug Hellmann
 *
 *  DATE: 26 August 1995
 *
 *  COMMENTS: 
 *
 *   RCSLOG:   $Log:	mctags.c,v $
** Revision 1.12  96/05/17  10:47:02  hellmann
** Try to optimize display function a little.
** 
** Revision 1.11  96/05/17  10:26:23  hellmann
** Don't need so sort items as they are added to the list, because they 
** should come to us sorted.
** 
** Revision 1.10  96/05/17  10:18:14  hellmann
** Renamed several functions which clashed with tags.c
** 
** Revision 1.9  96/03/18  13:47:48  hellmann
** Fixed aldora colors
** 
** Revision 1.8  96/03/04  16:38:31  hellmann
** Fixed resource specs.
** 
** Revision 1.7  96/03/01  15:10:13  hellmann
** Attempt to make colors and other resources more consistent.
** 
** Revision 1.6  96/02/29  08:59:35  hellmann
** Return NULL from menu creation function since we haven't implemented teh
** menu yet.
** 
** Revision 1.5  96/02/21  17:39:37  hellmann
** Added/updated header.
** 
 *
*/

#include <stdio.h>
#include <stdlib.h>

#include "client.h"
#include "getfiles.h"
#include "stack.h"
#include "tags.h"
#include <search.h>

	/* D E F I N I T I O N S */

#ifdef LOCAL_PREF_FILE_NAME
#undef LOCAL_PREF_FILE_NAME
#endif
#define LOCAL_PREF_FILE_NAME ".mctags"

#ifdef APP_NAME
#undef APP_NAME
#endif
#define APP_NAME "mctags" /* application name for loading resources */

#ifdef APP_CLASS
#undef APP_CLASS
#endif
#define APP_CLASS "MCtags"

#define MAXLINE 512
#define MAX_TAG_LEN 80

	/* T Y P E D E F S */

enum searchDirection {FORWARD, BACKWARD};

typedef struct {
  Widget shell;
  Widget file_list;
  Widget tag_list;
  Widget text_field;
  
  Widget update;
  Widget dismiss;
} tagListWindowStruct;

typedef struct {
    char *name;
    char *file;
    char *searchString;
} tag;

	/* G L O B A L S */

XtAppContext context;
static tag *Tags = NULL;
static char TagPath[MAXPATHLEN];
static char TagFileName[MAXPATHLEN];
tagListWindowStruct * tlws = NULL;
Widget tagHistoryDlg = NULL;
Widget tagHistoryList = NULL;
static Stack * tagStack = NULL;

char HeaderText[] = "\
# Preferences file for MCtags\n\
#\n\
# This file is created automatically by MCtags.\n\
#\n\
# Codes to be embedded in editor option: \n\
#\n\
#    %f - filename\n\
#    %t - tag name\n\
#    %l - line number\n\
#\n";

char *fallbackResources[] = {
    "NEdit*menuBar.marginHeight: 1",
    "NEdit*pane.sashHeight: 11",
    "NEdit*pane.sashWidth: 11",
    "NEdit*text.selectionArrayCount: 3",
    "NEdit*fontList:-adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",
    "NEdit*XmList.fontList:-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-*-*",
    "NEdit*XmText.fontList:-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-*-*",
    "NEdit*XmTextField.fontList:-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-*-*",
    "NEdit*background: grey",
    "NEdit*foreground: black",
    "NEdit*statisticsLine.background: #b3b3b3",
    
    "NEdit*text.background: #e5e5e5",
    "NEdit*text.foreground: black",
    "NEdit*XmText.foreground: black",
    "NEdit*XmText.background: white",
    "NEdit*XmList.foreground: black",
    "NEdit*XmList.background: white",
    "NEdit*XmTextField.background: #cccccc",
    "NEdit*XmTextField.foreground: black",
    
    "NEdit*call_tree_SW.scrollingPolicy: AUTOMATIC",
    "NEdit*call_tree_SW.scrollBarDisplayPolicy: STATIC",
    
    "NEdit*direction_menu*button_0.labelString: Functions referenced by :",
    "NEdit*direction_menu*button_1.labelString: Functions which reference :",

    "NEdit*mode_menu*button_0.labelString: Simple expansion",
    "NEdit*mode_menu*button_1.labelString: Search",
    
    "NEdit*call_tree.foreground: green",

    "*AldoraHelp*background: lemon chiffon",
    "*AldoraHelp*foreground: black",
    
    "NEdit*XmToggleButton.selectorColor: red",
    NULL
};

char cmdLineHelp[] =
#ifndef VMS
"Usage:  mctags [-tagFile ctags_file] \n\
		[-editor editor_name]\n\
		[-font font] [-display [host]:server[.screen]\n\
               [-geometry geometry] [-xrm resourcestring] [file...]\n";
#else
"";
#endif /*VMS*/

XrmOptionDescRec LocalOpTable[] = {
    {"-editor", ".editor", XrmoptionSepArg, (caddr_t)NULL},
    {"-tagFile", ".tagFile", XrmoptionSepArg, (caddr_t)NULL},
};

struct localPrefData {
    char editor[MAXPATHLEN];		/* name of editor command under X */
    char tagFile[MAXPATHLEN];		/* name of ctags file */
    char ctags[MAXPATHLEN];		/* name of ctags command */
} LocalPrefData;

PrefDescripRec LocalPrefDescrip[] = {
    {"tagFile", "TagFile", PREF_STRING,
    	"tags", LocalPrefData.tagFile, 
    	(void *)sizeof(LocalPrefData.tagFile), True},
    {"editor", "Editor", PREF_STRING,
    	"xterm -title %f -exec vi +%l %f &",
    	LocalPrefData.editor, (void *)sizeof(LocalPrefData.editor), True},
    {"ctags", "Ctags", PREF_STRING,
    	"ctags -tw *.[ch]",
    	LocalPrefData.ctags, (void *)sizeof(LocalPrefData.ctags), True},
};

	/* P R E D E C L A R A T I O N S   A N D   P R O T O T Y P E S */   

extern char * GetPrefCtagsCommand();
void editTagByName(char * string, Widget parent);
void ShowTagHistoryWindow(Widget w, XtPointer xtp1, XtPointer xtp2);
extern Widget TagMenuCreate(Widget);
static void freeTagList(tag *tags);
static void setTag(tag *t, char *name, char *file, char *searchString);

/*
** Functions to return preferences values.
*/
char * GetPrefEditorCommand(void)
{
  return LocalPrefData.editor;
}

char * GetPrefCtagsCommand(void)
{
  return LocalPrefData.ctags;
}

char * GetLocalPrefTagFile(void)
{
  return LocalPrefData.tagFile;
}

/*
 * update the contents of the tags list window
 *
*/
static void UpdateTagsWindow()
{
  int t;
  char *last_added;
  
  XmListDeleteAllItems(tlws->file_list);
  XmListDeleteAllItems(tlws->tag_list);
  
  if (Tags == (tag*)NULL)
  {
    fprintf(stderr, "No tags found\n");
    return;
  }

  last_added = NULL;
  for (t = 0; Tags[t].name != NULL; t++)
  {
     int add_it = FALSE;
     
     if (!last_added)
       add_it = TRUE;
     else
       add_it = strcmp(Tags[t].name, last_added);
     
     if (add_it)
     {
	XmString file_name;
	
     	last_added = Tags[t].file;
     	
	file_name = XmStringCreateSimple(Tags[t].file);

	if (!XmListItemExists(tlws->file_list, file_name))
	  XmListAddItemAlpha(tlws->file_list, Tags[t].file);

	XmStringFree(file_name);
     }
  }
  
  if (*TagFileName)
    XtVaSetValues(tlws->shell, XmNtitle, TagFileName, NULL);
  
}

static void UpdateTagInterface()
{
    UpdateTagsWindow();
}

/*
 *
 * PURPOSE: Add an item to the top of the stack and update the display.
 *
 * PARAMETERS: String name of tag, if NULL clears stack.
 *
 * RETURNS: None.
 *
*/
void TagStackUpdate(char * tag_name)
{
  
/*    if (!tagStack)
      tagStack = Stack_New();
      
  if (!tagHistoryList)
      ShowTagHistoryWindow(NULL, NULL, NULL);
 */  
      
  if (!tag_name)
  {
    Stack_Delete(tagStack, TRUE);
    tagStack = Stack_New();

    XmListDeleteAllItems(tagHistoryList);

    return;
  }
  else
  {
    XmString xstr;
    
    Stack_Push(tagStack, (void*)strdup(tag_name), free);
    
    xstr = XmStringCreateLtoR(tag_name, XmSTRING_DEFAULT_CHARSET);

    XmListAddItem(tagHistoryList, xstr, 0);
    XmStringFree(xstr);
    
  }
}

/*
 *
 * PURPOSE: Goes to one of the tags specified on the tag list, lowering
 *          the stack back to that point.
 * 
 * CALLED AS RESULT OF: List callback of tagHistoryList
 *
*/
void TagStackGotoCB(Widget w, XtPointer xtp1, XtPointer xtp2)
{
   XmListCallbackStruct * cbs = (XmListCallbackStruct*) xtp2;
   int click_pos, num_items;
   char * tag_name;
   
   /*
   ** Since we can't find out how many elements there are
   ** in a stack, we find that out from the XmList.  Then
   ** we pop the appropriate number of items from the stack
   ** based on which item in the list the user selected.
   **
   ** The last pop can be used to find out where to go.
   **
   ** When we use editTagByName, it is going to push this
   ** tag back on the stack for us.
   */
   
   BeginWait(tagHistoryDlg);
   
   click_pos = cbs->item_position;
   XtVaGetValues(w, XmNitemCount, &num_items, NULL);
   
   while (num_items >= click_pos)
   {
     XmListDeletePos(w, 0);
     tag_name = Stack_Pop(tagStack);
     num_items--;
   }
   
   editTagByName(tag_name, tagHistoryDlg);
   
   EndWait(tagHistoryDlg);
   
}

/*
** find the line in the file which matches the tag search pattern
*/
static int SearchForTagLine(char * pathname, char * filename, char * searchString)
{
    int startPos, endPos, found=FALSE, hasBOL, hasEOL, fileLen, searchLen, dir;
    char *fileString, searchSubs[MAXLINE];
    char fullName[MAXPATHLEN];
    int lineNum = 0, cPos;
    
    /* get the entire (sigh) text buffer from the text area widget */
    sprintf(fullName, "%s%s", pathname, filename);
    fileString = FileToString(fullName);
    fileLen = strlen(fileString);

    /* remove / .. / or ? .. ? and substitute ^ and $ with \n */
    searchLen = strlen(searchString);
    if (searchString[0] == '/')
    	dir = FORWARD;
    else if (searchString[0] == '?')
    	dir = BACKWARD;
    else {
    	fprintf(stderr, "NEdit: Error parsing tag file search string");
    	return 0;
    }
    searchLen -= 2;
    strncpy(searchSubs, &searchString[1], searchLen);
    searchSubs[searchLen] = '\0';
    hasBOL = searchSubs[0] == '^';
    hasEOL = searchSubs[searchLen-1] == '$';
    if (hasBOL) searchSubs[0] = '\n';
    if (hasEOL) searchSubs[searchLen-1] = '\n';

    /* search for newline-substituted string in the file */
    if (dir==FORWARD)
    	found = SearchString(fileString, searchSubs, SEARCH_FORWARD,
    		SEARCH_CASE_SENSE, False, 0, &startPos, &endPos);
    else
    	found = SearchString(fileString, searchSubs, SEARCH_BACKWARD,
    		SEARCH_CASE_SENSE, False, fileLen, &startPos, &endPos);
    if (found) {
    	if (hasBOL) startPos++;
    	if (hasEOL) endPos--;
    }
    
    /* if not found: ^ may match beginning of file, $ may match end */
    if (!found && hasBOL) {
    	found = strncmp(&searchSubs[1], fileString, searchLen-1);
    	if (found) {
    	    startPos = 0;
    	    endPos = searchLen - 2;
    	}
    }
    if (!found && hasEOL) {	    
    	found = strncmp(searchSubs, fileString+fileLen-searchLen+1,
    		 searchLen-1);
    	if (found) {
    	    startPos = fileLen-searchLen+2;
    	    endPos = fileLen;
    	}
    }

    
    /* return the result */
    if (!found) {
	XtFree(fileString);
	return 0;
    }

    for (cPos = 0; cPos <= startPos; cPos++)
    {
      switch (fileString[cPos])
      {
        case '\n':
          lineNum++;
        break;
        case '\r':
          lineNum++;
        break;
        default:
        break;
      }
    }
    
    XtFree(fileString);
    return lineNum + 1;
}

/*
** Given a name, lookup a file, search string.  Returned strings are pointers
** to internal storage which are valid until the next LoadTagsFile call.
*/
int LookupMCTag(char *name, char **file, char **searchString)
{
    int i;
    tag *t;
    
    for (i=0, t=Tags; t->name!=NULL; i++, t++) {
 	if (!strcmp(t->name, name)) {
 	    *file = t->file;
 	    *searchString = t->searchString;
 	    return TRUE;
	}
    }
    return FALSE;
}



/*
 * Goto the location of a tag by giving the tag name and a parent for error
 * dialogs.
 *
*/
void editTagByName(char * string, Widget parent)
{
    int selStart, selEnd, startPos, endPos, found, lineNum;
    char *fileToSearch, *searchString, *eptr;
    WindowInfo *windowToSearch;
    char filename[MAXPATHLEN], pathname[MAXPATHLEN], temp[MAXPATHLEN];
    
    if (string == NULL)
      return;
    if (parent == NULL)
      return;
      
    /* lookup the name in the tags file */
    found = LookupMCTag(string, &fileToSearch, &searchString);
    if (!found && parent) {
    	DialogF(DF_WARN, parent, 1, "%s not found in tags file", "OK",
    		string);
    	return;
    }

    /* if the path is not absolute, qualify file path with directory
       from which tags file was loaded */
    if (fileToSearch[0] == '/')
    	strcpy(temp, fileToSearch);
    else {
    	strcpy(temp, TagPath);
    	strcat(temp, fileToSearch);
    }
    ParseFilename(temp, filename, pathname);
    
    /* if the search string is a number, select the numbered line */
    lineNum = strtol(searchString, &eptr, 10);
    if (eptr == searchString) 
    {
      lineNum = SearchForTagLine(pathname, filename, searchString);
    }

    RemoteEditFile(pathname, filename, lineNum);    
}

/*
 * Update the tags file, re-load it, and rebuild the menus.
 *
*/
void tagUpdateCallback(Widget w, XtPointer ignore1, XtPointer ignore)
{
  char * command_output;
  char * command_str = NULL;
  int success;
  
  command_str = GetPrefCtagsCommand();
  if (!command_str || !*command_str)
    return;
    
  command_output = issueCommandToText(tlws->shell, command_str, "", 
  				ACCUMULATE | ERROR_DIALOGS, NULL, 0, 0, 
  				&success);
  
  LoadMCTagsFile("tags");
}


/*
 *
 * PURPOSE: Check to see if the file name specified matches the partial
 *          file name passed.
 *
 * PARAMETERS: full file name to compare to
 *             partial file name to compare against
 *
 * RETURNS: TRUE if full ends with partial, FALSE otherwise
 *
*/
Boolean PartialFileNameMatches(char * full_filename, char * partial_filename)
{
  int len_full, len_part;
  char * compare_point;
  Boolean Return;
  
  if (!full_filename)
    return False;
  if (!partial_filename)
    return False;
  
  len_full = strlen(full_filename);
  len_part = strlen(partial_filename);
  
  if (len_part > len_full)
    return False;
    
  compare_point = full_filename + len_full - len_part;
    
  Return = !strcmp(compare_point, partial_filename);
  
  return Return;
}


/*
 * Show the tags for the selected file.
 *
*/
void TagsWindowShowTags(Widget w, XtPointer xtp1, XmListCallbackStruct * cbs)
{
  int t;
  char * file_name;
  
  BeginWait(tlws->shell);
  
  XmListDeleteAllItems(tlws->tag_list);
  
  XmStringGetLtoR(cbs->item, XmSTRING_DEFAULT_CHARSET, &file_name);
  
  for (t = 0; Tags[t].name != NULL; t++)
  {
    if (PartialFileNameMatches(Tags[t].file, file_name))
    {
      XmString tag_name = XmStringCreateSimple(Tags[t].name);
      XmListAddItem(tlws->tag_list, tag_name, 0);
      XmStringFree(tag_name);
    }
  }
  
  EndWait(tlws->shell);
  
  free(file_name);
}

/*
 * Goto the tag selected in the tag list.
 *
*/
void TagsWindowGotoTag(Widget w, XtPointer xtp1, XmListCallbackStruct * cbs)
{
  char * tag_name;
  
  XmStringGetLtoR(cbs->item, XmSTRING_DEFAULT_CHARSET, &tag_name);
  BeginWait(tlws->shell);
  editTagByName(tag_name, w);
  free(tag_name);
  EndWait(tlws->shell);
}

/*
 * Goto the tag entered in the text field.
 *
*/
void TagsWindowTextFieldGotoTag(Widget w, XtPointer xtp1, XtPointer xtp2)
{
   char * tag_name = XmTextGetString(w);
   BeginWait(tlws->shell);
   editTagByName(tag_name, w);
   free(tag_name); 
   EndWait(tlws->shell);
}

/*
 * Open the file selected by the user.
 *
*/
void TagsWindowOpenFile(Widget parent, tagListWindowStruct * tlws,
			XmListCallbackStruct * cbs)
{
  char * temp;
  char filename[MAXPATHLEN], pathname[MAXPATHLEN];
  WindowInfo * windowToSearch;
  
  BeginWait(tlws->shell);

  XmStringGetLtoR(cbs->item, XmSTRING_DEFAULT_CHARSET, &temp);

    ParseFilename(temp, filename, pathname);
    
    /* open the file containing the definition */
    EditExistingFile(WindowList, filename, pathname, FALSE);
    windowToSearch = (WindowInfo*) FindWindowWithFile(filename, pathname);
    if (windowToSearch == NULL) {
    	DialogF(DF_WARN, parent, 1, "File %s not found", 
    		"OK", temp);
    	EndWait(tlws->shell);
    	return;    	
    }
    EndWait(tlws->shell);
}

/*
 *
 * PURPOSE: Display (create when needed) the tag history dialog.
 * 
 * CALLED AS RESULT OF: Tag menu button.
 *
*/
void ShowTagHistoryWindow(Widget w, XtPointer xtp1, XtPointer xtp2)
{	
	if (tagHistoryDlg == NULL)
	{
    	  Widget form;
    	  Widget menuBar, menuPane, btn;
          Pixmap iconPixmap = 0, maskPixmap = 0;

    	  tagHistoryDlg =   NeditCreateToplevelDialog("nedit_tag_history", 
    				  "MCtags History", "MCtags History");

    	  form = XtVaCreateManagedWidget("window_form", xmFormWidgetClass, tagHistoryDlg, NULL);

    	  /*
    	   * Create list of windows.
    	  */				
    	   tagHistoryList = XmCreateScrolledList(form, "tag_history_list", NULL, 0);
    	   XtVaSetValues(tagHistoryList,
  			  XmNvisibleItemCount, 15, 
  			  XmNwidth, 200,
  			  XmNselectionPolicy, XmSINGLE_SELECT,
  			  XmNscrollingPolicy, XmAUTOMATIC,
  			  XmNscrollBarDisplayPolicy, XmSTATIC,
  			  NULL);
    	   XtVaSetValues(XtParent(tagHistoryList),
     			  XmNleftAttachment, XmATTACH_FORM,
     			  XmNrightAttachment, XmATTACH_FORM,
     			  XmNbottomAttachment, XmATTACH_FORM,
     			  XmNtopAttachment, XmATTACH_FORM,
     			  NULL);
    	   /* XtAddCallback(tagHistoryList, XmNdefaultActionCallback, TagStackGotoCB, (XtPointer) NULL); */
    	   XtAddCallback(tagHistoryList, XmNsingleSelectionCallback, TagStackGotoCB, (XtPointer) NULL);

    	   XtManageChild(tagHistoryList);
	}
	
	XtPopdown(tagHistoryDlg);
	XtPopup(tagHistoryDlg, XtGrabNone);
}


/*
 * Show the user the dialog for quickly locating tags within files.
 *
*/
void ShowTagsWindow(Widget w, XtPointer xtp1, XtPointer xtp2)
{
  if (!tlws)
  {
    static Pixmap iconPixmap = 0, maskPixmap = 0;
    Widget menubar, menu;
    Widget form, sep;
    Widget tags_list_window;
    Widget label1, label2, label3;
    Widget frame1, frame2, frame3;
    
    tlws = (tagListWindowStruct*) malloc(sizeof(tagListWindowStruct));
  		  
    tags_list_window = NeditCreateToplevelDialog("nedit_tags_list",
    			"MCtags", "MCtags");
    
    tlws->shell = tags_list_window;
		
	/* the form is the root widget */

    form = XtVaCreateWidget("tags_list_form", xmFormWidgetClass, 
    				tags_list_window,
    				XmNtopOffset, 2,
    				XmNbottomOffset, 2,
    				XmNleftOffset, 2,
    				XmNrightOffset, 2,
    				NULL);
    				
    /* create the menu bar */
    
    menubar = XmCreateMenuBar(form, "menuBar", NULL, 0);
    XtVaSetValues(menubar, 
    		XmNtopAttachment, XmATTACH_FORM,
    		XmNleftAttachment, XmATTACH_FORM,
    		XmNrightAttachment, XmATTACH_FORM,
    		NULL);    
    menu = TagMenuCreate(menubar);

	/* create the action area buttons */

    tlws->update = XtVaCreateManagedWidget("Update", xmPushButtonWidgetClass,
    				form,
    				XmNleftAttachment, XmATTACH_FORM,
    				XmNbottomAttachment, XmATTACH_FORM,
    				XmNleftOffset, 4,
    				XmNbottomOffset, 4,
    				NULL);
    XtAddCallback(tlws->update, XmNactivateCallback, 
			(XtCallbackProc) tagUpdateCallback, (XtPointer) tlws);

    tlws->dismiss = XtVaCreateManagedWidget("Exit", xmPushButtonWidgetClass,
    				form,
    				XmNrightAttachment, XmATTACH_FORM,
    				XmNbottomAttachment, XmATTACH_FORM,
    				XmNrightOffset, 4,
    				XmNbottomOffset, 4,
    				NULL);
    XtAddCallback(tlws->dismiss, XmNactivateCallback, 
			(XtCallbackProc) ClientExitCB, (XtPointer) tlws);

    sep = XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, form,
    				XmNleftAttachment, XmATTACH_FORM,
    				XmNrightAttachment, XmATTACH_FORM,
    				XmNbottomAttachment, XmATTACH_WIDGET,
    				XmNbottomWidget, tlws->update,
    				NULL);

	/* text field where the user can enter a tag by hand */

    tlws->text_field = XtVaCreateManagedWidget("text_field", xmTextWidgetClass,
    				form,
    				XmNleftAttachment, XmATTACH_FORM,
    				XmNrightAttachment, XmATTACH_FORM,
    				XmNbottomAttachment, XmATTACH_WIDGET,
    				XmNbottomWidget, sep,
    				NULL);
    XtAddCallback(tlws->text_field, XmNactivateCallback, 
				(XtCallbackProc) TagsWindowTextFieldGotoTag, 
				(XtPointer) tlws);

	/* create labels and forms for the three lists */


    frame1 = XtVaCreateManagedWidget("frame1", xmFrameWidgetClass, form,
    				       XmNleftAttachment, XmATTACH_FORM,
    				       XmNtopAttachment, XmATTACH_WIDGET,
    				       XmNtopWidget, menubar,
    				       XmNbottomAttachment, XmATTACH_WIDGET,
    				       XmNbottomWidget, tlws->text_field,
    				       NULL);
    label1 = XtVaCreateManagedWidget("Files", xmLabelWidgetClass, 
				frame1,
				XmNchildType, XmFRAME_TITLE_CHILD,
    				NULL);

    frame2 = XtVaCreateManagedWidget("frame2", xmFrameWidgetClass, form,
				  XmNleftAttachment, XmATTACH_WIDGET,
    				  XmNleftWidget, frame1,
    				  XmNtopAttachment, XmATTACH_WIDGET,
    				  XmNtopWidget, menubar,
    				  XmNbottomAttachment, XmATTACH_WIDGET,
    				  XmNbottomWidget, tlws->text_field,
    				  XmNrightAttachment, XmATTACH_FORM,
    				  NULL);
    label2 = XtVaCreateManagedWidget("Tags", xmLabelWidgetClass, 
				frame2,
				XmNchildType, XmFRAME_TITLE_CHILD,
    				NULL);

	/* create the actual lists */

    tlws->file_list = XmCreateScrolledList(frame1, "file_list", NULL, 0);
    XtVaSetValues(tlws->file_list, XmNvisibleItemCount, 15, 
    				   XmNselectionPolicy, XmBROWSE_SELECT,
    				   XmNlistSizePolicy, XmVARIABLE,
    				   XmNwidth, 150,
		               XmNchildType, XmFRAME_WORKAREA_CHILD,
    				   NULL);
    				   
    XtAddCallback(tlws->file_list, XmNbrowseSelectionCallback, 
				(XtCallbackProc) TagsWindowShowTags, (XtPointer) tlws);
    XtAddCallback(tlws->file_list, XmNdefaultActionCallback, 
				(XtCallbackProc) TagsWindowOpenFile, (XtPointer) tlws);
    				      

    tlws->tag_list = XmCreateScrolledList(frame2, "tag_list", NULL, 0);
    XtVaSetValues(tlws->tag_list, XmNvisibleItemCount, 15, 
    				  XmNselectionPolicy, XmSINGLE_SELECT,
    				  XmNlistSizePolicy, XmVARIABLE,
    				  XmNwidth, 250,
				      XmNchildType, XmFRAME_WORKAREA_CHILD,
    				  NULL);
    
    XtAddCallback(tlws->tag_list, XmNsingleSelectionCallback, 
		   (XtCallbackProc) TagsWindowGotoTag, (XtPointer) tlws);

	XtManageChild(menubar);
    XtManageChild(tlws->file_list);
    XtManageChild(tlws->tag_list);

    XtManageChild(form);
    
  }

  UpdateTagsWindow();
  XtPopdown(tlws->shell);
  XtPopup(tlws->shell, XtGrabNone);

}

int LoadMCTagsFile(char *filename)
{
    FILE *fp = NULL;
    char line[MAXLINE], name[MAXLINE], file[MAXLINE], searchString[MAXLINE];
    char unused[MAXPATHLEN];
    char *charErr;
    tag *tags;
    int i, nTags, nRead;
    WindowInfo *w;

    /* Open the file */
    if ((fp = fopen(filename, "r")) == NULL)
    	return FALSE;
	
    /* Read it once to see how many lines there are */
    for (nTags=0; TRUE; nTags++) {
    	charErr = fgets(line, MAXLINE, fp);
    	if (charErr == NULL) {
    	    if (feof(fp))
    	    	break;
    	    else
    	    	return FALSE;
    	}
    }
    
    /* Allocate zeroed memory for list so that it is automatically terminated
       and can be freed by freeTagList at any stage in its construction*/
    tags = (tag *)calloc(nTags + 1, sizeof(tag));
    
    /* Read the file and store its contents */
    rewind(fp);
    for (i=0; i<nTags; i++) {
    	charErr = fgets(line, MAXLINE, fp);
    	if (charErr == NULL) {
    	    if (feof(fp))
    	    	break;
    	    else {
    	    	freeTagList(tags);
    	    	return FALSE;
    	    }
    	}
    	nRead = sscanf(line, "%s\t%s\t%[^\n]", name, file, searchString);
    	if (nRead != 3) {
    	    freeTagList(tags);
    	    return FALSE;
    	}
	setTag(&tags[i], name, file, searchString);
    }
    
    /* Make sure everything was read */
    if (i != nTags) {
    	freeTagList(tags);
    	return FALSE;
    }
    
    /* Replace current tags data and path for retrieving files */
    if (Tags != NULL)
    	freeTagList(Tags);
    Tags = tags;
    ParseFilename(filename, unused, TagPath);
    
    strcpy(TagFileName, filename);
    
    /* Undim the "Find Definition" menu item in the existing windows */
    for (w=WindowList; w!=NULL; w=w->next)
    	XtSetSensitive(w->findDefItem, TRUE);
    return TRUE;
}

int MCTagsFileLoaded()
{
    return Tags != NULL;
}


/*
** Main function
*/

int main(int argc, char **argv)
{
    int i, fileSpecified;
    XrmDatabase prefDB, localPrefDB;
    Widget dialog;
    
    /* Initialize toolkit and open display. */

    XtToolkitInitialize();
    context = XtCreateApplicationContext();
    if (!context)
    {
	XtWarning ("MCflow: Can't get context\n");
	exit(0);
    }

    /* Set up a warning handler to trap obnoxious Xt grab warnings */
    SuppressPassiveGrabWarnings();

    /* Set up default resources if no app-defaults file is found */
    XtAppSetFallbackResources(context, fallbackResources);
    
#if XmVersion >= 1002
    /* Allow users to change tear off menus with X resources */
    XmRepTypeInstallTearOffModelConverter();
#endif /* XmVersion */
    
#ifdef VMS
    /* Convert the command line to Unix style (This is not an ideal solution) */
    ConvertVMSCommandLine(&argc, &argv);
#endif /*VMS*/
    
    /* Read the preferences file and command line into a database */
    prefDB = CreateNEditPrefDB(&argc, argv);
    localPrefDB = CreateLocalPrefDB(LOCAL_PREF_FILE_NAME, APP_NAME, 
    			LocalOpTable, XtNumber(LocalOpTable), &argc, argv);

    /* Open the display and read X database and remaining command line args */
    
    TheDisplay = XtOpenDisplay (context, NULL, APP_NAME, APP_CLASS, NULL,
    	    0, &argc, argv);

    if (!TheDisplay) {
	XtWarning ("MCflow: Can't open display\n");
	exit(0);
    }

    /* Store preferences from the command line and .mcflow file, 
       and set the appropriate preferences */
    RestoreNEditPrefs(prefDB, XtDatabase(TheDisplay));
    RestoreLocalPrefs(APP_NAME, LocalPrefDescrip, XtNumber(LocalPrefDescrip),
    			localPrefDB, XtDatabase(TheDisplay));
    SetPointerCenteredDialogs(GetPrefRepositionDialogs());
    
    SavePreferences(TheDisplay, LOCAL_PREF_FILE_NAME, HeaderText,
    	    LocalPrefDescrip, XtNumber(LocalPrefDescrip)); 
    	    
    /* Process any command line arguments (-tags, and files to edit) not
       already processed by RestoreNEditPrefs. */
    fileSpecified = FALSE;
        
    /* Load the default tags file (as long as -tags was not specified).
       Don't complain if it doesn't load, the tag file resource is
       intended to be set and forgotten.  Running mcflow in a directory
       without a tags should not cause it to spew out errors. */
    if (!MCTagsFileLoaded() && *GetLocalPrefTagFile() != '\0')
    {
       char tagPath[MAXPATHLEN], tagFile[MAXPATHLEN];
       char fullTagFile[MAXPATHLEN];
       char wd[MAXPATHLEN];

       getcwd(wd, sizeof(wd));
       
       ParseFilename(GetLocalPrefTagFile(), tagFile, tagPath);
       
       sprintf(fullTagFile, "%s%s", tagPath, tagFile);
       
       if (!LoadMCTagsFile(fullTagFile))
         fprintf(stderr, "Could not load tags file '%s'\n", fullTagFile);
    }
    	

    ShowTagsWindow(NULL, NULL, NULL);

    /* Process events. */
    XtAppMainLoop(context);

    return 0;
}

void loadTagsCB(Widget w, WindowInfo *in_window, XtPointer callData) 
{
    char filename[MAXPATHLEN];
    int response;
    
    while(True) {
	  response = GetExistingFilename(w, "ctags file:", filename);
	  if (response == GFN_OK)
	  {
    	      if (!LoadMCTagsFile(filename))
    			  DialogF(DF_WARN, w, 1,
    	    		  "Error reading ctags file,\ntags not loaded", "OK");
    	      else
    	    	  break;
      }
      else
    	  break;
    }
}

/*
 * Creates the sub menu for the tags menu item on a window.  The menu contains
 * one button for each tag in the window, and the callback will go to that
 * tag when the button is pressed.
 *
*/
Widget TagMenuCreate(Widget parent)
{
  Widget button;
  Widget menuPane;
  int t;
  
/*   
  menuPane = (Widget) createMenu(parent, "tagMenu", "Tags");
  
  createMenuItem(menuPane, "makeTags", "Update Tags File", 
  		'U', tagUpdateCallback, NULL);
  createMenuItem(menuPane, "stackDialog", "Tag History Dialog", 'T', ShowTagHistoryWindow, NULL);
  createMenuItem(menuPane, "loadTagsFile", "Load Tags File...", 'L',
    	    loadTagsCB, NULL);
  return menuPane;
 */  
 
 return NULL;
}

static void freeTagList(tag *tags)
{
    int i;
    tag *t;
    
    for (i=0, t=tags; t->name!=NULL; i++, t++) {
    	free(t->name);
    	free(t->file);
    	free(t->searchString);
    }
    free(tags);
}

static void setTag(tag *t, char *name, char *file, char *searchString)
{
    t->name = (char *)malloc(sizeof(char) * strlen(name) + 1);
    strcpy(t->name, name);
    t->file = (char *)malloc(sizeof(char) * strlen(file) + 1);
    strcpy(t->file, file);
    t->searchString = (char *)malloc(sizeof(char) * strlen(searchString) + 1);
    strcpy(t->searchString, searchString);
}

