#ifndef _LIST_DRH
#define _LIST_DRH

/**************************************************************
 *
 * list.h
 *
 * This is a module which provides a "linked list" for the
 * programmer.
 *
 * Started on 2/17/93 hellmann
 *
*/

#include <stdlib.h>
#include <stdio.h>

#ifndef FALSE
#define FALSE (0)
#endif

#ifndef TRUE
#define TRUE (1)
#endif

struct ListStruct;
typedef struct ListStruct List;

/**************************************************************
 *
 * ListCreate
 *
 * PURPOSE : Create one occurance of a List.
 *
 * PARAMS  : None.
 *
 * RETURNS : Pointer to a new List.
 *
*/
List * ListCreate(void);

/**************************************************************
 *
 * ListDestroy
 *
 * PURPOSE : To free all of the memory used by a list and it's contents.
 *
 * PARAMS  : TheList - the list to erase
 *
*/
extern int ListDestroy ( List * TheList, int FreeSpace );

/**************************************************************
 *
 * ListPos
 *
 * PURPOSE : Return the Data item from the list at a given position
 *
 * PARAMS  : TheList
 *           position - what position to return
 *                (-1 indicates last item in list)
*/
extern void * ListPos( List * TheList, int position );

/**************************************************************
 *
 * ListAddItem
 *
 * PURPOSE : Add a data element to the given list.
 *
 * PARAMS  : TheList - list to add item to
 *           TheData - data item to add, cast to void *
 *           Destroy - function to erase TheData for
 *                     when List is destroyed
 *           position - where in the list to add it
 *                (-1 indicates end, anywhere else is in list)
 *
*/
extern int ListAddItem( List * TheList,
		  void * TheData,
		  void (* Destroy) (),
		  int position);

/**************************************************************
 *
 * ListFindItem
 *
 * PURPOSE : Find the item if it is in the list
 *
 * PARAMS  : TheList
 *           TheItem - pointer to object of same type
 *           CompareFcn - comparison function
 *
 * RETURNS : position (0 or > ) if in list, -1 if not)
 *
*/
extern int ListFindItem( List * TheList, void * TheItem, int (* CompareFcn) ());


/**************************************************************
 *
 * ListRemoveItem
 *
 * PURPOSE : To take an item out of the list.
 *
 * PARAMS  : TheList
 *           TheItem    - pointer to object of same type
 *           CompareFcn - comparison function
 *           DeleteIt   - if true, delete item, else don't
 *
*/
extern int ListRemoveItem( List * TheList,
		     void * TheItem,
		     int (* CompareFcn) (),
		     int DeleteIt);

/**************************************************************
 *
 * ListRemovePos
 *
 * PURPOSE : To remove the item at the given position in the list.
 *
 * PARAMS  : TheList
 *           position
 *           DeleteIt   - if true, delete item, else don't
 *
*/
extern int ListRemovePos ( List * TheList,
		     int position,
		     int DeleteIt);


/**************************************************************
 *
 * ListIterate
 *
 * PURPOSE : Perform the specified function on each of the node's
 *           data areas.
 *
 * PARAMS  : TheList - list to iterate over
 *           Action  - function to perform on the item.
 *
*/
extern int ListIterate( List * TheList,
		 int (* Action) ());

/**************************************************************
 *
 *
 * ListReset
 *
 * PURPOSE : Return the First element of the list, reset the
 *           list cursor to point to the second element.
 *
 * PARAMS  : TheList - list to look at
 *
 * RETURNS : pointer to data area of the first item in the list
 *
*/
extern void * ListReset ( List * TheList );

/**************************************************************
 *
 *
 * ListNext
 *
 * PURPOSE : Return the next item in the list.
 *
 * PARAMS  : TheList - the list to look at
 *
 * RETURNS : the next item in the list, the first item if it hasn't been
 *           called before
 *
*/
extern void * ListNext ( List * TheList );


/**************************************************************
 *
 * ListSort
 *
 * PURPOSE : To sort the list with a specified comparison function.
 *
 * PARAMS  : TheList
 *           CompareFcn - function to compare two Data items
 *                      if D1 < D2, returns -1
 *                      if D1 = D2, returns 0
 *                      if D1 > D2, returns 1
*/
extern int ListSortAsc( List * TheList,
	       int (* CompareFcn) ());

/**************************************************************
 *
 * ListSize
 *
 * PURPOSE : Return the number of data items in the list.
 *
 * PARAMS  : TheList
 *
*/
extern int ListSize ( List * TheList );


#endif
