/*
 *	MP3 quantization
 *
 *	Copyright (c) 1999 Mark Taylor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <assert.h>
#include "globalflags.h"
#include "util.h"
#include "l3side.h"
#include "quantize.h"
#include "l3bitstream.h"
#include "reservoir.h"
#include "quantize-pvt.h"
#include "gtkanal.h"

#define PRECALC_SIZE 8206 /* 8191+15. should never be outside this. see count_bits() */

static double masking_lower=1;


static int convert_mdct, convert_psy, reduce_sidechannel;
/*
mt 5/99.  These global flags denote 4 possibilities:
                                                                mode    l3_xmin
1   MDCT input L/R, quantize L/R,   psy-model thresholds: L/R   -m s     either
2   MDCT input L/R, quantize M/S,   psy-model thresholds: L/R   -m j     orig
3   MDCT input M/S, quantize M/S,   psy-model thresholds: M/S   -m f     either
4   MDCT input L/R, quantize M/S,   psy-model thresholds: M/S   -m j -h  m/s

1:  convert_mdct = 0, convert_psy=0,  reduce_sidechannel=0          
2:  convert_mdct = 1, convert_psy=1,  reduce_sidechannel=1
3:  convert_mdct = 0, convert_psy=0,  reduce_sidechannel=1
4:  convert_mdct = 1, convert_psy=0,  reduce_sidechannel=1

if (convert_mdct), then iteration_loop will quantize M/S data from
the L/R input MDCT coefficients.

if (convert_psy), then calc_noise will compute the noise for the L/R
channels from M/S MDCT data and L/R psy-model threshold information.
Distortion in ether L or R channel will be marked as distortion in
both Mid and Side channels.  

if (reduce_sidechannel) then outer_loop will allocate less bits
to the side channel and more bits to the mid channel based on relative 
energies.
*/




/*
  Here are MPEG1 Table B.8 and MPEG2 Table B.1
  -- Layer III scalefactor bands.
  Index into this using a method such as:
    idx  = fr_ps->header->sampling_frequency
           + (fr_ps->header->version * 3)
*/

struct scalefac_struct sfBandIndex[6] =
{

  { /* Table B.2.b: 22.05 kHz */
    {0,6,12,18,24,30,36,44,54,66,80,96,116,140,168,200,238,284,336,396,464,522,576},
    {0,4,8,12,18,24,32,42,56,74,100,132,174,192}
  },
  { /* Table B.2.c: 24 kHz */
    {0,6,12,18,24,30,36,44,54,66,80,96,114,136,162,194,232,278,332,394,464,540,576},
    {0,4,8,12,18,26,36,48,62,80,104,136,180,192}
  },
  { /* Table B.2.a: 16 kHz */
    {0,6,12,18,24,30,36,44,54,66,80,96,116,140,168,200,238,284,336,396,464,522,576},
    {0,4,8,12,18,26,36,48,62,80,104,134,174,192}
  },
  { /* Table B.8.b: 44.1 kHz */
    {0,4,8,12,16,20,24,30,36,44,52,62,74,90,110,134,162,196,238,288,342,418,576},
    {0,4,8,12,16,22,30,40,52,66,84,106,136,192}
  },
  { /* Table B.8.c: 48 kHz */
    {0,4,8,12,16,20,24,30,36,42,50,60,72,88,106,128,156,190,230,276,330,384,576},
    {0,4,8,12,16,22,28,38,50,64,80,100,126,192}
  },
  { /* Table B.8.a: 32 kHz */
    {0,4,8,12,16,20,24,30,36,44,54,66,82,102,126,156,194,240,296,364,448,550,576},
    {0,4,8,12,16,22,30,42,58,78,104,138,180,192}
  }
};

/*
  The following table is used to implement the scalefactor
  partitioning for MPEG2 as described in section
  2.4.3.2 of the IS. The indexing corresponds to the
  way the tables are presented in the IS:

  [table_number][row_in_table][column of nr_of_sfb]
*/
static unsigned nr_of_sfb_block[6][3][4] =
{
  {
    {6, 5, 5, 5},
    {9, 9, 9, 9},
    {6, 9, 9, 9}
  },
  {
    {6, 5, 7, 3},
    {9, 9, 12, 6},
    {6, 9, 12, 6}
  },
  {
    {11, 10, 0, 0},
    {18, 18, 0, 0},
    {15,18,0,0}
  },
  {
    {7, 7, 7, 0},
    {12, 12, 12, 0},
    {6, 15, 12, 0}
  },
  {
    {6, 6, 6, 3},
    {12, 9, 9, 6},
    {6, 12, 9, 6}
  },
  {
    {8, 8, 5, 0},
    {15,12,9,0},
    {6,18,9,0}
  }
};

/*
  table of largest scalefactors for MPEG2
*/
static unsigned max_sfac_tab[6][4] =
{
    {4, 4, 3, 3},
    {4, 4, 3, 0},
    {3, 2, 0, 0},
    {4, 5, 5, 0},
    {3, 3, 3, 0},
    {2, 2, 0, 0}
};

/* Table B.6: layer3 preemphasis */
int  pretab[21] =
{
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    1, 1, 1, 1, 2, 2, 3, 3, 3, 2
};

/* This is the scfsi_band table from 2.4.2.7 of the IS */
int scfsi_band_long[5] = { 0, 6, 11, 16, 21 };

int *scalefac_band_long  = &sfBandIndex[3].l[0];
int *scalefac_band_short = &sfBandIndex[3].s[0];



/* 
compute the ATH for each scalefactor band 
cd range:  0..96db

Input:  3.3kHz signal  32767 amplitude  (3.3kHz is where ATH is smallest = -5db)
longblocks:  sfb=12   en0/bw=-11db    max_en0 = 1.3db
shortblocks: sfb=5           -9db              0db

Input:  1 1 1 1 1 1 1 -1 -1 -1 -1 -1 -1 -1 (repeated)
longblocks:  amp=1      sfb=12   en0/bw=-103 db      max_en0 = -92db
            amp=32767   sfb=12           -12 db                 -1.4db 

Input:  1 1 1 1 1 1 1 -1 -1 -1 -1 -1 -1 -1 (repeated)
shortblocks: amp=1      sfb=5   en0/bw= -99                    -86 
            amp=32767   sfb=5           -9  db                  4db 


MAX energy of largest wave at 3.3kHz = 1db
AVE energy of largest wave at 3.3kHz = -11db
Let's take AVE:  -11db = maximum signal in sfb=12.  
Dynamic range of CD: 96db.  Therefor energy of smallest audible wave 
in sfb=12  = -11  - 96 = -107db = ATH at 3.3kHz.  

ATH formula for this wave: -5db.  To adjust to LAME scaling, we need
ATH = ATH_formula  - 103  (db)
ATH = ATH * 2.5e-10      (ener)

*/
static double ATH_l[CBLIMIT];
static double ATH_s[CBLIMIT];

double ATHformula(double f)
{
  double ath;
  f  = Max(0.02, f);
  /* from Painter & Spanias, 1997 */
  /* minimum: (i=77) 3.3kHz = -5db */
  ath=(3.640 * pow(f,-0.8)
       -  6.500 * exp(-0.6*pow(f-3.3,2.0))
       +  0.001 * pow(f,4.0));
  
  /* convert to energy */
  ath -= 114;    /* MDCT scaling.  From tests by macik and MUS420 code */
  //ath -= 200; /* disabled */
  ath = pow( 10.0, ath/10.0 );
  return ath;
}
 

void compute_ath(layer *info,double ATH_l[CBLIMIT],double ATH_s[CBLIMIT])
{
  int sfb,i,start,end;
  double ATH_f;
  double samp_freq = s_freq[info->version][info->sampling_frequency];

  /* last sfb is not used */
  for ( sfb = 0; sfb < SFB_LMAX-1; sfb++ ) {
    start = scalefac_band_long[ sfb ];
    end   = scalefac_band_long[ sfb+1 ];
    ATH_l[sfb]=1e99;
    for (i=start ; i < end; i++) {
      ATH_f = ATHformula(samp_freq*i/(2*576)); /* freq in kHz */
      ATH_l[sfb]=Min(ATH_l[sfb],ATH_f);
    }
    /*
    printf("sfb=%i %f  ATH=%f %f  %f   \n",sfb,samp_freq*start/(2*576),
10*log10(ATH_l[sfb]),
10*log10( ATHformula(samp_freq*start/(2*576)))  ,
10*log10(ATHformula(samp_freq*end/(2*576))));
    */
  }

  for ( sfb = 0; sfb < SFB_SMAX - 1; sfb++ ){
    start = scalefac_band_short[ sfb ];
    end   = scalefac_band_short[ sfb+1 ];
    ATH_s[sfb]=1e99;
    for (i=start ; i < end; i++) {
      ATH_f = ATHformula(samp_freq*i/(2*192));     /* freq in kHz */
      ATH_s[sfb]=Min(ATH_s[sfb],ATH_f);
    }
  }
}





/* convert from L/R <-> Mid/Side */
void ms_convert(double xr[2][576],double xr_org[2][576])
{
  int i;
  for ( i = 0; i < 576; i++ ) {
    xr[0][i] = (xr_org[0][i]+xr_org[1][i])/SQRT2;
    xr[1][i] = (xr_org[0][i]-xr_org[1][i])/SQRT2;
  }
}



/************************************************************************
 * allocate bits among 2 channels based on PE
 * mt 6/99
 ************************************************************************/
void on_pe(double pe[2][2],III_side_info_t *l3_side,
int targ_bits[2],int mean_bits,int stereo, int gr)
{
  gr_info *cod_info;
  int extra_bits,tbits,bits;
  int add_bits[2]; 
  int ch;

  /* allocate targ_bits for granule */
  ResvMaxBits2( mean_bits, &tbits, &extra_bits, gr);
    

  for (ch=0 ; ch < stereo ; ch ++) {
    /******************************************************************
     * allocate bits for each channel 
     ******************************************************************/
    cod_info = &l3_side->gr[gr].ch[ch].tt;
    
    targ_bits[ch]=tbits/stereo;
    
    // allocate extra bits from reservoir based on PE 
    bits=0;
    
    /* extra bits based on PE > 700 */
    add_bits[ch]=(pe[gr][ch]-750)/1.4;  /* 2.0; */
    
    /* short blocks need extra, no matter what the pe */
    if (cod_info->block_type==2) 
      if (add_bits[ch]<500) add_bits[ch]=500;
    
    if (add_bits[ch] < 0) add_bits[ch]=0;
    bits += add_bits[ch];
    
    if (bits > extra_bits) add_bits[ch] = (extra_bits*add_bits[ch])/bits;
    if ((targ_bits[ch]+add_bits[ch]) > 4095) 
      add_bits[ch]=4095-targ_bits[ch];

    targ_bits[ch] = targ_bits[ch] + add_bits[ch];
    extra_bits -= add_bits[ch];
  }
}

void reduce_side(int targ_bits[2],double ms_ener_ratio,int mean_bits)
{
int ch;
int stereo=2;
    /*  ms_ener_ratio = 0:  allocate 66/33  mid/side  fac=.33  
     *  ms_ener_ratio =.5:  allocate 50/50 mid/side   fac= 0 */
    /* 75/25 split is fac=.5 */
    /* float fac = .50*(.5-ms_ener_ratio[gr])/.5;*/
    float fac = .33*(.5-ms_ener_ratio)/.5;
    if (fac<0) fac=0;
    
    if (targ_bits[1] >= 125) {
      /* dont reduce side channel below 125 bits */
      if (targ_bits[1]-targ_bits[1]*fac > 125) {
	targ_bits[0] += targ_bits[1]*fac;
	targ_bits[1] -= targ_bits[1]*fac;
      } else {
	targ_bits[0] += targ_bits[1] - 125;
	targ_bits[1] = 125;
      }
    }
    
    /* dont allow to many bits per channel */  
    for (ch=0; ch<stereo; ch++) {
      int max_bits = Min(4095,mean_bits/2 + 1200);
      if (targ_bits[ch] > max_bits) {
	targ_bits[ch] = max_bits;
      }
    }

}




/************************************************************************/
/*  initialization for iteration_loop */
/************************************************************************/
void
iteration_init( double xr_org[2][2][576], 
		III_side_info_t *l3_side, int l3_enc[2][2][576],
		frame_params *fr_ps, III_psy_xmin *l3_xmin)
{
  gr_info *cod_info;
  layer *info  = fr_ps->header;
  int stereo = fr_ps->stereo;
  int ch, gr, i, mode_gr;
  //  int bit_rate,samp_freq;


  if ( frameNum==0 ) {
    l3_side->main_data_begin = 0;
    memset((char *) &l3_xmin, 0, sizeof(l3_xmin));
    compute_ath(info,ATH_l,ATH_s);
  }

  l3_side->resvDrain = 0;
  mode_gr = (info->version == 1) ? 2 : 1;
  //bit_rate = bitrate[info->version][info->lay-1][info->bitrate_index];
  //samp_freq = s_freq[info->version][info->sampling_frequency]*1000 +.5;

  scalefac_band_long  = &sfBandIndex[info->sampling_frequency + (info->version * 3)].l[0];
  scalefac_band_short = &sfBandIndex[info->sampling_frequency + (info->version * 3)].s[0];
  

  convert_mdct=0;
  convert_psy=0;
  reduce_sidechannel=0;
  if (info->mode_ext==MPG_MD_MS_LR) {
    if (highq) {
      convert_mdct = 1;
      convert_psy = 0;
      reduce_sidechannel=1;
    }else{
      convert_mdct = 1;
      convert_psy = 1;
      reduce_sidechannel=1;
    }
  }
  if (force_ms) {
    convert_mdct = 0;
    convert_psy = 0;
    reduce_sidechannel=1;
  }
  if (convert_psy) memset(l3_enc,0,sizeof(int)*2*2*576);


  /* voice mode disables short blocks */  
  if (voice_mode) {
    for ( gr = 0; gr < mode_gr; gr++ ) {
      for (ch =0 ; ch < stereo ; ch++) {
		int start = scalefac_band_long[ SFB_LMAX-3 ];
		  for ( i = start; i < 576; i++ )
			  xr_org[gr][ch][i]=0;
      }
      for (ch =0 ; ch < stereo ; ch++) {
		for ( i = 0; i < 2; i++ )
			  xr_org[gr][ch][i]=0;
      }
    }
  }

  if (sfb21) {
    for ( gr = 0; gr < mode_gr; gr++ ) {
      for (ch =0 ; ch < stereo ; ch++) {
	int shortblock = (l3_side->gr[gr].ch[0].tt.block_type==2);
	if (shortblock) {
	  int j;
	  for (j=0; j<3; j++) {
	    int start = scalefac_band_short[ SFB_SMAX-1 ];
	    for ( i = start; i < 192; i++ ) {
	      int i0 = 3*i+j; 
	      xr_org[gr][ch][i0]=0;
	    }
	  }
	}else{
	  int start = scalefac_band_long[ SFB_LMAX-1 ];
	  for ( i = start; i < 576; i++ ) xr_org[gr][ch][i]=0;
	}
      }
    }
  }
  
  
  
  /* some intializations. */
  for ( gr = 0; gr < mode_gr; gr++ ){
    for ( ch = 0; ch < stereo; ch++ ){
      cod_info = (gr_info *) &(l3_side->gr[gr].ch[ch]);
      gr_deco(cod_info);
    }
  }


  /* dont bother with scfsi. */
  for ( ch = 0; ch < stereo; ch++ )
    for ( i = 0; i < 4; i++ )
      l3_side->scfsi[ch][i] = 0;


  

}




/************************************************************************/
/*  iteration_loop()                                                    */
/************************************************************************/
void
iteration_loop( double pe[2][2], double ms_ener_ratio[2],
		double xr_org[2][2][576], III_psy_ratio *ratio,
		III_side_info_t *l3_side, int l3_enc[2][2][576],
		III_scalefac_t *scalefac, frame_params *fr_ps)
{
  III_psy_xmin l3_xmin;
  gr_info *cod_info;
  layer *info;
  int VBRbits[2][2];
  int over[2];
  double noise[4]; /* over,max_noise,over_noise,tot_noise; */
  double targ_noise[4]; /* over,max_noise,over_noise,tot_noise; */
  int bitsPerFrame;
  int mean_bits;
  int stereo = fr_ps->stereo;
  int ch, gr, i, mode_gr, bit_rate;
  double xr[2][2][576];


  iteration_init(xr_org,l3_side,l3_enc,fr_ps,&l3_xmin);
  info = fr_ps->header;
  mode_gr = (info->version == 1) ? 2 : 1;
  bit_rate = bitrate[info->version][info->lay-1][info->bitrate_index];


  getframebits(info,stereo,&bitsPerFrame,&mean_bits);
  ResvFrameBegin( fr_ps, l3_side, mean_bits, bitsPerFrame );

  /* quantize! */

  for ( gr = 0; gr < mode_gr; gr++ ) {
    if (convert_psy) {
      /* dual channel version can quantize Mid/Side channels with L/R
       * maskings (by constantly reconstructing L/R data).  Used before we
       * we had proper mid/side maskings. */
      outer_loop_dual( xr, xr_org, mean_bits, VBRbits, bit_rate, over,
	       &l3_xmin,l3_enc, fr_ps, 
	       scalefac,gr,stereo, l3_side, ratio, pe, ms_ener_ratio);
    } else {
      int targ_bits[2];
      /* copy data to be quantized into xr */
      if (convert_mdct) ms_convert(xr[gr],xr_org[gr]);
      else memcpy(xr[gr],xr_org[gr],sizeof(double)*2*576);   
      
      for (ch=0 ; ch < stereo ; ch ++) targ_bits[ch]=VBRbits[gr][ch];
      on_pe(pe,l3_side,targ_bits,mean_bits,stereo,gr);
      if (reduce_sidechannel) 
	reduce_side(targ_bits,ms_ener_ratio[gr],mean_bits);
      
      for (ch=0 ; ch < stereo ; ch ++) {
	outer_loop( xr, targ_bits[ch], noise, targ_noise, 0, &l3_xmin,l3_enc, 
	    fr_ps, scalefac,gr,stereo, l3_side, ratio, ms_ener_ratio[gr],ch);
	cod_info = &l3_side->gr[gr].ch[ch].tt;
	ResvAdjust( fr_ps, cod_info, l3_side, mean_bits );
      }
    }
  }

  
  /* set the sign of l3_enc */
  for ( gr = 0; gr < mode_gr; gr++ ) {
    for ( ch =  0; ch < stereo; ch++ ) {
      int *pi = &l3_enc[gr][ch][0];
      for ( i = 0; i < 576; i++) 	    {
	double pr = xr[gr][ch][i];
	if ( (pr < 0) && (pi[i] > 0) )   pi[i] *= -1;
      }
    }
  }
  ResvFrameEnd( fr_ps, l3_side, mean_bits );
}

      
/************************************************************************
 *
 * VBR_iteration_loop()   
 *
 * tries to find out how many bits are needed for each granule and channel
 * to get an acceptable quantization. An appropriate bitrate will then be
 * choosed for quantization.  rh 8/99                                                
 *
 ************************************************************************/
void
VBR_iteration_loop (double pe[2][2], double ms_ener_ratio[2],
                double xr_org[2][2][576], III_psy_ratio * ratio,
                III_side_info_t * l3_side, int l3_enc[2][2][576],
                III_scalefac_t * scalefac, frame_params * fr_ps)
{
  III_psy_xmin l3_xmin;
  gr_info  *cod_info = NULL;
  layer    *info;
  int       save_bits[2][2];
  double    noise[4];      /* over,max_noise,over_noise,tot_noise; */
  double    targ_noise[4]; /* over,max_noise,over_noise,tot_noise; */
  int       this_bits, dbits;
  int       used_bits=0;
  int       min_bits,max_bits,min_mean_bits=0;
  int       frameBits[15];
  int       bitsPerFrame;
  int       bits;
  int       mean_bits;
  int       stereo = fr_ps->stereo;
  int       i,ch, gr, mode_gr;
  double    xr[2][2][576];
  double xr_save[576];
  double masking_lower_db;

  iteration_init(xr_org,l3_side,l3_enc,fr_ps,&l3_xmin);
  info = fr_ps->header;
  mode_gr = (info->version == 1) ? 2 : 1;


  /*******************************************************************
   * how many bits are available for each bitrate?
   *******************************************************************/
  for( info->bitrate_index = VBR_min_bitrate;
       info->bitrate_index <= VBR_max_bitrate;
       info->bitrate_index++    ) {
    getframebits (info, stereo, &bitsPerFrame, &mean_bits);
    if (info->bitrate_index == VBR_min_bitrate) {
      /* always use at least this many bits per granule per channel */
      min_mean_bits=mean_bits/stereo;
    }
    frameBits[info->bitrate_index]=
      ResvFrameBegin (fr_ps, l3_side, mean_bits, bitsPerFrame);
    //printf("framebits = %i \n",frameBits[info->bitrate_index]);
  }

  info->bitrate_index=VBR_max_bitrate;

  
  /*******************************************************************
   * how many bits would we use of it?
   *******************************************************************/
  for (gr = 0; gr < mode_gr; gr++) {
    int num_chan=stereo;
    /* determine quality based on mid channel only */
    if (reduce_sidechannel) num_chan=1;  

    /* copy data to be quantized into xr */
    if (convert_mdct) ms_convert(xr[gr],xr_org[gr]);
    else memcpy(xr[gr],xr_org[gr],sizeof(double)*2*576);   

    



    for (ch = 0; ch < num_chan; ch++) { 
      /******************************************************************
       * find smallest number of bits for an allowable quantization
       ******************************************************************/
      memcpy(xr_save,xr[gr][ch],sizeof(double)*576);   
      cod_info = &l3_side->gr[gr].ch[ch].tt;
      min_bits = Max(125,min_mean_bits);
      if (cod_info->block_type ==2) {
	min_bits += Max(1100,pe[gr][ch]);
	min_bits=Min(min_bits,1800);
      }

      max_bits = 1200 + frameBits[VBR_max_bitrate]/(stereo*mode_gr);
      max_bits=Min(max_bits,2500);
      max_bits=Max(max_bits,min_bits);


      /** in the case we will not find any better, we allocate max_bits
       ****************************************************************/
      save_bits[gr][ch] = max_bits;

      dbits = (max_bits-min_bits)/4;
      this_bits = (max_bits+min_bits)/2;  
      /* bin search to within +/- 10 bits of optimal */
      do {
	int better;
	double fac;
	assert(this_bits>=min_bits);
	assert(this_bits<=max_bits);

	/* quality setting */
	/* Adjust allowed masking based on quality setting */
	/* db_lower varies from -10 to +8 db */
	masking_lower_db = -10 + 2*VBR_q;
	/* adjust by -6(min)..0(max) depending on bitrate */
	fac=(double)(this_bits-125)/(double)(2500-125);
	fac = 6*(fac-1);
	//fac = 3*(fac-1);

	masking_lower_db += fac;
	masking_lower = pow(10.0,masking_lower_db/10);


	

	/* VBR will look for a quantization which has better values
	 * then those specified below.*/
	targ_noise[0]=0;          /* over */
	targ_noise[1]=0;          /* max_noise */
	targ_noise[2]=0;          /* over_noise */
	targ_noise[3]=-2;          /* tot_noise */
	
	targ_noise[0]=Max(0,targ_noise[0]);
	targ_noise[2]=Max(0,targ_noise[2]);


	/* restore xr */
	memcpy(xr[gr][ch],xr_save,sizeof(double)*576);   
	outer_loop( xr, this_bits, noise, targ_noise, 1,&l3_xmin,l3_enc, fr_ps, 
	    scalefac,gr,stereo, l3_side, ratio, ms_ener_ratio[gr], ch);

	/* is quantization as good as we are looking for ? */
	better=VBR_compare((int)targ_noise[0],targ_noise[3],targ_noise[2],
            targ_noise[1],(int)noise[0],noise[3],noise[2],noise[1]);

       if (better) {
	  save_bits[gr][ch] = this_bits;
          this_bits -= dbits;
        } else {
          this_bits += dbits;
        }
        dbits /= 2;
      } while (dbits>10) ;
      used_bits += save_bits[gr][ch];
      
    } /* for */
  } /* for */


  if (reduce_sidechannel) {
    /* number of bits needed was found for MID channel above.  Use formula
     * (fixed bitrate code) to set the side channel bits */
    for (gr = 0; gr < mode_gr; gr++) {
      double fac = .33*(.5-ms_ener_ratio[gr])/.5;
      save_bits[gr][1]=((1-fac)/(1+fac))*save_bits[gr][0];
      used_bits += save_bits[gr][1];
    }
  }

  
  /******************************************************************
   * find lowest bitrate able to hold used bits
   ******************************************************************/
  for( info->bitrate_index = VBR_min_bitrate;
       info->bitrate_index < VBR_max_bitrate;
       info->bitrate_index++    )
    if( used_bits <= frameBits[info->bitrate_index] ) break;


  /*******************************************************************
   * calculate quantization for this bitrate
   *******************************************************************/  
  getframebits (info, stereo, &bitsPerFrame, &mean_bits);
  bits=ResvFrameBegin (fr_ps, l3_side, mean_bits, bitsPerFrame);

  /* repartion available bits in same proportion */
  if (used_bits > bits ) {
    for( gr = 0; gr < mode_gr; gr++) {
      for(ch = 0; ch < stereo; ch++) {
	save_bits[gr][ch]=(save_bits[gr][ch]*frameBits[info->bitrate_index])/used_bits;
      }
    }
    used_bits=0;
    for( gr = 0; gr < mode_gr; gr++) {
      for(ch = 0; ch < stereo; ch++) {
	used_bits += save_bits[gr][ch];
      }
    }
  }
  assert(used_bits <= bits);

  for(gr = 0; gr < mode_gr; gr++) {
    /* copy data to be quantized into xr */
    if (convert_mdct) ms_convert(xr[gr],xr_org[gr]);
    else memcpy(xr[gr],xr_org[gr],sizeof(double)*2*576);   
    for(ch = 0; ch < stereo; ch++) {
      outer_loop( xr, save_bits[gr][ch], noise,targ_noise,0,
		  &l3_xmin,l3_enc, fr_ps, 
		  scalefac,gr,stereo, l3_side, ratio, ms_ener_ratio[gr], ch);
    }
  }

  /*******************************************************************
   * update reservoir status after FINAL quantization/bitrate 
   *******************************************************************/
  for (gr = 0; gr < mode_gr; gr++)
    for (ch = 0; ch < stereo; ch++) {
      cod_info = &l3_side->gr[gr].ch[ch].tt;
      ResvAdjust (fr_ps, cod_info, l3_side, mean_bits);
    }

  /*******************************************************************
   * set the sign of l3_enc 
   *******************************************************************/
  for (gr = 0; gr < mode_gr; gr++)
    for (ch = 0; ch < stereo; ch++) {
      int      *pi = &l3_enc[gr][ch][0];

      for (i = 0; i < 576; i++) {
        double    pr = xr[gr][ch][i];

        if ((pr < 0) && (pi[i] > 0))
          pi[i] *= -1;
      }
    }

  ResvFrameEnd (fr_ps, l3_side, mean_bits);
}




/************************************************************************/
/*  init_outer_loop  mt 6/99                                            */
/************************************************************************/
void init_outer_loop(
    double xr[2][2][576],        /*  could be L/R OR MID/SIDE */
    III_psy_xmin  *l3_xmin,   /* the allowed distortion of the scalefactor */
    III_scalefac_t *scalefac, /* scalefactors */
    int gr, int stereo, III_side_info_t *l3_side,
    III_psy_ratio *ratio, int ch)
{
  int sfb,i;
  gr_info *cod_info;  
  cod_info = &l3_side->gr[gr].ch[ch].tt;

  /* compute max allowed distortion */
  calc_xmin( xr, ratio, cod_info, l3_xmin, gr, ch );

  /* if ( info->version == 1 )
     calc_scfsi( xr[gr][ch], l3_side, &l3_xmin, ch, gr ); 
  */
  
    
  /* reset of iteration variables */
    
  for ( sfb = 0; sfb < SFB_LMAX; sfb++ )
    scalefac->l[gr][ch][sfb] = 0;
  for ( sfb = 0; sfb < SFB_SMAX; sfb++ )
    for ( i = 0; i < 3; i++ )
      scalefac->s[gr][ch][sfb][i] = 0;
  
  for ( i = 0; i < 4; i++ )
    cod_info->slen[i] = 0;
  cod_info->sfb_partition_table = &nr_of_sfb_block[0][0][0];
  
  cod_info->part2_3_length    = 0;
  cod_info->big_values        = ((cod_info->block_type==2)?288:0);
  cod_info->count1            = 0;
  cod_info->scalefac_compress = 0;
  cod_info->table_select[0]   = 0;
  cod_info->table_select[1]   = 0;
  cod_info->table_select[2]   = 0;
  cod_info->subblock_gain[0]  = 0;
  cod_info->subblock_gain[1]  = 0;
  cod_info->subblock_gain[2]  = 0;
  cod_info->region0_count     = 0;
  cod_info->region1_count     = 0;
  cod_info->part2_length      = 0;
  cod_info->preflag           = 0;
  cod_info->scalefac_scale    = 0;
  cod_info->quantizerStepSize = 0.0;
  cod_info->count1table_select= 0;
  cod_info->address1          = 0;
  cod_info->address2          = 0;
  cod_info->address3          = 0;
  
  
  if (experimentalZ) {
    /* compute subblock gains */
    int j,b;  double en[3],mx;
    if ((cod_info->block_type ==2) ) {
      /* estimate energy within each subblock */
      for (b=0; b<3; b++) en[b]=0;
      for ( i=0,j = 0; j < 192; j++ ) {
	for (b=0; b<3; b++) {
	  en[b]+=xr[gr][ch][i]*xr[gr][ch][i];
	  i++;
	}
      }
      mx = 1e-12;
      for (b=0; b<3; b++) mx=Max(mx,en[b]);
      for (b=0; b<3; b++) en[b] = Max(en[b],1e-12)/mx;
      //	printf("ener = %4.2f  %4.2f  %4.2f  \n",en[0],en[1],en[2]);
      /* pick gain so that 2^(2gain)*en[0] = 1  */
      /* gain = .5* log( 1/en[0] )/log(2) = -.5*log(en[])/log(2) */
      for (b=0; b<3; b++) {
	cod_info->subblock_gain[b]=nint2(-.5*log(en[b])/log(2.0));
	if (cod_info->subblock_gain[b] > 2) 
	  cod_info->subblock_gain[b]=2;
	if (cod_info->subblock_gain[b] < 0) 
	  cod_info->subblock_gain[b]=0;
      }
    }
  }
}


/* old outer_loop routines which will eventually be removed are in: */
#include "loopold.c"    


/************************************************************************/
/*  outer_loop                                                         */
/************************************************************************/
/*  Function: The outer iteration loop controls the masking conditions  */
/*  of all scalefactorbands. It computes the best scalefac and          */
/*  global gain. This module calls the inner iteration loop             
 * 
 *  mt 5/99 completely rewritten to allow for bit reservoir control,   
 *  mid/side channels with L/R or mid/side masking thresholds, 
 *  and chooses best quantization instead of last quantization when 
 *  no distortion free quantization can be found.  
 *  
 *  added VBR support mt 5/99
 ************************************************************************/
void outer_loop(
    double xr[2][2][576],        /*  could be L/R OR MID/SIDE */
    int targ_bits,
    double best_noise[4],
    double targ_noise[4],
    int sloppy,
    III_psy_xmin  *l3_xmin,   /* the allowed distortion of the scalefactor */
    int l3_enc[2][2][576],    /* vector of quantized values ix(0..575) */
    frame_params *fr_ps,
    III_scalefac_t *scalefac, /* scalefactors */
    int gr, int stereo, III_side_info_t *l3_side,
    III_psy_ratio *ratio, double ms_ener_ratio,
    int ch)
{

  int i, iteration;
  int status,count=0,bits_found=0;
  int real_bits=0;
  int scalesave_l[CBLIMIT], scalesave_s[CBLIMIT][3];
  int sfb, huff_bits;
  double xfsf[4][CBLIMIT];
  double xrpow[576],temp;
  double distort[4][CBLIMIT];
  int save_l3_enc[576];  
  int save_real_bits=0;
  int save_preflag=0, save_compress=0;
  int better;
  int over=0;
  double max_noise;
  double over_noise;
  double tot_noise;
  int best_over=0;
  double best_max_noise=0;
  double best_over_noise=0;
  double best_tot_noise=0;
  gr_info save_cod_info;
  gr_info *cod_info;  
  double xr_save[576];

  int compute_stepsize=1;
  int pre_just_turned_on;
  int notdone=1;


  

  if (experimentalY) memcpy(xr_save,xr[gr][ch],sizeof(double)*576);   
  cod_info = &l3_side->gr[gr].ch[ch].tt;
  init_outer_loop(xr,l3_xmin,scalefac,gr,stereo,l3_side,ratio,ch);  
  best_over = 100;
  count=0;
  for (i=0; i<576; i++) {
    if ( fabs(xr[gr][ch][i]) > 0 ) count++; 
  }
  if (count==0) {
    best_over=0;
    notdone = 0;
  }
  

  
  /* BEGIN MAIN LOOP */
  iteration = 0;
  while ( notdone  ) {
    int try_scale=0;
    iteration ++;
    
    if (compute_stepsize) {
      /* init and compute initial quantization step */
      compute_stepsize=0;
      for(i=0;i<576;i++) 	    {
	temp=fabs(xr[gr][ch][i]);
	xrpow[i]=sqrt(sqrt(temp)*temp);
      }
      bits_found=bin_search_StepSize2(targ_bits,-211.0,46,
		  l3_enc[gr][ch],xr[gr][ch],xrpow,cod_info); 
    }
    
    
    
    /* inner_loop starts with the initial quantization step computed above
     * and slowly increases until the bits < huff_bits.
     * Thus is it important not to start with too large of an inital
     * quantization step.  Too small is ok, but inner_loop will take longer 
     */
    huff_bits = targ_bits - cod_info->part2_length;
    if (huff_bits < 0) {
      if (iteration==1) {
	fprintf(stderr,"ERROR: outer_loop(): huff_bits < 0. \n");
	exit(-5);
      }else{
	/* scale factors too large, not enough bits. use previous quantizaton */
	notdone=0;
      }
    }else{
      /* if this is the first iteration, see if we can reuse the quantization
       * computed in bin_search_StepSize above */
      if (iteration==1) {
	if(bits_found>huff_bits) {
	  cod_info->quantizerStepSize+=1.0;
	  real_bits = inner_loop( xr, xrpow, l3_enc, huff_bits, cod_info, gr, ch );
	} else real_bits=bits_found;
      }
      else 
	real_bits=inner_loop( xr, xrpow, l3_enc, huff_bits, cod_info, gr, ch );
    }
    
    
    if (notdone) {
      /* compute the distortion in this quantization */
      if (fast_mode) {
      	over=0;
      }else{
	/* coefficients and thresholds both l/r (or both mid/side) */
	over=calc_noise1( xr[gr][ch], l3_enc[gr][ch], cod_info, 
			  xfsf,distort, l3_xmin,gr,ch, &over_noise, 
			  &tot_noise, &max_noise);
      }
            
      /* check if this quantization is better the our saved quantization */
      if (iteration == 1) better=1;
      else 
	better=quant_compare(
	     best_over,best_tot_noise,best_over_noise,best_max_noise,
                  over,     tot_noise,     over_noise,     max_noise);
      
      
      /* save data so we can restore this quantization later */    
      if (better) {
	best_over=over;
	best_max_noise=max_noise;
	best_over_noise=over_noise;
	best_tot_noise=tot_noise;
	
        /* we need not save the quantization when in sloppy mode, so skip
         */ 
        if (!sloppy) {
	for ( sfb = 0; sfb < SFB_LMAX-1; sfb++ ) /* save scaling factors */
	  scalesave_l[sfb] = scalefac->l[gr][ch][sfb];
	
	for ( sfb = 0; sfb < SFB_SMAX-1; sfb++ )
	  for ( i = 0; i < 3; i++ )
	    scalesave_s[sfb][i] = scalefac->s[gr][ch][sfb][i];
	
	save_preflag  = cod_info->preflag;
	save_compress = cod_info->scalefac_compress;
	
	memcpy(save_l3_enc,l3_enc[gr][ch],sizeof(l3_enc[gr][ch]));   
	memcpy(&save_cod_info,cod_info,sizeof(save_cod_info));
	save_real_bits=real_bits;
	
#ifdef HAVEGTK
	if (gtkflag) {
	  int l,start,end,bw;
	  double en0;
	  D192_3 *xr_s = (D192_3 *)xr[gr][ch];

	  for ( i = 0; i < 3; i++ ) {
	    for ( sfb = cod_info->sfb_smax; sfb < 12; sfb++ )  {
	      start = scalefac_band_short[ sfb ];
	      end   = scalefac_band_short[ sfb + 1 ];
	      bw = end - start;
	      for ( en0 = 0.0, l = start; l < end; l++ ) 
		en0 += (*xr_s)[l][i] * (*xr_s)[l][i];
	      en0=Max(en0/bw,1e-20);
	      /*
	      pinfo->xfsf_s[gr][ch][3*sfb+i] =  
		pinfo->thr_s[gr][ch][3*sfb+i]*xfsf[i+1][sfb]/
		l3_xmin->s[gr][ch][sfb][i];
	      */
	      pinfo->xfsf_s[gr][ch][3*sfb+i] =  
		pinfo->en_s[gr][ch][3*sfb+i]*xfsf[i+1][sfb]/en0;
	      pinfo->thr_s[gr][ch][3*sfb+i] =  
		pinfo->en_s[gr][ch][3*sfb+i]*l3_xmin->s[gr][ch][sfb][i]/en0;

	    }
	  }
	  for ( sfb = 0; sfb < cod_info->sfb_lmax; sfb++ )   {
	    start = scalefac_band_long[ sfb ];
	    end   = scalefac_band_long[ sfb+1 ];
	    bw = end - start;
	    for ( en0 = 0.0, l = start; l < end; l++ ) 
	      en0 += xr[gr][ch][l] * xr[gr][ch][l];
	    en0=Max(en0/bw,1e-20);
	    /* 
	    pinfo->xfsf[gr][ch][sfb] =  
	      pinfo->thr[gr][ch][sfb]*xfsf[0][sfb]/
	      l3_xmin->l[gr][ch][sfb];
	    */
	    pinfo->xfsf[gr][ch][sfb] =  
	      pinfo->en[gr][ch][sfb]*xfsf[0][sfb]/en0;
	    pinfo->thr[gr][ch][sfb] =  
	      pinfo->en[gr][ch][sfb]*l3_xmin->l[gr][ch][sfb]/en0;


	  }
	  pinfo->over[gr][ch]=over;
	  pinfo->max_noise[gr][ch]=max_noise;
	  pinfo->tot_noise[gr][ch]=tot_noise;
	  pinfo->over_noise[gr][ch]=over_noise;
	}
#endif
        } /* end of sloppy skipping */
      }
    }

    /* if no bands with distortion, we are done */
    if (experimentalX==0)
      if (over==0) notdone=0;

    /* in sloppy mode, as soon as we know we can do better than targ_noise,
       quit.  This is used for the inital VBR bin search.  Turn it off for
       final (optimal) quantization */
    if (sloppy && notdone) notdone = 
        !VBR_compare((int)targ_noise[0],targ_noise[3],targ_noise[2],
         targ_noise[1],over,tot_noise,over_noise,max_noise);



    
    
    if (notdone) {
      /* see if we should apply preemphasis */
      pre_just_turned_on=
	preemphasis(xr[gr][ch],xrpow,l3_xmin,gr,ch,l3_side,distort);

      /* if we didn't just apply pre-emph, let us see if we should 
       * amplify some scale factor bands */
      if (!pre_just_turned_on) {
	notdone = amp_scalefac_bands( xr[gr][ch], xrpow, l3_xmin,
			     l3_side, scalefac, gr, ch, iteration,distort);
      }
    }
      
      
    if (notdone) {            
      /* check to make sure we have not amplified too much */
      if ( (status = loop_break(scalefac, cod_info, gr, ch)) == 0 ) {
	if ( fr_ps->header->version == 1 ) {
	  status = scale_bitcount( scalefac, cod_info, gr, ch );
	  if (status && (cod_info->scalefac_scale==0)) try_scale=1; 
	}else{
	  status = scale_bitcount_lsf( scalefac, cod_info, gr, ch );
	  if (status && (cod_info->scalefac_scale==0)) try_scale=1; 
	}
      }
      notdone = !status;
    }    
    

    
    if (try_scale && experimentalY) {
      memcpy(xr[gr][ch],xr_save,sizeof(double)*576);   
      init_outer_loop(xr,l3_xmin,scalefac,gr,stereo,l3_side,ratio,ch);  
      compute_stepsize=1;  /* compute a new global gain */
      notdone=1;
      cod_info->scalefac_scale=1;
    }
  }    /* done with main iteration */
  
  
  /* in sloppy mode we dont need to restore the quantization, 
   * cos we didnt saved it and we dont need it, as we only want
   * to know if we can do better
   */
  if (!sloppy)
  /* restore some data */
  if (count ) {
    cod_info->preflag = save_preflag;
    cod_info->scalefac_compress = save_compress;
    
    for ( sfb = 0; sfb < SFB_LMAX-1; sfb++ ) {
      scalefac->l[gr][ch][sfb] = scalesave_l[sfb];    
    }
    
    for ( i = 0; i < 3; i++ )
      for ( sfb = 0; sfb < SFB_SMAX-1; sfb++ ) {
	scalefac->s[gr][ch][sfb][i] = scalesave_s[sfb][i];    
      }
    
    { 
      real_bits=save_real_bits;  
      memcpy(l3_enc[gr][ch],save_l3_enc,sizeof(l3_enc[gr][ch]));   
      memcpy(cod_info,&save_cod_info,sizeof(save_cod_info));
      
      if ( fr_ps->header->version == 1 ) {
	status = scale_bitcount( scalefac, cod_info, gr, ch );
      } else {
	status = scale_bitcount_lsf( scalefac, cod_info, gr, ch );
      }
      if (status) {
	fprintf(stderr,"Error recomputing scalefac_compress...this should not happen");
	exit(-10);
      }
    }
    cod_info->part2_3_length = cod_info->part2_length + real_bits;
    
#ifdef HAVEGTK
    if (gtkflag)
      pinfo->LAMEmainbits[gr][ch]=cod_info->part2_3_length;
#endif
  }      
  
  
  /* finish up */
  cod_info->global_gain = nint2( cod_info->quantizerStepSize + 210.0 );
  assert( cod_info->global_gain < 256 );

  best_noise[0]=best_over;
  best_noise[1]=best_max_noise;
  best_noise[2]=best_over_noise;
  best_noise[3]=best_tot_noise;
}





  





/*************************************************************************** 
 *         inner_loop                                                      * 
 *************************************************************************** 
 * The code selects the best quantizerStepSize for a particular set
 * of scalefacs                                                            */
 
int
inner_loop( double xr[2][2][576], double xrpow[576],
	    int l3_enc[2][2][576], int max_bits,
	    gr_info *cod_info, int gr, int ch )
{
    int bits;
    assert( max_bits >= 0 );
    cod_info->quantizerStepSize -= 1.0;;
    do
    {
      cod_info->quantizerStepSize += 1.0;
      quantize_xrpow( xrpow, l3_enc[gr][ch], cod_info );
      bits = count_bits(l3_enc[gr][ch],cod_info);  
    }
    while ( bits > max_bits );
    return bits;
}



/*************************************************************************/
/*            scale_bitcount                                             */
/*************************************************************************/

/* Also calculates the number of bits necessary to code the scalefactors. */

int scale_bitcount( III_scalefac_t *scalefac, gr_info *cod_info,
		int gr, int ch )
{
    int i, k, sfb, max_slen1 = 0, max_slen2 = 0, /*a, b, */ ep = 2;

    //    static int slen1[16] = { 0, 0, 0, 0, 3, 1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4 };
    //    static int slen2[16] = { 0, 1, 2, 3, 0, 1, 2, 3, 1, 2, 3, 1, 2, 3, 2, 3 };
    static int slen1[16] = { 1, 1, 1, 1, 8, 2, 2, 2, 4, 4, 4, 8, 8, 8,16,16 };
    static int slen2[16] = { 1, 2, 4, 8, 1, 2, 4, 8, 2, 4, 8, 2, 4, 8, 4, 8 };

    static int slen1_tab[16] = {0,
	18, 36, 54, 54, 36, 54, 72, 54, 72, 90, 72, 90,108,108,126
    };
    static int slen2_tab[16] = {0,
	10, 20, 30, 33, 21, 31, 41, 32, 42, 52, 43, 53, 63, 64, 74
    };
    int *tab;


    if ( cod_info->block_type == 2 )
    {
            tab = slen1_tab;
            /* a = 18; b = 18;  */
            for ( i = 0; i < 3; i++ )
            {
                for ( sfb = 0; sfb < 6; sfb++ )
                    if ( scalefac->s[gr][ch][sfb][i] > max_slen1 )
                        max_slen1 = scalefac->s[gr][ch][sfb][i];
                for (sfb = 6; sfb < 12; sfb++ )
                    if ( scalefac->s[gr][ch][sfb][i] > max_slen2 )
                        max_slen2 = scalefac->s[gr][ch][sfb][i];
            }
    }
    else
    { /* block_type == 1,2,or 3 */
        tab = slen2_tab;
        /* a = 11; b = 10;   */
        for ( sfb = 0; sfb < 11; sfb++ )
            if ( scalefac->l[gr][ch][sfb] > max_slen1 )
                max_slen1 = scalefac->l[gr][ch][sfb];
        for ( sfb = 11; sfb < 21; sfb++ )
            if ( scalefac->l[gr][ch][sfb] > max_slen2 )
                max_slen2 = scalefac->l[gr][ch][sfb];
    }



    /* from Takehiro TOMINAGA <tominaga@isoternet.org> 10/99
     * loop over *all* posible values of scalefac_compress to find the
     * one which uses the smallest number of bits.  ISO would stop
     * at first valid index */
    cod_info->part2_length=10000;
    for ( k = 0; k < 16; k++ )
    {
        if ( (max_slen1 < slen1[k]) && (max_slen2 < slen2[k]) &&
             (cod_info->part2_length > tab[k])) {
	  cod_info->part2_length=tab[k];
	  cod_info->scalefac_compress=k;
	  ep=0;  /* we found a suitable scalefac_compress */
	  //break;  
	}
    }

    //    if ( ep == 0 )  cod_info->scalefac_compress = k;
    return ep;
}




/*************************************************************************/
/*            scale_bitcount_lsf                                         */
/*************************************************************************/

/* Also counts the number of bits to encode the scalefacs but for MPEG 2 */ 
/* Lower sampling frequencies  (24, 22.05 and 16 kHz.)                   */
 
/*  This is reverse-engineered from section 2.4.3.2 of the MPEG2 IS,     */
/* "Audio Decoding Layer III"                                            */

int scale_bitcount_lsf( III_scalefac_t *scalefac, gr_info *cod_info,
		    int gr, int ch )
{
    int table_number, row_in_table, partition, nr_sfb, window, over;
    int i, sfb, max_sfac[ 4 ];
    unsigned *partition_table;

    /*
      Set partition table. Note that should try to use table one,
      but do not yet...
    */
    if ( cod_info->preflag )
	table_number = 2;
    else
	table_number = 0;

    for ( i = 0; i < 4; i++ )
	max_sfac[i] = 0;

    if ( cod_info->window_switching_flag != 0 && cod_info->block_type == 2 )
    {
	    row_in_table = 1;
	    partition_table = &nr_of_sfb_block[table_number][row_in_table][0];
	    for ( sfb = 0, partition = 0; partition < 4; partition++ )
	    {
		nr_sfb = partition_table[ partition ] / 3;
		for ( i = 0; i < nr_sfb; i++, sfb++ )
		    for ( window = 0; window < 3; window++ )
			if ( scalefac->s[gr][ch][sfb][window] > max_sfac[partition] )
			    max_sfac[partition] = scalefac->s[gr][ch][sfb][window];
	    }
    }
    else
    {
	row_in_table = 0;
	partition_table = &nr_of_sfb_block[table_number][row_in_table][0];
	for ( sfb = 0, partition = 0; partition < 4; partition++ )
	{
	    nr_sfb = partition_table[ partition ];
	    for ( i = 0; i < nr_sfb; i++, sfb++ )
		if ( scalefac->l[gr][ch][sfb] > max_sfac[partition] )
		    max_sfac[partition] = scalefac->l[gr][ch][sfb];
	}
    }

    for ( over = 0, partition = 0; partition < 4; partition++ )
    {
	if ( max_sfac[partition] > max_sfac_tab[table_number][partition] )
	    over++;
    }
    if ( !over )
    {
	/*
	  Since no bands have been over-amplified, we can set scalefac_compress
	  and slen[] for the formatter
	*/
	static int log2tab[] = { 0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4 };

	unsigned slen1, slen2, slen3, slen4;

        cod_info->sfb_partition_table = &nr_of_sfb_block[table_number][row_in_table][0];
	for ( partition = 0; partition < 4; partition++ )
	    cod_info->slen[partition] = log2tab[max_sfac[partition]];

	/* set scalefac_compress */
	slen1 = cod_info->slen[ 0 ];
	slen2 = cod_info->slen[ 1 ];
	slen3 = cod_info->slen[ 2 ];
	slen4 = cod_info->slen[ 3 ];

	switch ( table_number )
	{
	  case 0:
	    cod_info->scalefac_compress = (((slen1 * 5) + slen2) << 4)
		+ (slen3 << 2)
		+ slen4;
	    break;

	  case 1:
	    cod_info->scalefac_compress = 400
		+ (((slen1 * 5) + slen2) << 2)
		+ slen3;
	    break;

	  case 2:
	    cod_info->scalefac_compress = 500 + (slen1 * 3) + slen2;
	    break;

	  default:
	    fprintf( stderr, "intensity stereo not implemented yet\n" );
	    exit( EXIT_FAILURE );
	    break;
	}
    }
#ifdef DEBUG
    if ( over ) 
        printf( "---WARNING !! Amplification of some bands over limits\n" );
#endif
    if (!over) {
      assert( cod_info->sfb_partition_table );     
      cod_info->part2_length=0;
      for ( partition = 0; partition < 4; partition++ )
	cod_info->part2_length += cod_info->slen[partition] * cod_info->sfb_partition_table[partition];
    }
    return over;
}





/*************************************************************************/
/*            calc_noise                                                 */
/*************************************************************************/
/*  mt 5/99:  Function: Improved calc_noise for a single channel   */
int calc_noise1( double xr[576], int ix[576], gr_info *cod_info,
            double xfsf[4][CBLIMIT], double distort[4][CBLIMIT],
            III_psy_xmin *l3_xmin,int gr, int ch, double *over_noise,
            double *tot_noise, double *max_noise)

{
    int start, end, sfb, l, i, over=0;
    double sum,step,bw;

    D192_3 *xr_s;
    I192_3 *ix_s;

    static double pow43[PRECALC_SIZE];
    static int init=0;
    int count=0;
    double noise;
    *over_noise=0;
    *tot_noise=0;
    *max_noise=-999;

    if (init==0) {
      init++;
      for(i=0;i<PRECALC_SIZE;i++)
        pow43[i] = pow((double)i, 4.0/3.0);
    }
      
    xr_s = (D192_3 *) xr;
    ix_s = (I192_3 *) ix;

    step = pow( 2.0, (cod_info->quantizerStepSize) * 0.25 );
    for ( sfb = 0; sfb < cod_info->sfb_lmax; sfb++ )
    {
        start = scalefac_band_long[ sfb ];
        end   = scalefac_band_long[ sfb+1 ];
        bw = end - start;

        for ( sum = 0.0, l = start; l < end; l++ )
        {
            double temp;
            temp = fabs(xr[l]) - pow43[ix[l]] * step;
            sum += temp * temp;
        }
        xfsf[0][sfb] = sum / bw;
	// max -30db noise below threshold 
	noise = 10*log10(Max(.001,xfsf[0][sfb] / l3_xmin->l[gr][ch][sfb]));
        distort[0][sfb] = noise;
        if (noise>0) {
	  over++;
          //*over_noise += xfsf[0][sfb] - l3_xmin->l[gr][ch][sfb];
	  *over_noise += noise;
	}
	*tot_noise += noise;
	*max_noise=Max(*max_noise,noise);
	count++;

    }


    for ( i = 0; i < 3; i++ )
    {
        step = pow( 2.0, (cod_info->quantizerStepSize) * 0.25 ); /*
subblock_gain ? */
        if (cod_info->subblock_gain[i] )
          step *= pow(2.0,-2.0*cod_info->subblock_gain[i]);

        for ( sfb = cod_info->sfb_smax; sfb < 12; sfb++ )
        {
            start = scalefac_band_short[ sfb ];
            end   = scalefac_band_short[ sfb+1 ];
            bw = end - start;
            
            for ( sum = 0.0, l = start; l < end; l++ )
            {
                double temp;
                temp = fabs((*xr_s)[l][i]) - pow43[(*ix_s)[l][i]] * step;
                sum += temp * temp;
            }       
            xfsf[i+1][sfb] = sum / bw;
	    // max -30db noise below threshold 
	    noise = 10*log10(Max(.001,xfsf[i+1][sfb] / l3_xmin->s[gr][ch][sfb][i] ));
            distort[i+1][sfb] = noise;
            if (noise > 0) {
	      over++;
	      //*over_noise +=xfsf[i+1][sfb]-l3_xmin->s[gr][ch][sfb][i];
	      *over_noise += noise;
	    }
	    *tot_noise += noise;
	    *max_noise=Max(*max_noise,noise);
	    count++;	    
        }
    }

	if (count>1) *tot_noise /= count;
	if (over>1) *over_noise /= over;
	return over;
}




/*************************************************************************/
/*            calc_xmin                                                  */
/*************************************************************************/

/*
  Calculate the allowed distortion for each scalefactor band,
  as determined by the psychoacoustic model.
  xmin(sb) = ratio(sb) * en(sb) / bw(sb)
*/
void calc_xmin( double xr[2][2][576], III_psy_ratio *ratio,
	   gr_info *cod_info, III_psy_xmin *l3_xmin,
	   int gr, int ch )
{
    int start, end, sfb, l, b;
    double en0, bw, ener;

    D192_3 *xr_s;
    xr_s = (D192_3 *) xr[gr][ch] ;

    if (ATHonly) {    
      for ( sfb = cod_info->sfb_smax; sfb < SFB_SMAX - 1; sfb++ )
	  for ( b = 0; b < 3; b++ )
	      l3_xmin->s[gr][ch][sfb][b]=ATH_s[sfb];
      for ( sfb = 0; sfb < cod_info->sfb_lmax; sfb++ )
	  l3_xmin->l[gr][ch][sfb]=ATH_l[sfb];
      
    }else{

      for ( sfb = cod_info->sfb_smax; sfb < SFB_SMAX - 1; sfb++ ){
	start = scalefac_band_short[ sfb ];
        end   = scalefac_band_short[ sfb + 1 ];
	bw = end - start;
        for ( b = 0; b < 3; b++ ) {
	  for ( en0 = 0.0, l = start; l < end; l++ ) {
	    ener = (*xr_s)[l][b] * (*xr_s)[l][b];
	    en0 += ener;
	  }
	  l3_xmin->s[gr][ch][sfb][b] = masking_lower*ratio->s[gr][ch][sfb][b] * en0/bw;
	  l3_xmin->s[gr][ch][sfb][b]=Max(ATH_s[sfb],l3_xmin->s[gr][ch][sfb][b]);
	}
      }
      
      
      for ( sfb = 0; sfb < cod_info->sfb_lmax; sfb++ ){
	start = scalefac_band_long[ sfb ];
	end   = scalefac_band_long[ sfb+1 ];
	bw = end - start;
	
        for ( en0 = 0.0, l = start; l < end; l++ ) {
	  ener = xr[gr][ch][l] * xr[gr][ch][l];
	  en0 += ener;
	}
	
        l3_xmin->l[gr][ch][sfb] =masking_lower*ratio->l[gr][ch][sfb] * en0/bw;
	l3_xmin->l[gr][ch][sfb]=Max(ATH_l[sfb],l3_xmin->l[gr][ch][sfb]);
      }
    }
}



/*************************************************************************/
/*            loop_break                                                 */
/*************************************************************************/

/*  Function: Returns zero if there is a scalefac which has not been
    amplified. Otherwise it returns one. 
*/

int loop_break( III_scalefac_t *scalefac, gr_info *cod_info,
	    int gr, int ch )
{
    int i, sfb;

    for ( sfb = 0; sfb < cod_info->sfb_lmax; sfb++ )
        if ( scalefac->l[gr][ch][sfb] == 0 )
	    return 0;

    for ( sfb = cod_info->sfb_smax; sfb < 12; sfb++ )
        for ( i = 0; i < 3; i++ )
            if ( scalefac->s[gr][ch][sfb][i] == 0 )
		return 0;

    return 1;
}




/*************************************************************************/
/*            preemphasis                                                */
/*************************************************************************/

/*
  See ISO 11172-3  section  C.1.5.4.3.4
*/
int preemphasis( double xr[576], double xrpow[576], 
     III_psy_xmin  *l3_xmin,
     int gr, int ch, III_side_info_t *l3_side, double distort[4][CBLIMIT] )
{
    int i, sfb, start, end, over;
    double ifqstep;
    gr_info *cod_info = &l3_side->gr[gr].ch[ch].tt;


    /*
      Preemphasis is switched on if in all the upper four scalefactor
      bands the actual distortion exceeds the threshold after the
      first call of the inner loop
    */
    over = 0;
    if ( cod_info->block_type != 2 && cod_info->preflag == 0 )
    {	
	for ( sfb = 17; sfb < 21; sfb++ )
	    if ( distort[0][sfb]>0 ) over++;

	if (over == 4 )
	{
	    double t,t34;
	    cod_info->preflag = 1;
	    ifqstep = ( cod_info->scalefac_scale == 0 ) ? SQRT2 : 2.0;
	    for ( sfb = 11; sfb < cod_info->sfb_lmax; sfb++ )
	    {
		t=pow( ifqstep, (double) pretab[sfb] );
		t34=sqrt(sqrt(t)*t);
		l3_xmin->l[gr][ch][sfb] *= t*t;
		start = scalefac_band_long[ sfb ];
		end   = scalefac_band_long[ sfb+1 ];
		for( i = start; i < end; i++ ) xr[i]*=t;
		for( i = start; i < end; i++ ) xrpow[i]*=t34;
	    }
	}
    }
    return (over == 4);
}




/*************************************************************************/
/*            amp_scalefac_bands                                         */
/*************************************************************************/

/* 
  Amplify the scalefactor bands that violate the masking threshold.
  See ISO 11172-3 Section C.1.5.4.3.5
*/
int amp_scalefac_bands( double xr[576], double xrpow[576], 
		    III_psy_xmin *l3_xmin, III_side_info_t *l3_side,
		    III_scalefac_t *scalefac,
		    int gr, int ch, int iteration, double distort[4][CBLIMIT])
{
    int start, end, l, sfb, i, over = 0;
    double ifqstep, ifqstep2, ifqstep34;
    double distort_thresh;
    D192_3 *xr_s;
    D192_3 *xrpow_s;
    gr_info *cod_info;
    cod_info = &l3_side->gr[gr].ch[ch].tt;

    xr_s = (D192_3 *) xr;
    xrpow_s = (D192_3 *) xrpow;


    if ( cod_info->scalefac_scale == 0 )
	ifqstep = SQRT2;
    else
	ifqstep = 2.0;


    ifqstep2 = ifqstep * ifqstep;
    ifqstep34=sqrt(sqrt(ifqstep)*ifqstep);

    /* distort_thresh = 0, unless all bands have distortion 
     * less than masking.  In that case, just amplify bands with distortion
     * within 95% of largest distortion/masking ratio */
    distort_thresh = -900;
    for ( sfb = 0; sfb < cod_info->sfb_lmax; sfb++ )
      distort_thresh = Max(1.05*distort[0][sfb],distort_thresh);
    distort_thresh=Min(distort_thresh,0.0);


    for ( sfb = 0; sfb < cod_info->sfb_lmax; sfb++ )
    {
	if ( distort[0][sfb]>distort_thresh  ) 
	{
	    over++;
	    l3_xmin->l[gr][ch][sfb] *= ifqstep2;
	    scalefac->l[gr][ch][sfb]++;
	    start = scalefac_band_long[sfb];
	    end   = scalefac_band_long[sfb+1];
	    for ( l = start; l < end; l++ ) xr[l] *= ifqstep;
	    for ( l = start; l < end; l++ ) xrpow[l] *= ifqstep34;
	}
    }



    /*
      Note that scfsi is not enabled for frames containing
      short blocks
    */

    distort_thresh = -900;
    for ( i = 0; i < 3; i++ )
      for ( sfb = cod_info->sfb_smax; sfb < 12; sfb++ ) {
	  distort_thresh = Max(1.05*distort[i+1][sfb],distort_thresh);
      }
    distort_thresh=Min(distort_thresh,0.0);


    for ( i = 0; i < 3; i++ )
      for ( sfb = cod_info->sfb_smax; sfb < 12; sfb++ ) {
            if ( distort[i+1][sfb]>distort_thresh)
            {
                over++;
                l3_xmin->s[gr][ch][sfb][i] *= ifqstep2;
                scalefac->s[gr][ch][sfb][i]++;
                start = scalefac_band_short[sfb];
                end   = scalefac_band_short[sfb+1];
                for ( l = start; l < end; l++ ) (*xr_s)[l][i] *= ifqstep;
		for ( l = start; l < end; l++ ) (*xrpow_s)[l][i] *= ifqstep34;
            }
      }
    return over;
}




#ifdef _MSC_VER
#define MSVC_XRPOW_ASM
#ifdef MSVC_XRPOW_ASM
# define MSVC_FTOL(src, dest) do { \
    double src_ = (src); \
    int dest_; \
    { \
      __asm fld src_ \
      __asm fistp dest_ \
    } \
    (dest) = dest_; \
  } while (0)
# endif
#endif

void quantize_xrpow( double xr[576], int ix[576], gr_info *cod_info )
{
  /* quantize on xr^(3/4) instead of xr */
  register int j;
  double quantizerStepSize;
  double istep_l,istep0,istep1,istep2;
#if defined(__GNUC__) && defined(__i386__) 
#elif defined(MSVC_XRPOW_ASM)
#else
  double compareval0;
#endif

  quantizerStepSize = cod_info->quantizerStepSize;
  
  istep_l = pow ( 2.0, quantizerStepSize * -0.1875 );
  
  if ((cod_info->block_type==2))
    {
      istep0 = istep_l * pow(2.0,1.5* (double) cod_info->subblock_gain[0]);
      istep1 = istep_l * pow(2.0,1.5* (double) cod_info->subblock_gain[1]);
      istep2 = istep_l * pow(2.0,1.5* (double) cod_info->subblock_gain[2]);
      for (j=192;j>0;j--) 
        {
#if defined(__GNUC__) && defined(__i386__)
          asm ("fistpl %0 ": "=m"(*(ix++)): "t"(istep0*(*(xr++)) - 0.0946): "st");
          asm ("fistpl %0 ": "=m"(*(ix++)): "t"(istep1*(*(xr++)) - 0.0946): "st");
          asm ("fistpl %0 ": "=m"(*(ix++)): "t"(istep2*(*(xr++)) - 0.0946): "st");
#elif defined(MSVC_XRPOW_ASM)
          MSVC_FTOL((istep0*(*(xr++)) - 0.0946), *(ix++));
          MSVC_FTOL((istep1*(*(xr++)) - 0.0946), *(ix++));
          MSVC_FTOL((istep2*(*(xr++)) - 0.0946), *(ix++));
#else
          *(ix++) = (int)( istep0*(*(xr++))  + 0.4054);
          *(ix++) = (int)( istep1*(*(xr++))  + 0.4054);
          *(ix++) = (int)( istep2*(*(xr++))  + 0.4054);
#endif
        }
    }
  else
    {
#if defined(__GNUC__) && defined(__i386__) 
      for (j=576;j>0;j--) 
          asm ("fistpl %0 ": "=m"(*(ix++)): "t"(istep_l*(*(xr++)) - 0.0946): "st");
#elif defined(MSVC_XRPOW_ASM)
      for (j=576;j>0;j--) {
        MSVC_FTOL((istep_l*(*(xr++)) - 0.0946), *(ix++));
      }
#else
      compareval0 = (1.0 - 0.4054)/istep_l;
      /* depending on architecture, it may be worth calculating a few more compareval's.
         eg.  compareval1 = (2.0 - 0.4054/istep_l); 
              .. and then after the first compare do this ...
              if compareval1>*xr then ix = 1;
         On a pentium166, it's only worth doing the one compare (as done here), as the second
         compare becomes more expensive than just calculating the value. Architectures with 
         slow FP operations may want to add some more comparevals. try it and send your diffs 
         statistically speaking
         73% of all xr*istep_l values give ix=0
         16% will give 1
         4%  will give 2
      */
      for (j=576;j>0;j--) 
        {
          if (compareval0 > *xr) {
            *(ix++) = 0;
            xr++;
          } else
            *(ix++) = (int)( istep_l*(*(xr++))  + 0.4054);
        }
#endif
    }
}






/*
  Seymour's comment:  Jan 8 1995
  When mixed_block_flag is set, the low subbands 0-1 undergo the long
  window transform and are each split into 18 frequency lines, while
  the remaining 30 subbands undergo the short window transform and are
  each split into 6 frequency lines. A problem now arises, as neither
  the short or long scale factor bands apply to this mixed spectrum.
  The standard resolves this situation by using the first 8 long scale
  factor bands for the low spectrum and the short scale factor bands
  in the range of 3 to 11 (inclusive) for the remaining frequency lines.
  These scale factor bands do not match exactly to the 0-1 subbands
  for all sampling frequencies (32,44.1 and 48 kHz); however they
  were designed so that there would not be a frequency gap or overlap
  at the switch over point. (Note multiply short frequency lines by 3
  to account for wider frequency line.) 
  */

/* mt 4/99:  ISO code cannot produces mixed blocks,
 * Fhg Code also never seems to use them, so no need to add them
 * to this code 
 */
/*************************************************************************/
/*            gr_deco                                                    */
/*************************************************************************/

void gr_deco( gr_info *cod_info )
{
    if ( cod_info->window_switching_flag != 0 && cod_info->block_type == 2 )
        {
            cod_info->sfb_lmax = 0; /* No sb*/
            cod_info->sfb_smax = 0;
        }
    else
    {
      /* MPEG 1 doesnt use last scalefactor band? */
        cod_info->sfb_lmax = SFB_LMAX - 1;
        cod_info->sfb_smax = SFB_SMAX - 1;    /* No sb */
    }
}







static int OldValue = -30; /* guess it or so. */

typedef enum {
    BINSEARCH_NONE,
    BINSEARCH_UP, 
    BINSEARCH_DOWN
} binsearchDirection_t;

/*-------------------------------------------------------------------------*/
int 
bin_search_StepSize2(int      desired_rate, 
		    double   start, 
		    int      bot, 
		    int     *ix,
		    double   xrs[576], 
		    double   xrspow[576], 
		    gr_info *cod_info)
/*-------------------------------------------------------------------------*/
{
    int flag_GoneOver = 0;
    int CurrentStep = 4;
    int nBits;
    int StepSize = OldValue;
    binsearchDirection_t Direction = BINSEARCH_NONE;
    do
    {
	cod_info->quantizerStepSize = StepSize;
	quantize_xrpow(xrspow, ix, cod_info);
	nBits = count_bits(ix,cod_info);  

	if (CurrentStep == 1 )
        {
	    break; /* nothing to adjust anymore */
	}
	if (flag_GoneOver)
	{
	    CurrentStep /= 2;
	}
	if (nBits > desired_rate)  /* increase Quantize_StepSize */
	{
	    if (Direction == BINSEARCH_DOWN && !flag_GoneOver)
	    {
		flag_GoneOver = 1;
		CurrentStep /= 2; /* late adjust */
	    }
	    Direction = BINSEARCH_UP;
	    StepSize += CurrentStep;
	}
	else if (nBits < desired_rate)
	{
	    if (Direction == BINSEARCH_UP && !flag_GoneOver)
	    {
		flag_GoneOver = 1;
		CurrentStep /= 2; /* late adjust */
	    }
	    Direction = BINSEARCH_DOWN;
	    StepSize -= CurrentStep;
	}
	else break; /* nBits == desired_rate;; most unlikely to happen.
*/
    } while (1); /* For-ever, break is adjusted. */
    OldValue = StepSize;
    return nBits;
}







int quant_compare(
int best_over,double best_tot_noise,double best_over_noise,double best_max_noise,
int over,double tot_noise, double over_noise, double max_noise)
{
  /*
    noise is given in decibals (db) relative to masking thesholds.

    over_noise:  sum of quantization noise > masking
    tot_noise:   sum of all quantization noise
    max_noise:   max quantization noise 

   */
  int better=0;

  if (experimentalX==0) {
    better = ((over < best_over) ||
	    ((over==best_over) && (over_noise<best_over_noise)) ) ;
  }

  if (experimentalX==1) 
    better = max_noise < best_max_noise;

  if (experimentalX==2) {
    better = tot_noise < best_tot_noise;
  }
  if (experimentalX==3) {
    better = (tot_noise < best_tot_noise) &&
      (max_noise < best_max_noise + 2);
  }
  if (experimentalX==4) {
    better = ( ( (0>=max_noise) && (best_max_noise>2)) ||
     ( (0>=max_noise) && (best_max_noise<0) && ((best_max_noise+2)>max_noise) && (tot_noise<best_tot_noise) ) ||
     ( (0>=max_noise) && (best_max_noise>0) && ((best_max_noise+2)>max_noise) && (tot_noise<(best_tot_noise+best_over_noise)) ) ||
     ( (0<max_noise) && (best_max_noise>-0.5) && ((best_max_noise+1)>max_noise) && ((tot_noise+over_noise)<(best_tot_noise+best_over_noise)) ) ||
     ( (0<max_noise) && (best_max_noise>-1) && ((best_max_noise+1.5)>max_noise) && ((tot_noise+over_noise+over_noise)<(best_tot_noise+best_over_noise+best_over_noise)) ) );
  }
  if (experimentalX==5) {
    better =   (over_noise <  best_over_noise)
      || ((over_noise == best_over_noise)&&(tot_noise < best_tot_noise));
  }

  return better;
}


int VBR_compare(
int best_over,double best_tot_noise,double best_over_noise,double best_max_noise,
int over,double tot_noise, double over_noise, double max_noise)
{
  /*
    noise is given in decibals (db) relative to masking thesholds.

    over_noise:  sum of quantization noise > masking
    tot_noise:   sum of all quantization noise
    max_noise:   max quantization noise 

   */
  int better=0;

  better = ((over <= best_over) &&
	    (over_noise<=best_over_noise) &&
	    (tot_noise<=best_tot_noise) &&
	    (max_noise<=best_max_noise));
  return better;
}
  







