/*
 * ishido.c - Ancient stone matching game.  
 * 
 * Authors:  John Sullivan, Amdahl Corporation (jjs40@cd.amdahl.com)
 *           Sean Cummins, Amdahl Corporation (spc10@cd.amdahl.com)
 * 
 * "No matter how great your achievements or how stunning your defeats, over a
 * billion Chinese couldn't care less."
 * 
 */

/****************************************************************************/

#include <stdio.h>
#ifndef vax11c
#include <pwd.h>
#endif /* !vax11c */
#include "gl.h"

/****************************************************************************/

#include "bitmap/hwood.bm"
#include "bitmap/vwood.bm"
#include "bitmap/atom.bm"
#include "bitmap/dollar.bm"
#include "bitmap/hammer.bm"
#include "bitmap/maple.bm"
#include "bitmap/smile.bm"
#include "bitmap/star.bm"

/****************************************************************************/

#define TILE_WIDTH		50
#define TILE_HEIGHT		50
#define	BOARD_WIDTH		12
#define BOARD_HEIGHT		8
#define MARGIN_X		10
#define MARGIN_Y		10

/****************************************************************************/

#define BOARD_X		MARGIN_X
#define BOARD_Y		MARGIN_Y
#define BOARD_W		(TILE_WIDTH*BOARD_WIDTH)
#define BOARD_H		(TILE_HEIGHT*BOARD_HEIGHT)
#define TILE_X(x)	(BOARD_X+TILE_WIDTH*(x))
#define TILE_Y(y)	(BOARD_Y+TILE_HEIGHT*(y))
#define SCORE_X		MARGIN_X*2+BOARD_W
#define SCORE_Y		BOARD_Y
#define	SCORE_W		20*GL_FONT_WIDTH
#define SCREEN_W	MARGIN_X*3+BOARD_W+SCORE_W
#define SCREEN_H	MARGIN_Y*2+BOARD_H
#define QUITB_H		25
#define QUITB_W		SCORE_W
#define QUITB_X		SCORE_X
#define QUITB_Y		SCREEN_H-MARGIN_Y-QUITB_H-1

#define RULEB_H		25
#define RULEB_W		SCORE_W
#define RULEB_X		SCORE_X
#define RULEB_Y		QUITB_Y-MARGIN_Y-GAMEB_H

#define GAMEB_H		25
#define GAMEB_W		SCORE_W
#define GAMEB_X		SCORE_X
#define GAMEB_Y		RULEB_Y-MARGIN_Y-GAMEB_H
#define SCORE_H		GAMEB_Y-MARGIN_Y-SCORE_Y

/****************************************************************************/

#define IS_INTERIOR(x,y) ((x > 0) && (x < BOARD_WIDTH-1) && \
			  (y > 0) && (y < BOARD_HEIGHT-1))

/****************************************************************************/

#define TILE_SHAPES		6
#define DECK_SIZE		TILE_SHAPES*TILE_SHAPES*2

#define TILE_EMPTY 		-1
typedef int     TILE;

#define TILE_COLOR(x) 	((x)%TILE_SHAPES)
#define TILE_SHAPE(x) 	((x)/TILE_SHAPES)

/****************************************************************************/

#ifndef HIGH_SCORE_FILE
#ifdef vax11c
#define HIGH_SCORE_FILE		"x11data:ishido.scores"
#else
#define HIGH_SCORE_FILE		"ishido_scores"
#endif /* vax11c */
#endif

#define NUM_HIGH_SCORES 	20

typedef struct HIGH_SCORE_STRUCT {
	char            name[12];
	int             score;
	int             fourways;
	int             tiles;
}               HIGH_SCORE;

/****************************************************************************/

extern void     quit_button();
extern void     new_game_button();
extern void     end_game_button();
extern void     rule_button();

/****************************************************************************/

#define streq(a,b) (strcmp(a,b)==0)

/****************************************************************************/

GL_PIXEL        pix_bg;
GL_PIXEL        pix_border;
GL_PIXEL        pix_text;
GL_PIXEL        pix_wood_fg[2];
GL_PIXEL        pix_wood_bg[2];
GL_PIXEL        pix_tile_bg[TILE_SHAPES];
GL_PIXEL        pix_tile_fg;
GL_PIXEL	pix_highlight;

GL_BITMAP       bitmap_wood[2];
GL_BITMAP       bitmap_tile[TILE_SHAPES];

GB_BUTTON       button[3];

/****************************************************************************/

HIGH_SCORE      high_score[NUM_HIGH_SCORES];

int             game_over;
int             rule_screen;

TILE            board[BOARD_WIDTH][BOARD_HEIGHT];

int             num_deck_tiles;
TILE            deck[DECK_SIZE];

int             score;
int             fourways;
int		bonehead;
int 		silent;

int		num_possible;
int		possible_x[BOARD_WIDTH*BOARD_HEIGHT];
int		possible_y[BOARD_WIDTH*BOARD_HEIGHT];

/****************************************************************************/

int
int_rand(m)
	int             m;
{
	int             v;

#ifdef vax11c
	v = (rand() % m);
#else
	v = (lrand48() % m);
#endif /* vax11c */
	return (v);
};

/****************************************************************************/

void
seed_rand()
{
#ifdef vax11c
	srand(time(NULL));
#else
	srand48(time(NULL));
#endif /* vax11c */
};

/****************************************************************************/

char           *
get_user_name()
{
	int             uid;
#ifdef vax11c
	return(getenv("USER"));	/* under VMS this always works */
#else
	struct passwd  *passwd;

	uid = geteuid();
	passwd = getpwuid(uid);
	return (passwd->pw_name);
#endif /* vax11c */
};

/****************************************************************************/

void
deck_init()
{
	int             i, j, n, c;
	TILE            tile;

	num_deck_tiles = DECK_SIZE;
	for (i = 0; i < DECK_SIZE; i++) {
		deck[i] = i / 2;
	};
	for (i = 0; i < DECK_SIZE - 1; i++) {
		n = int_rand(DECK_SIZE - i - 1);
		tile = deck[i + n + 1];
		deck[i + n + 1] = deck[i];
		deck[i] = tile;
	};
	i = 0;
	c = 0;
	while (c < TILE_SHAPES) {
		n = 0;
		for (j = 0; j < c; j++) {
			if (TILE_COLOR(deck[i]) == TILE_COLOR(deck[j])) {
				n++;
			};
			if (TILE_SHAPE(deck[i]) == TILE_SHAPE(deck[j])) {
				n++;
			};
		};
		if (n == 0) {
			tile = deck[c];
			deck[c] = deck[i];
			deck[i] = tile;
			c++;
		};
		i++;
	};
};

/****************************************************************************/

void
deck_print()
{
	int             i, c;

	printf("num_deck_tiles = %d:\n", num_deck_tiles);

	c = 0;
	for (i = 0; i < num_deck_tiles; i++) {
		printf("%4d ", deck[i]);
		c++;
		if (c == 14) {
			printf("\n");
			c = 0;
		};
	};
	printf("\n");
};

/****************************************************************************/

TILE
deck_take_tile()
{
	int             i;
	TILE            tile;

	tile = deck[0];

	for (i = 0; i < num_deck_tiles - 1; i++) {
		deck[i] = deck[i + 1];
	};
	num_deck_tiles = num_deck_tiles - 1;
	return (tile);
};

/****************************************************************************/

void
board_init()
{
	int             i, j;

	for (j = 0; j < BOARD_HEIGHT; j++) {
		for (i = 0; i < BOARD_WIDTH; i++) {
			board[i][j] = TILE_EMPTY;
		};
	};
	board[0][0] = deck_take_tile();
	board[0][7] = deck_take_tile();
	board[11][0] = deck_take_tile();
	board[11][7] = deck_take_tile();
	board[5][3] = deck_take_tile();
	board[6][4] = deck_take_tile();
};

/****************************************************************************/

TILE
board_get_tile(x, y)
	int             x, y;
{
	if ((x < 0) || (x >= BOARD_WIDTH)) {
		return (TILE_EMPTY);
	};
	if ((y < 0) || (y >= BOARD_HEIGHT)) {
		return (TILE_EMPTY);
	};
	return (board[x][y]);
};

/****************************************************************************/

int
board_adjacent_tiles(tile, x, y, c, s)
	TILE            tile;
	int             x, y;
	int            *c;
	int            *s;
{
	int             i;
	int             num;
	TILE            adjtile[4];

	adjtile[0] = board_get_tile(x - 1, y);
	adjtile[1] = board_get_tile(x + 1, y);
	adjtile[2] = board_get_tile(x, y - 1);
	adjtile[3] = board_get_tile(x, y + 1);
	num = 0;
	for (i = 0; i < 4; i++) {
		if (adjtile[i] != TILE_EMPTY) {
			c[num] = (TILE_COLOR(tile) == TILE_COLOR(adjtile[i]));
			s[num] = (TILE_SHAPE(tile) == TILE_SHAPE(adjtile[i]));
			num++;
		};
	};
	return (num);
};

/****************************************************************************/

int
board_can_place_tile(x, y)
	int             x, y;
{
	int             num;
	TILE            tile;
	int             c[4], s[4];

	if (board[x][y] != TILE_EMPTY) {
		return (0);
	};
	tile = deck[0];
	num = board_adjacent_tiles(tile, x, y, c, s);
	if (num == 1) {
		if (s[0] || c[0])
			return (num);
	};
	if (num == 2) {
		if (s[0] && c[1])
			return (num);
		if (s[1] && c[0])
			return (num);
	};
	if (num == 3) {
		if (s[0] && c[1] && c[2])
			return (num);
		if (s[1] && c[2] && c[0])
			return (num);
		if (s[2] && c[0] && c[1])
			return (num);
		if (c[0] && s[1] && s[2])
			return (num);
		if (c[1] && s[2] && s[0])
			return (num);
		if (c[2] && s[0] && s[1])
			return (num);
	};
	if (num == 4) {
		if (s[0] && s[1] && c[2] && c[3])
			return (num);
		if (s[0] && s[2] && c[1] && c[3])
			return (num);
		if (s[0] && s[3] && c[1] && c[2])
			return (num);
		if (s[1] && s[2] && c[0] && c[3])
			return (num);
		if (s[1] && s[3] && c[0] && c[2])
			return (num);
		if (s[2] && s[3] && c[0] && c[1])
			return (num);
	};
	return (0);
};

/****************************************************************************/

void board_update_possible_moves()
{
	int i,j;
	int c;
	
	num_possible = 0;
	if (num_deck_tiles == 0) {
		return;
	};
	for (j = 0; j < BOARD_HEIGHT; j++) {
	        for (i = 0; i < BOARD_WIDTH; i++) {
			if (board_can_place_tile(i,j)) {
				possible_x[num_possible] = i;
				possible_y[num_possible] = j;
				num_possible++;
			};
		};
	};
};

/****************************************************************************/

int             four_way_bonus[] = {
	25, 50, 100, 200, 400, 600, 800, 1000, 5000, 10000, 25000, 50000
};

/****************************************************************************/

int
board_place_tile(x, y)
	int             x, y;
{
	int             num;

	if (num_deck_tiles == 0) {
		return (0);
	};
	num = board_can_place_tile(x, y);
	if (num == 0) {
		return (0);
	};
	board[x][y] = deck_take_tile();
	if (IS_INTERIOR(x, y)) {
		if (num == 1)
			score = score + 1;
		if (num == 2)
			score = score + 2;
		if (num == 3)
			score = score + 4;
		if (num == 4) {
			score = score + 8;
			if (fourways < 12) {
				score = score + four_way_bonus[fourways];
			} else {
				score = score + four_way_bonus[11];
			};
			fourways++;
		};
	};
	return (1);
};

/****************************************************************************/

void
high_score_load()
{
	FILE           *f;
	int             i;

	f = fopen(HIGH_SCORE_FILE, "r");
	if (f != NULL) {
		fread(high_score, sizeof(HIGH_SCORE), NUM_HIGH_SCORES, f);
	} else {
		for (i = 0; i < NUM_HIGH_SCORES; i++) {
			strcpy(high_score[i].name, "nobody");
			high_score[i].score = 0;
			high_score[i].fourways = 0;
			high_score[i].tiles = 66;
		};
	};
	fclose(f);
};

/****************************************************************************/

void
high_score_save()
{
	FILE           *f;

	f = fopen(HIGH_SCORE_FILE, "w");
	if (f != NULL) {
		fwrite(high_score, sizeof(HIGH_SCORE), NUM_HIGH_SCORES, f);
	};
	fclose(f);
};

/****************************************************************************/

void
high_score_add(name, sc, fw, ti)
	char           *name;
	int             sc;
	int             fw;
	int             ti;
{
	int             i;
	int             pos;

	high_score_load();
	pos = NUM_HIGH_SCORES;
	for (i = NUM_HIGH_SCORES - 1; i >= 0; i--) {
		if (sc >= high_score[i].score) {
			pos = i;
		};
	};
	if (pos < NUM_HIGH_SCORES) {
		for (i = NUM_HIGH_SCORES - 1; i > pos; i--) {
			strcpy(high_score[i].name, high_score[i - 1].name);
			high_score[i].score = high_score[i - 1].score;
			high_score[i].fourways = high_score[i - 1].fourways;
			high_score[i].tiles = high_score[i - 1].tiles;
		};
		strcpy(high_score[pos].name, name);
		high_score[pos].score = sc;
		high_score[pos].fourways = fw;
		high_score[pos].tiles = ti;
		high_score_save();
	};
};

/****************************************************************************/

void
init_game_play()
{
	strcpy(button[0].label, "End Game");
	button[0].event_fn = end_game_button;

	game_over = 0;
	rule_screen = 0;

	seed_rand();
	deck_init();
	board_init();
	board_update_possible_moves();

	score = 0;
	fourways = 0;
};

/****************************************************************************/

void
init_game_over()
{
	strcpy(button[0].label, "New Game");
	button[0].event_fn = new_game_button;

	high_score_load();

	game_over = 1;
	rule_screen = 0;
};

/****************************************************************************/

void
init_game_end()
{
	char           *name;

	if (num_deck_tiles == 0)
		score = score + 1000;
	if (num_deck_tiles == 1)
		score = score + 500;
	if (num_deck_tiles == 2)
		score = score + 100;
	name = get_user_name();
	high_score_add(name, score, fourways, num_deck_tiles);
};

/****************************************************************************/

void
init_game(argc, argv)
   int argc;
   char **argv;
{
	int i;
	int help;

	pix_bg = gl_alloc_color("black");
	pix_border = gl_alloc_color("red");
	pix_text = gl_alloc_color("white");
	pix_wood_fg[0] = gl_alloc_color("tan4");
	pix_wood_fg[1] = gl_alloc_color("tan3");
	pix_wood_bg[0] = gl_alloc_color("sienna4");
	pix_wood_bg[1] = gl_alloc_color("sienna3");
	pix_tile_fg = gl_alloc_color("black");
	pix_tile_bg[0] = gl_alloc_color("red");
	pix_tile_bg[1] = gl_alloc_color("green");
	pix_tile_bg[2] = gl_alloc_color("skyblue1");
	pix_tile_bg[3] = gl_alloc_color("yellow");
	pix_tile_bg[4] = gl_alloc_color("mediumpurple1");
	pix_tile_bg[5] = gl_alloc_color("pink");
	pix_highlight = gl_alloc_color("magenta");

	bitmap_wood[0] = gl_load_bitmap(hwood_bits, hwood_width, hwood_height);
	bitmap_wood[1] = gl_load_bitmap(vwood_bits, vwood_width, vwood_height);

	bitmap_tile[0] = gl_load_bitmap(atom_bits, atom_width, atom_height);
	bitmap_tile[1] = gl_load_bitmap(dollar_bits, dollar_width, dollar_height);
	bitmap_tile[2] = gl_load_bitmap(hammer_bits, hammer_width, hammer_height);
	bitmap_tile[3] = gl_load_bitmap(maple_bits, maple_width, maple_height);
	bitmap_tile[4] = gl_load_bitmap(smile_bits, smile_width, smile_height);
	bitmap_tile[5] = gl_load_bitmap(star_bits, star_width, star_height);

	button[0].x = GAMEB_X;
	button[0].y = GAMEB_Y;
	button[0].w = GAMEB_W;
	button[0].h = GAMEB_H;
	button[0].border = pix_border;
	button[0].background = pix_bg;
	button[0].text = pix_text;

	button[1].x = RULEB_X;
	button[1].y = RULEB_Y;
	button[1].w = RULEB_W;
	button[1].h = RULEB_H;
	button[1].border = pix_border;
	button[1].background = pix_bg;
	button[1].text = pix_text;
	strcpy(button[1].label, "Rules");
	button[1].event_fn = rule_button;

	button[2].x = QUITB_X;
	button[2].y = QUITB_Y;
	button[2].w = QUITB_W;
	button[2].h = QUITB_H;
	button[2].border = pix_border;
	button[2].background = pix_bg;
	button[2].text = pix_text;
	strcpy(button[2].label, "Quit");
	button[2].event_fn = quit_button;

	num_possible = 0;

	bonehead = 0;
	silent = 0;
	help = 0;

	for (i = 1; i < argc; i++) {
		if (streq(argv[i],"-bonehead")) {
			bonehead = 1;
		} else if (streq(argv[i],"-silent")) {
			silent = 1;
		} else if (streq(argv[i],"-h")) {
			help = 1;
		} else {
			help = 1;
		};
	};

	if (help) {
		printf("usage: ishido [-bonehead] [-silent]\n");
		printf("       -bonehead       Shows possible moves.\n");
		printf("       -silent         Turns off bell.\n");
		printf("       -h              This message.\n");
		gl_exit();
	};

	init_game_play();
	init_game_over();
};

/****************************************************************************/

void
draw_tile(x, y, tile)
	int             x, y;
	TILE            tile;
{

	gl_set_fg(pix_tile_fg);
	gl_set_bg(pix_tile_bg[TILE_COLOR(tile)]);
	gl_draw_bitmap(bitmap_tile[TILE_SHAPE(tile)], x, y,
		       TILE_WIDTH, TILE_HEIGHT);
};

/****************************************************************************/

void
draw_board_tile(x, y)
	int             x, y;
{
	TILE            tile;

	tile = board[x][y];
	if (tile == TILE_EMPTY) {
		gl_set_fg_bg(pix_wood_fg[IS_INTERIOR(x, y)],
			     pix_wood_bg[IS_INTERIOR(x, y)]);
		gl_draw_bitmap(bitmap_wood[(x + y) % 2],
			       TILE_X(x), TILE_Y(y),
			       TILE_WIDTH, TILE_HEIGHT);
	} else {
		draw_tile(TILE_X(x), TILE_Y(y), tile);
	};
};

/****************************************************************************/

void
draw_board_highlight_tile(x, y)
	int             x, y;
{
	gl_set_fg_bg(pix_highlight,
		     pix_wood_bg[IS_INTERIOR(x, y)]);
	gl_draw_bitmap(bitmap_wood[(x + y) % 2],
			       TILE_X(x), TILE_Y(y),
			       TILE_WIDTH, TILE_HEIGHT);
};

/****************************************************************************/

void draw_board_show_possible_moves()
{
	int i;

        for (i = 0; i < num_possible; i++) {
		draw_board_highlight_tile(possible_x[i],possible_y[i]);
	};
};


/****************************************************************************/

void
draw_board_hide_possible_moves()
{
	int i;
	
	for (i = 0; i < num_possible; i++) {
		draw_board_tile(possible_x[i], possible_y[i]);
	};
};

/****************************************************************************/

void
draw_board()
{
	int             i, j;

	for (j = 0; j < BOARD_HEIGHT; j++) {
		for (i = 0; i < BOARD_WIDTH; i++) {
			draw_board_tile(i, j);
		};
	};
	if (bonehead) {
		draw_board_show_possible_moves();
	};
};

/****************************************************************************/

void
draw_score()
{
	char            buf[128];

	gl_set_fg_bg(pix_text, pix_bg);
	gu_draw_centered_text(SCORE_X + SCORE_W / 2,
			      SCORE_Y + GL_FONT_HEIGHT * 1, "Ishido v1.0");
	gu_draw_centered_text(SCORE_X + SCORE_W / 2,
			      SCORE_Y + GL_FONT_HEIGHT * 3, "Score");
	sprintf(buf, "%06d", score);
	gu_draw_centered_text(SCORE_X + SCORE_W / 2,
			      SCORE_Y + GL_FONT_HEIGHT * 4, buf);
	gu_draw_centered_text(SCORE_X + SCORE_W / 2,
			      SCORE_Y + GL_FONT_HEIGHT * 5, "4-Way Matches");
	sprintf(buf, "%d", fourways);
	gu_draw_centered_text(SCORE_X + SCORE_W / 2,
			      SCORE_Y + GL_FONT_HEIGHT * 6, buf);
	gu_draw_centered_text(SCORE_X + SCORE_W / 2,
			  SCORE_Y + GL_FONT_HEIGHT * 7, "Tiles Remaining");
	sprintf(buf, "%02d", num_deck_tiles);
	gu_draw_centered_text(SCORE_X + SCORE_W / 2,
			      SCORE_Y + GL_FONT_HEIGHT * 8, buf);
	if ((game_over) || (num_deck_tiles == 0)) {
		gu_draw_centered_text(SCORE_X + SCORE_W / 2,
			SCORE_Y + GL_FONT_HEIGHT * 10, "              ");
		gu_draw_centered_text(SCORE_X + SCORE_W / 2,
				SCORE_Y + GL_FONT_HEIGHT * 11, "    ");
		gu_draw_centered_text(SCORE_X + SCORE_W / 2,
				SCORE_Y + GL_FONT_HEIGHT * 12, "         ");
		gl_set_fg_bg(pix_bg, pix_bg);
		gl_fill_rect(SCORE_X + SCORE_W / 2 - TILE_WIDTH / 2,
		    SCORE_Y + GL_FONT_HEIGHT * 13, TILE_WIDTH, TILE_HEIGHT);
	} else {
		gu_draw_centered_text(SCORE_X + SCORE_W / 2,
			SCORE_Y + GL_FONT_HEIGHT * 10, "Possible Moves");
		if (bonehead) {
			sprintf(buf, "%02d", num_possible);
		} else {
			if (num_possible > 0) {
				sprintf(buf,"Yes");
			} else {
				sprintf(buf,"None ");
			};
		};
		gu_draw_centered_text(SCORE_X + SCORE_W / 2,
				SCORE_Y + GL_FONT_HEIGHT * 11, buf);
		gu_draw_centered_text(SCORE_X + SCORE_W / 2,
				SCORE_Y + GL_FONT_HEIGHT * 12, "Next Tile");
		draw_tile(SCORE_X + SCORE_W / 2 - TILE_WIDTH / 2,
			  SCORE_Y + GL_FONT_HEIGHT * 13, deck[0]);
	};
};

/****************************************************************************/

int             rule_lines = 21;
char           *rule_text[] = {
	"There are 72 tiles, two of each combination of six colors and six shapes.",
	"At the start of the game, there are six tiles on the board representing",
	"each shape and color.  To place an additional tile on the board, you",
	"must match at least one adjacent tile according to the following rules:",
	"   - To match a single tile, you must match either the color or shape.",
	"   - To match two tiles, you must match one shape and one color.",
	"   - To match three tiles, you must match two colors and one shape OR two",
	"     shapes and one color.",
	"   - To match four tiles, you must match two colors and two shapes.",
	"",
	"Tiles placed on the interior (lighter) squares score points when placed,",
	"according to the number of adjacent tiles that they match.",
	"   - Matching one tile:    1 point.    - Matching three tiles:  4 points.",
	"   - Matching two tiles:   2 points.   - Matching four tiles:   8 points.",
	"",
	"In addition to the regular score there is an increasing bonus for each",
	"four-way match starting with 25 points for the first and continuing with",
	"50, 100, 200, 400, 600, 800, 1000, 5000, 10000, and 50000 points.",
	"",
	"A bonus of 1000 points is awarded at the end of the game placing for all",
	"72 tiles.  500 and 100 points are awarded for one and two tiles remaining."
};

/****************************************************************************/

void
draw_rule_screen()
{
	int             i;

	gl_set_fg(pix_bg);
	gl_fill_rect(BOARD_X, BOARD_Y, BOARD_W, BOARD_H);
	gl_set_fg_bg(pix_text, pix_bg);
	gu_draw_centered_text(BOARD_X + BOARD_W / 2, BOARD_Y + GL_FONT_HEIGHT,
			      "Rules");
	for (i = 0; i < rule_lines; i++) {
		gl_draw_text(BOARD_X + GL_FONT_WIDTH,
			     BOARD_Y + GL_FONT_HEIGHT * (i + 2),
			     rule_text[i]);
	};
};

/****************************************************************************/

void
draw_high_scores()
{
	int             i;
	char            buf[200];

	gl_set_fg(pix_bg);
	gl_fill_rect(BOARD_X, BOARD_Y, BOARD_W, BOARD_H);
	gl_set_fg_bg(pix_text, pix_bg);
	gu_draw_centered_text(BOARD_X + BOARD_W / 2, BOARD_Y + GL_FONT_HEIGHT * 1,
			      "-- Game Over --");
	gu_draw_centered_text(BOARD_X + BOARD_W / 2, BOARD_Y + GL_FONT_HEIGHT * 2,
			      "##  Name          Score  4-Ways  Tiles");
	gu_draw_centered_text(BOARD_X + BOARD_W / 2, BOARD_Y + GL_FONT_HEIGHT * 3,
			      "--------------------------------------");
	for (i = 0; i < NUM_HIGH_SCORES; i++) {
		sprintf(buf, "%2d  %-12s  %5d    %2d      %2d ", i + 1,
			high_score[i].name, high_score[i].score,
			high_score[i].fourways, high_score[i].tiles);
		gu_draw_centered_text(BOARD_X + BOARD_W / 2,
				   BOARD_Y + GL_FONT_HEIGHT * (i + 4), buf);
	};
};

/****************************************************************************/

void
redraw()
{
	gl_set_fg_bg(pix_border, pix_bg);
	gu_draw_border(BOARD_X, BOARD_Y, BOARD_W, BOARD_H, 2);
	gu_draw_border(SCORE_X, SCORE_Y, SCORE_W, SCORE_H, 2);

	gl_set_fg(pix_bg);
	gl_fill_rect(SCORE_X, SCORE_Y, SCORE_W, SCORE_H);
	draw_score();

	gb_draw_buttons(3, button);

	if (rule_screen) {
		draw_rule_screen();
	} else if (game_over) {
		draw_high_scores();
	} else {
		draw_board();
	};
};

/****************************************************************************/

void
quit_button()
{
	if (!game_over) {
		init_game_end();
	};
	gl_exit_main();
};

/****************************************************************************/

void
new_game_button()
{
	init_game_play();
	redraw();
};

/****************************************************************************/

void
end_game_button()
{
	init_game_end();
	init_game_over();
	redraw();
};

/****************************************************************************/

void
rule_button()
{
	rule_screen = 1 - rule_screen;
	redraw();
};

/****************************************************************************/

void
board_button(event)
	GL_EVENT       *event;
{
	int             x, y;

	x = (event->x - BOARD_X) / TILE_WIDTH;
	y = (event->y - BOARD_Y) / TILE_WIDTH;
	if (board_place_tile(x, y)) {
		if (bonehead) {
		        draw_board_hide_possible_moves();
		};
		board_update_possible_moves();
		draw_board_tile(x, y);
		if (bonehead) {
			draw_board_show_possible_moves();
		};
		draw_score();
	} else if (silent == 0) {
		gl_ring_bell();
	};
};

/****************************************************************************/

void
event(event)
	GL_EVENT       *event;
{
	if ((gb_button_event(event, 3, button) == GL_FALSE) && (!game_over)) {
		if (gu_event_in_rect(event, BOARD_X, BOARD_Y, BOARD_W, BOARD_H)) {
			board_button(event);
		}
	};
};

/****************************************************************************/

main(argc, argv)
	int             argc;
	char           **argv;
{


	gl_init(argc, argv, SCREEN_W, SCREEN_H);

	gl_redraw_func(redraw);
	gl_event_func(event);

	init_game(argc, argv);

	gl_start();
	gl_main_loop();
	gl_exit();
};
