/*
**
** misc.c
**
** Copyright (C) 1995, 1996 Johannes Plass
** 
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
** 
** Author:   Johannes Plass (plass@dipmza.physik.uni-mainz.de)
**           Department of Physic
**           Johannes Gutenberg-University
**           Mainz, Germany
**
*/

/*
 * This code is derived from:
*/

/*
 * misc.c -- Everything that isn't a callback or action.
 * Copyright (C) 1992  Timothy O. Theisen
 *   Author: Tim Theisen           Systems Programmer
 * Internet: tim@cs.wisc.edu       Department of Computer Sciences
 *     UUCP: uwvax!tim             University of Wisconsin-Madison
 *    Phone: (608)262-0438         1210 West Dayton Street
 *      FAX: (608)262-9777         Madison, WI   53706
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
*/

/*
#define MESSAGES
*/
#include "message.h"

#include "config.h"

#include <stdio.h>
#include <stdlib.h>

#ifndef SEEK_SET
#   define SEEK_SET 0
#endif

#include <signal.h>
#ifdef SIGNALRETURNSINT
#   define SIGVAL int
#else
#   define SIGVAL void
#endif

#define GV_MAXLENGTH 512

#ifdef VMS
#   include<descrip.h>
#   include<lnmdef.h>
#   include<lib$routines.h>
#   include<starlet.h>
#   include<rmsdef.h>
#else  
#   include <sys/stat.h>  /* for linux ###jp###*/
#   include <unistd.h>    /* for linux ###jp###*/
#endif

#include <math.h>

#include "paths.h"
#include INC_X11(Xos.h)
#include INC_X11(Xatom.h)
#include INC_X11(Intrinsic.h)
#include INC_X11(StringDefs.h)
#include INC_X11(Shell.h)
#include INC_XAW(Cardinals.h)
#include INC_XAW(SimpleMenu.h)
#include INC_XAW(SmeBSB.h)
#include INC_XAW(SmeLine.h)
#include INC_XAW(Scrollbar.h)
#include INC_XAW(AsciiText.h)
#include INC_X11(IntrinsicP.h)
#include INC_XAW(TextP.h)
#include INC_XMU(StdCmap.h)
#include "Aaa.h"
#include "Frame.h"

#ifdef VMS
#   include <unixio.h>
#endif

#include "gv.h"
#include "ps.h"
#include "info.h"
#include "note.h"
#include "error.h"

#ifndef max
#   define max(a, b)	((a) > (b) ? (a) : (b))
#endif
#ifndef min
#   define min(a, b)	((a) < (b) ? (a) : (b))
#endif

#define UNMAP_PAGEVIEW	(1<<0)
#define UNMAP_CONTROL	(1<<1)
#define MAP_WIDGETS	(1<<3)


static void			layout_ghostview (
#if NeedFunctionPrototypes
#endif
);


#ifndef VMS

/*############################################################*/
/* translateTildeInPath */
/* Replaces tilde in string by user's home directory. */
/*############################################################*/

void
translateTildeInPath(path)
   char *path;
{
   char *pos;

   BEGINMESSAGE(translateTildeInPath)
   if ((pos=strchr(path,'~'))) {
      char *home;
      char tmp[GV_MAX_FILENAME_LENGTH];
      home=getenv("HOME");
      if (strlen(home)+strlen(path)-1 < GV_MAX_FILENAME_LENGTH-1) {
         *pos='\0'; pos++;
         strcpy(tmp,path);
         strcat(tmp,home);
         strcat(tmp,pos);
         strcpy(path,tmp);
      }
   }
   ENDMESSAGE(translateTildeInPath)
}
#endif /* end of not VMS */

/*############################################################*/
/* xorient */
/* Translate orientations defined by the enum in "ps.h" to
 * XtPageOrientations defined in "Ghostview.h".
*/
/*############################################################*/

static XtPageOrientation
xorient(psorient)
    int psorient;
{
    BEGINMESSAGE(xorient)
    switch (psorient) {
    case PORTRAIT: { ENDMESSAGE(xorient) return XtPageOrientationPortrait; }
    case LANDSCAPE:
#       ifdef USE_SWAP_LANDSCAPE
	   if (app_res.swap_landscape) {ENDMESSAGE(xorient) return XtPageOrientationSeascape;}
	   else {ENDMESSAGE(xorient) return XtPageOrientationLandscape; }
#       else
	   {ENDMESSAGE(xorient) return XtPageOrientationLandscape; }
#       endif
    }    
    ENDMESSAGE(xorient)
    return XtPageOrientationPortrait;
}


/*############################################################*/
/* show_widgets */
/*############################################################*/

static void
show_widgets(flag)
   int flag;
{
   static int u=0;
   INFMESSAGE(executing show_widgets)
   if (flag&UNMAP_CONTROL) {
       XtSetMappedWhenManaged(viewControl,False);
       u = u|UNMAP_CONTROL;
   }
   if (flag&UNMAP_PAGEVIEW) { 
      XtSetMappedWhenManaged(viewControl,False);
      u = u|UNMAP_PAGEVIEW;
   }
   if (flag&MAP_WIDGETS) {
      if (u&UNMAP_PAGEVIEW)      XtSetMappedWhenManaged(viewControl,True);
      if (u&UNMAP_CONTROL)       XtSetMappedWhenManaged(control,True);
      u=0;
   }
}


/*############################################################*/
/* check_file */
/* check if there is a new version of the file */
/* returns -1 if no filename or error in checking file */
/*          0 if no new version exists */
/*          1 if new version exists */
/*############################################################*/

int
check_file(mode)
int mode;
{
   int status=0;
   struct stat sbuf;
   char *tmpname;
   int  r = -1;
#  ifdef VMS
      char *pos;
#  endif

   BEGINMESSAGE(check_file)

   if (!filename) { INFMESSAGE(no filename) ENDMESSAGE(check_file) return(r); }
   if (!strcmp(filename,"-")) {
      INFMESSAGE(reading from stdin; nothing to update) ENDMESSAGE(check_file) return(0);
   }
   if (mode&CHECK_FILE_DATE) {
      INFMESSAGE(checking file date)
      status = stat(filename, &sbuf);
      if (!status && mtime != sbuf.st_mtime) {
         INFMESSAGE(file has changed)
         ENDMESSAGE(check_file)
         return(1);
      }
   }

   tmpname=filename;
   r=status;

#ifdef VMS
   if ( (mode&CHECK_FILE_VERSION) ||
        (status&&(mode&CHECK_FILE_DATE)&&strrchr(filename,';'))
      ) {
      tmpname = XtNewString(filename);
      pos = strrchr(tmpname,';'); /* strip the version */
      if (pos) *pos='\0';
      status = stat(tmpname, &sbuf);
      if (!status) {
         if (mtime != sbuf.st_mtime) {
            if (pos) { /* get the full specification of the new file*/
               unsigned long s;
               char newfile[256];
               struct dsc$descriptor_s sd;
               struct dsc$descriptor_s fd;
               unsigned long context = 0;       

               sd.dsc$w_length  = (unsigned short)strlen(tmpname);
               sd.dsc$b_dtype   = DSC$K_DTYPE_T;
               sd.dsc$b_class   = DSC$K_CLASS_S;
               sd.dsc$a_pointer = tmpname;
               fd.dsc$w_length  = sizeof(newfile)-1;
               fd.dsc$b_dtype   = DSC$K_DTYPE_T;
               fd.dsc$b_class   = DSC$K_CLASS_S;
               fd.dsc$a_pointer = newfile;
               s=lib$find_file(&sd,&fd,&context,0,0,0,0);
               if (s != RMS$_SUC) {
                  INFMESSAGE(not found) r = -1;
               } else {
                  newfile[fd.dsc$w_length]='\0';
                  pos = strchr(newfile,' ');
                  if (pos) *pos = '\0';
                  INFSMESSAGE(found new file:,newfile)
                  if (oldfilename) XtFree(oldfilename);
                  oldfilename = filename;
                  filename=XtNewString(newfile);
                  INFSMESSAGE(new file:,filename)
                  r = 1;
               }
               lib$find_file_end(&context);
             } else { /* file changed, but filename shows no version number */
                INFSMESSAGE(new file:,filename)
                r = 1;
             }
         } else {
           INFMESSAGE(no new version)
           r = 0;
         }
      } else r=status;
   }
#endif /* VMS */

   if (r<0) {
     char message[GV_MAXLENGTH]; 
     if (r != -2) {
        INFSMESSAGE(cannot access file:,tmpname)
        sprintf(message,"Unable to access file '%s'\n",tmpname);
     } else { /* privilege violation */
        INFSMESSAGE(user not authorized to access file:,tmpname)
        sprintf(message,"User is not authorized to access file '%s'\n",tmpname);
     }
     NotePopupShowMessage(message);
   }
   if (filename!=tmpname) XtFree(tmpname);
   ENDMESSAGE(check_file)
   return(r);
}

/*############################################################*/
/* show_currentPageMarker */
/*############################################################*/

static void
show_currentPageMarker(show)
   Boolean show;
{
   INFMESSAGE(executing set_page_mark)
   if (toc_text && (current_page >= 0)) {
      int marker = current_page*toc_entry_length + toc_entry_length-2;
      if (show) toc_text[marker] = '<';
      else      toc_text[marker] = ' ';
      XawTextInvalidate(toc, marker, marker+1);
   }
}

/*############################################################*/
/* render_page */
/* Start rendering a new page */
/*############################################################*/

void
render_page(gvw,number)
    Widget gvw;
    int number;
{
    int i;

    BEGINMESSAGE(render_page)

    if (!filename) { INFMESSAGE(no file) ENDMESSAGE(render_page) return; }

    INFIMESSAGE(displaying page,number)

    if (toc_text) {
       Boolean processflag;
       Boolean idleflag;
       Boolean noinputflag;
       INFMESSAGE(toc available)
       GhostviewState(gvw,&processflag,&idleflag,&noinputflag);
#      ifdef MESSAGES
          if (processflag) {INFMESSAGE(interpreter running)}
          else             {INFMESSAGE(no interpreter running)}
          if (idleflag)    {INFMESSAGE(widget is idle)}
          else             {INFMESSAGE(widget is busy)}
          if (noinputflag) {INFMESSAGE(interpreter has no input)}
          else             {INFMESSAGE(interpreter has input)}
#      endif
       /* Check first what the state of the ghostview widget is.
          Some documents show additional lines between
          the 'showpage' and the next '%%Page' comment.
          In this case the 'noinputflag' is 'False' but the additional
          lines are not really of significance (at least in no document I have
          encountered).
          So we ignore this flag and start from scratch only if the widget is
          busy or if no interpreter is running.
          Only if 'GV_RESTART_IF_CBUSY' is defined the noinputflag will be
          considered.
       */
#ifdef GV_RESTART_IF_BUSY /* ###jp### added 1.2.95 */
       if (processflag && idleflag && noinputflag) {
#else
       if (processflag && idleflag) {
#endif
          INFMESSAGE(displaying next page)
 	  GhostviewNextPage(gvw);
       } else {
          INFMESSAGE(starting new interpreter)
 	  GhostviewEnableInterpreter(gvw);
	  GhostviewSendPS(gvw, psfile, doc->beginprolog,
			  doc->lenprolog, False);
	  GhostviewSendPS(gvw, psfile, doc->beginsetup,
			  doc->lensetup, False);
       }
       if (doc->pageorder == DESCEND) i = (doc->numpages - 1) - current_page;
       else                           i = current_page;
       GhostviewSendPS(gvw, psfile, doc->pages[i].begin,doc->pages[i].len, False);
    } else {
       INFMESSAGE(no toc available)
       if (!GhostviewIsInterpreterRunning(gvw)) {
          INFMESSAGE(enabling interpreter for unstructured document)
          GhostviewEnableInterpreter(gvw);
       }
       else if (GhostviewIsInterpreterReady(gvw)) {
          INFMESSAGE(displaying page of unstructured document)
          GhostviewNextPage(gvw);
       }
       else {
          INFMESSAGE(interpreter running but not ready)
          XBell(XtDisplay(gvw), 0);
       }
    }

    if (gvw == page) {
       INFIMESSAGE(current page:,current_page)
       if (toc_text) {
          XtSetSensitive(prevEntry, current_page != 0);
          XtSetSensitive(nextEntry, current_page != (int)doc->numpages-1);
       }
       if (toc_text) {
          INFMESSAGE(marking current_page as current)
          XawTextUnsetSelection(toc);
          show_currentPageMarker(True);
          XawTextSetInsertionPoint(toc, current_page * toc_entry_length);
       }
    }

    ENDMESSAGE(render_page)
}

/*############################################################*/
/* show_page */
/* This routine is probably the heart of GV */
/* It receives requests from the various callbacks and actions, */
/* maps them onto three flags (need_layout, need_setup, need_render) */
/* and calls the necessary subroutines */
/*############################################################*/

void
show_page(number)
    int number;
{
    Bool need_layout = False;
    Bool need_setup  = False;
    Bool need_render = False;
    String error=NULL;
    int request=number;

    BEGINMESSAGE(show_page)
    INFIMESSAGE(received,request)

    if (!filename && request!=REQUEST_SETUP && request!=REQUEST_TOGGLE_RESIZE) {
       INFMESSAGE(no filename) ENDMESSAGE(show_page) return;
    }

    if ( /* check if file has changed */
         (request != REQUEST_NEW_FILE) &&
         (request != REQUEST_TOGGLE_RESIZE) &&
         (request != REQUEST_SETUP) 
       ) {
       int changed = check_file(CHECK_FILE_DATE);
       if (changed==1) {
          INFMESSAGE(file has changed; requesting new file)
          request = REQUEST_NEW_FILE;
       } else
       if (changed == -1) {
          INFMESSAGE(file is not accessible)
          ENDMESSAGE(show_page)
          return;
       }
    }

    if (!toc_text && (request==REQUEST_REDISPLAY)) {
       INFMESSAGE(request to redisplay non DSC file; changing to request for new file)
       request=REQUEST_NEW_FILE;
    }

    if (request >= NO_CURRENT_PAGE) {
	INFMESSAGE(request for new page)
        if (GhostviewIsBusy(page)) {
           INFMESSAGE(busy state)
           if (toc_text) {
              if (number >= (int)doc->numpages) number = (int)doc->numpages - 1;
              if (number < 0) number = 0;
              gv_pending_page_request=number;
              INFIMESSAGE(will remember,gv_pending_page_request)
           }
           ENDMESSAGE(show_page)
           return;
        }
        show_currentPageMarker(False);
	need_layout = need_setup = 
		set_new_orientation(number)|set_new_pagemedia(number);
	need_render = True;
    } else if (request<NO_CURRENT_PAGE) {
        INFIMESSAGE(analyzing,request)
	switch (request) {
	case REQUEST_TOGGLE_RESIZE:
		INFMESSAGE(### request for change of resize behaviour)
		number=current_page;
		need_layout = True;
		need_setup  = False;
		need_render = False;
		break;
	case REQUEST_REDISPLAY:
		INFMESSAGE(### request for redisplay)
		number=current_page;
		need_layout = False;
		need_setup  = False;
		need_render = True;
		break;
	case REQUEST_SETUP:
		INFMESSAGE(### request for setup)
		number=current_page;
		need_layout =	set_new_magstep()
				|set_new_orientation(number)
				|set_new_pagemedia(number);
                need_setup  = True;
		need_render = True;
		break;
	case REQUEST_NEW_MAGSTEP:
		INFMESSAGE(### request for new magstep)
		number=current_page;
		need_layout = need_setup = need_render = 
				set_new_magstep();
                if (!need_layout) {ENDMESSAGE(show_page) return;}
                break;
	case REQUEST_NEW_PAGEMEDIA:
		INFMESSAGE(### request for new pagemedia)
		number=current_page;
		need_layout = need_setup = need_render =
				set_new_pagemedia(number);
                if (!need_layout) {ENDMESSAGE(show_page) return;}
                break;
	case REQUEST_NEW_ORIENTATION:
		INFMESSAGE(### request for new orientation)
		number=current_page;
		need_layout = need_setup = need_render =
				set_new_orientation(number);
                if (!need_layout) {ENDMESSAGE(show_page) return;}
		break;
	case REQUEST_REOPEN:
	case REQUEST_NEW_FILE:
		INFMESSAGE(### request to open or reopen file)
		if (psfile) { fclose(psfile); psfile=NULL; }
		error = open_file(filename);
		if (error) {
		   NotePopupShowMessage(error);
		   XtFree(error);
		   INFMESSAGE(file cannot be openend) ENDMESSAGE(show_page) 
		   return;
		}
		if (request==REQUEST_REOPEN) {
		   INFMESSAGE(request to reopen file)
		   number=current_page;
		} else {
		   INFMESSAGE(request to open new file)
		   if (request==number) number=NO_CURRENT_PAGE;
		}
		need_layout = setup_ghostview();
		if (toc_text) {
		   if (number >= (int)doc->numpages) number = (int)doc->numpages - 1;
		   if (number < 0) number = 0;
                }
		need_layout = need_layout
				|set_new_orientation(number)
				|set_new_pagemedia(number);
                need_setup  = True;
                need_render = True;
		break;
	default:
		INFMESSAGE(### unknown request)
		fprintf(stderr,"  %s: Unknown request in show_page\n",gv_application_name);
		return;
	}
    }

    if (!psfile && need_render) {
       INFMESSAGE(no psfile; forcing setup and layout)
       need_setup=True;
       need_layout=True;
    }

#   ifdef MESSAGES
       if (need_layout) {INFMESSAGE(### need layout)} else {INFMESSAGE(### do not layout)}
       if (need_setup)  {INFMESSAGE(### need setup)}  else {INFMESSAGE(### do not setup)}
       if (need_render) {INFMESSAGE(### need render)} else {INFMESSAGE(### do not render)}
#   endif

/*
    if (need_setup && filename) {
       if (we have unmapped windows) GhostviewClearBackground(page);
    }
*/
    if (need_layout) layout_ghostview();

    if (need_setup)  GhostviewSetup(page);
    if (!filename) { need_render=False; INFMESSAGE(no filename; forcing no render) }
    if (toc_text) {
       if (number >= (int)doc->numpages) number = (int)doc->numpages - 1;
       if (number < 0) number = 0;
       current_page = number;
    }
    if (need_render)  render_page(page,number);
    /* note that this may cause rendering before the windows are mapped.
       Up to now I had no problems due to the retardation in the display
       caused by starting gs, so let's try ....
       If this fails somehow we have to move the next line
       a little up and activate the disabled GhostviewClearBackground 
       lines above.
    */
    if (need_layout) show_widgets(MAP_WIDGETS);

    gv_pending_page_request=NO_CURRENT_PAGE; /* eliminate any pending requests now */

    ENDMESSAGE(show_page)
}

/*############################################################*/
/* setup_ghostview */
/* This includes:
 *  scanning the PostScript file,
 *  setting the title and date labels,
 *  building the pagemedia menu,
 *  building the toc (table of contents)
 *  sensitizing the appropriate menu buttons,
 *  popping down and erasing the infotext popup.
 */
/*############################################################*/

static Boolean useful_page_labels;

Boolean
setup_ghostview()
{
    Arg args[20];
    Cardinal num_args;
    int oldtoc_entry_length;
    char *tocp;
    static String nothing = "";
    Pixmap bitmap;
    String label,buttonlabel;

    BEGINMESSAGE(setup_ghostview)
    /* Reset to a known state. */
    psfree(olddoc);
    olddoc = doc;
    doc = NULL;
    current_page = NO_CURRENT_PAGE;
    if (toc_text) XtFree(toc_text);
    oldtoc_entry_length = toc_entry_length;
    toc_text = NULL;

    /* Scan document and start setting things up */
    if (psfile) doc = psscan(psfile);

    if (show_title) {
       if (doc && doc->title) {
          buttonlabel = doc->title;
          label = doc->title;
          bitmap = app_res.document_bitmap;
       } 
       else {
          if (filename) {
#            ifdef VMS
                buttonlabel = strrchr(filename,']');
                if (!buttonlabel) buttonlabel = strrchr(filename,':');
                if (buttonlabel) buttonlabel++;
                else buttonlabel = filename;
                label = filename;
#            else
                buttonlabel = filename;
                label = filename;
#            endif
          }
          else { buttonlabel = "No File"; label = "No File"; }
          bitmap = None;
       }
       XtSetArg(args[0], XtNlabel, buttonlabel);
       XtSetValues(titlebutton, args, ONE);  
       if (titlemenu) XtDestroyWidget(titlemenu); 
       titlemenu = build_label_menu(titlebutton, "title", label, bitmap);
    }

    if (show_date) {
       if (doc && doc->date) {
          label = doc->date;
          bitmap = app_res.document_bitmap;
       } 
       else {
          if (psfile) { label = ctime(&mtime); } 
          else { label = "No Date"; }
          bitmap = None;
       }
       XtSetArg(args[0], XtNlabel, label);
       XtSetValues(datebutton, args, ONE);
       if (datemenu) XtDestroyWidget(datemenu);
       datemenu = build_label_menu(datebutton, "date", label, bitmap);   
    }

    if (app_res.force_orientation) gv_forced_orientation = gv_default_orientation;
    else                           gv_forced_orientation = XtPageOrientationUnspecified;
    gv_orientation = gv_default_orientation;

    if (app_res.force_pagemedia) gv_forced_pagemedia = gv_default_pagemedia;
    else                         gv_forced_pagemedia = -1;
    gv_pagemedia = gv_default_pagemedia;

    build_pagemedia_menu();

    /* Reset ghostscript and output messages popup */
#   ifdef VMS
    {
       Bool disable=False;
       if ( !doc || !olddoc ||
	    olddoc->beginprolog != doc->beginprolog ||
	    olddoc->endprolog != doc->endprolog ||
	    olddoc->beginsetup != doc->beginsetup ||
	    olddoc->endsetup != doc->endsetup
       ) disable=True;
       if (!disable) {
          char *fn = XtNewString(filename);
          char *ofn= XtNewString(oldfilename);
          char *pos;
          pos = strrchr(fn,';');  if (pos) *pos='\0';
          pos = strrchr(ofn,';'); if (pos) *pos='\0';
          if (strcmp(fn, ofn)) disable=True;
          XtFree(fn);
          XtFree(ofn);
       }
       if (disable==True) {   
	  GhostviewDisableInterpreter(page);
          cb_popdownInfoPopup((Widget)NULL,(XtPointer)NULL,(XtPointer)NULL);
          cb_resetInfoPopup((Widget)NULL,(XtPointer)NULL,(XtPointer)NULL);
       }
    }
#   else
    if (!doc || !olddoc ||
	strcmp(oldfilename, filename) ||
	olddoc->beginprolog != doc->beginprolog ||
	olddoc->endprolog != doc->endprolog ||
	olddoc->beginsetup != doc->beginsetup ||
	olddoc->endsetup != doc->endsetup) {
	GhostviewDisableInterpreter(page);
        cb_popdownInfoPopup((Widget)NULL,(XtPointer)NULL,(XtPointer)NULL);
        cb_resetInfoPopup((Widget)NULL,(XtPointer)NULL,(XtPointer)NULL);
    }
#   endif

    /* Build table of contents */
    if (doc && ((!doc->epsf && doc->numpages > 0) ||
		(doc->epsf && doc->numpages > 1))) {
	int maxlen = 0;
	int i, j;
	useful_page_labels = False;

	if (doc->numpages == 1) useful_page_labels = True;
	for (i = 1; i < doc->numpages; i++)
	    if ((useful_page_labels = (useful_page_labels ||
		    strcmp(doc->pages[i-1].label, doc->pages[i].label)))) break;
	if (useful_page_labels) {
	    for (i = 0; i < doc->numpages; i++) 
		maxlen = max(maxlen, strlen(doc->pages[i].label));
	} else {
	    double x;
	    x = doc->numpages;
	    maxlen = log10(x) + 1;
	}
	toc_entry_length = maxlen + 3;
	toc_length = doc->numpages * toc_entry_length - 1;
	toc_text = XtMalloc(toc_length + 2); /* include final NULL */

	for (i = 0, tocp = toc_text; i < doc->numpages;
	     i++, tocp += toc_entry_length) {
	    if (useful_page_labels) {
		if (doc->pageorder == DESCEND) {
		    j = (doc->numpages - 1) - i;
		} else {
		    j = i;
		}
		sprintf(tocp, " %*s \n", maxlen, doc->pages[j].label);
	    } else {
		sprintf(tocp, " %*d \n", maxlen, i+1);
	    }
	}
	toc_text[toc_length] = '\0';
							      	num_args = 0;
	XtSetArg(args[num_args], XtNfilename, NULL);      	num_args++;
	XtSetValues(page, args, num_args);
    } else {
	toc_length = 0;
	toc_entry_length = 3;
							      	num_args = 0;
	XtSetArg(args[num_args], XtNfilename, filename);      	num_args++;
	XtSetValues(page, args, num_args);
    }
								num_args = 0;
    XtSetArg(args[num_args], XtNlength, toc_length);		num_args++;
    if (toc_text) {
	XtSetArg(args[num_args], XtNstring, toc_text);		num_args++;
    } else {
	/* Text widget sometime blows up when given a NULL pointer */
	XtSetArg(args[num_args], XtNstring, nothing);		num_args++;
    }
    XtSetValues(toc, args, num_args);

    if (show_saveMarkedPages)	XtSetSensitive(w_saveMarkedPages,  (toc_text != NULL));
    if (show_toggleCurrentPage)	XtSetSensitive(w_toggleCurrentPage,(toc_text != NULL));
    if (show_toggleAllPages)	XtSetSensitive(w_toggleAllPages,   (toc_text != NULL));
    if (show_toggleEvenPages)	XtSetSensitive(w_toggleEvenPages,  (toc_text != NULL));
    if (show_toggleOddPages)	XtSetSensitive(w_toggleOddPages,   (toc_text != NULL));
    if (show_unmarkAllPages)	XtSetSensitive(w_unmarkAllPages,   (toc_text != NULL));
    if (show_saveMarkedPages)	XtSetSensitive(w_saveMarkedPages,  (toc_text != NULL));
    if (show_saveAllPages)	XtSetSensitive(w_saveAllPages,     (psfile   != NULL));
    if (show_printMarkedPages)	XtSetSensitive(w_printMarkedPages, (toc_text != NULL));
    if (show_printAllPages)	XtSetSensitive(w_printAllPages,    (psfile   != NULL));
    if (show_checkFile)		XtSetSensitive(w_checkFile,        (filename != NULL));
    if (show_updateFile)	XtSetSensitive(w_updateFile,       (filename != NULL));
    if (show_showThisPage)	XtSetSensitive(w_showThisPage,     (filename != NULL));

    XtSetSensitive(reopenEntry,      (psfile   != NULL));
    XtSetSensitive(printAllEntry,    (psfile   != NULL));
    XtSetSensitive(printMarkedEntry, (toc_text != NULL));
    XtSetSensitive(saveAllEntry,     (psfile   != NULL));
    XtSetSensitive(saveMarkedEntry,  (toc_text != NULL));
    XtSetSensitive(nextEntry,        (filename != NULL));
    XtSetSensitive(showEntry,        (filename != NULL));
    XtSetSensitive(prevEntry,        (toc_text != NULL));
    XtSetSensitive(centerEntry,      (filename != NULL));

    ENDMESSAGE(setup_ghostview)
    return oldtoc_entry_length != toc_entry_length;
}

/*############################################################*/
/* find_page */
/*############################################################*/

int
find_page(label)
    String label;
{
    int i, j;

    BEGINMESSAGE(find_page)
    if (label == NULL || doc == NULL) {ENDMESSAGE(find_page)return 0;}

    if (useful_page_labels) {
	for (i = 0; i < doc->numpages; i++) {
	    if (doc->pageorder == DESCEND) {
		j = (doc->numpages - 1) - i;
	    } else {
		j = i;
	    }
	    if (!strcmp(label, doc->pages[j].label)) {ENDMESSAGE(find_page)return i;}
	}
	ENDMESSAGE(find_page)
        return 0;
    } else {
	ENDMESSAGE(find_page)
	return atoi(label) - 1;
    }
    ENDMESSAGE(find_page)
}


Dimension view_width, view_height, view_border;
Dimension toc_width, toc_height, toc_border;
Dimension toc_leftMargin, toc_rightMargin;
Dimension control_width, control_height;
Dimension page_width, page_height;
XFontStruct *toc_font;
Dimension tocFrame_width,tocFrame_desiredWidth,tocFrame_height,tocFrame_hSpace,tocFrame_shadowWidth;


/*------------------------------------------------------------*/
/* layout_ghostview */
/*------------------------------------------------------------*/

static void
layout_ghostview()
{
   Arg       args[10];
   Cardinal  n;
   Dimension tocFrame_width;
   Dimension page_prefWidth, page_prefHeight;
   Dimension page_width, page_height;
   static Boolean firsttime=True;
   Boolean auto_resize;

   BEGINMESSAGE(layout_ghostview)

   if (!firsttime) {
      XtSetArg(args[0], XtNallowShellResize,&auto_resize);
      XtGetValues(toplevel, args,ONE);
      if (auto_resize != app_res.auto_resize) {
         INFMESSAGE(######## changing resize behaviour)
#        ifdef MESSAGES
            if (app_res.auto_resize) {INFMESSAGE(shell is allowed to resize)}
            else                     {INFMESSAGE(shell must not resize)}
#        endif
         XtSetArg(args[0], XtNallowShellResize,app_res.auto_resize);
         XtSetValues(toplevel, args,ONE);
         if (show_autoResize) {
            if (app_res.auto_resize) XtSetArg(args[0],XtNlabel,GV_AUTO_RESIZE_YES);
            else                     XtSetArg(args[0],XtNlabel,GV_AUTO_RESIZE_NO);
            XtSetValues(w_autoResize, args,ONE);
            if (app_res.auto_resize==False) {
               ENDMESSAGE(layout_ghostview) return;
            }
            INFIMESSAGE(setting tocFrame height:,TOC3D_INITIAL_HEIGHT)
            XtSetArg(args[0], XtNheight,TOC3D_INITIAL_HEIGHT);
            XtSetValues(tocFrame, args, ONE);
         }
      }
   }

   INFMESSAGE(#### retrieving dimensions)
   XtSetArg(args[0], XtNpreferredWidth, &page_prefWidth);
   XtSetArg(args[1], XtNpreferredHeight, &page_prefHeight);
   XtSetArg(args[2], XtNwidth, &page_width);
   XtSetArg(args[3], XtNheight, &page_height);
   XtGetValues(page, args, FOUR);
   INFIIMESSAGE(## preferred,page_prefWidth,page_prefHeight)
   INFIIMESSAGE(## actual,page_width,page_height)


   toc_width = (toc_font->max_bounds.width+toc_font->min_bounds.width)/2*(toc_entry_length-1)
               + toc_leftMargin+toc_rightMargin;
   XtSetArg(args[0], XtNwidth, &tocFrame_width);
   XtGetValues(tocFrame, args, ONE); INFIMESSAGE(tocFrame:,tocFrame_width)
   tocFrame_desiredWidth = toc_width+2*toc_border+2*tocFrame_hSpace+2*tocFrame_shadowWidth;
   IIMESSAGE(tocFrame_width,tocFrame_desiredWidth)

   if (tocFrame_width!=tocFrame_desiredWidth) {
      INFMESSAGE(#### setting tocFrame to its desired width)
      XtSetArg(args[0], XtNwidth, tocFrame_desiredWidth);
      XtSetValues(tocFrame, args, ONE);
   }

   if (page_prefWidth != page_width || page_prefHeight != page_height) {
      INFMESSAGE(#### setting ghostview widget size to its preferred size)
      XtSetArg(args[0], XtNwidth,           page_prefWidth);
      XtSetArg(args[1], XtNheight,          page_prefHeight);
      XtSetValues(page, args, TWO);
      if (app_res.auto_center) cb_centerPage(page,(XtPointer)NULL,(XtPointer)NULL);
   }

   if (firsttime) {
      if (show_autoResize) {
         if (app_res.auto_resize) XtSetArg(args[0], XtNlabel,GV_AUTO_RESIZE_YES);
         else                     XtSetArg(args[0], XtNlabel,GV_AUTO_RESIZE_NO);
                                  XtSetValues(w_autoResize, args,ONE);
      }
						          n=0;
      XtSetArg(args[n], XtNminWidth,GV_MINIMUM_SIZE);    n++;
      XtSetArg(args[n], XtNminHeight,GV_MINIMUM_SIZE);   n++;
      if (app_res.auto_resize==False) {
         INFMESSAGE(switching to No-Resize mode)
         XtSetArg(args[n], XtNallowShellResize,app_res.auto_resize); n++;
      }
      XtSetValues(toplevel, args,n);
      firsttime=False;
   }
 
  ENDMESSAGE(layout_ghostview)
}

/*############################################################*/
/* setup_layout_ghostview */
/*############################################################*/

void
setup_layout_ghostview()
{
    Arg args[5];
    Cardinal n;

    BEGINMESSAGE(setup_layout_ghostview )

    INFMESSAGE(#### getting tocFrame dimensions)
								n=0;
    XtSetArg(args[n], XtNhSpace,          &tocFrame_hSpace);	n++;
    XtSetArg(args[n], XtNshadowWidth,     &tocFrame_shadowWidth);n++;
    XtGetValues(tocFrame, args, n);

    INFMESSAGE(#### getting toc info)
								n=0;
    XtSetArg(args[n], XtNfont, &toc_font);			n++;
    XtSetArg(args[n], XtNleftMargin, &toc_leftMargin);		n++;
    XtSetArg(args[n], XtNrightMargin, &toc_rightMargin);	n++;
    XtSetArg(args[n], XtNborderWidth, &toc_border);		n++;
    XtGetValues(toc, args, n);

    ENDMESSAGE(setup_layout_ghostview)
}


/*############################################################*/
/* magnify */
/* Compute new dpi from magstep */
/*############################################################*/

void
magnify(dpi, magstep)
    float *dpi;
    int    magstep;
{
    BEGINMESSAGE(magnify)
    if (magstep<0) { while (magstep++) *dpi /= 1.2; }
    else           { while (magstep--) *dpi *= 1.2; }
    ENDMESSAGE(magnify)
}

/*############################################################*/
/* open_file */
/* Attempt to open file, return error message string on failure */
/*############################################################*/

String
open_file(name)
    String name;
{
    FILE *fp;
    struct stat sbuf;

    BEGINMESSAGE(open_file)
    if (*name == '\0') {	/* Null filename */
        INFMESSAGE(no filename) ENDMESSAGE(open_file)
	return(NULL);
    }
    if (strcmp(name, "-")) {
	if ((fp = fopen(name, "r")) == NULL) {
	    String buf;
            INFIMESSAGE(error number,errno)
            buf= open_fail_error(errno,GV_ERROR_OPEN_FAIL,name,False);
            INFSMESSAGE(error:, buf)
            ENDMESSAGE(open_file)
	    return buf;
	} else {
	    if (oldfilename) XtFree(oldfilename);
	    oldfilename = filename;
	    filename = XtNewString(name); 
	    if (psfile) fclose(psfile);
	    psfile = fp;
	    stat(filename, &sbuf);
	    mtime = sbuf.st_mtime;
	}
    } else {
	if (oldfilename) XtFree(oldfilename);
	oldfilename = filename;
	filename = XtNewString(name); 
	if (psfile) fclose(psfile);
	psfile = NULL;
    }
    ENDMESSAGE(open_file)
    return(NULL);
}

/*############################################################*/
/* save_file */
/* Attempt to save file, return error message string on failure */
/*############################################################*/

String
save_file(name,pagelist)
    String name;
    Boolean *pagelist;
{
    FILE *pswrite;
    int bytes;
    char buf[BUFSIZ];
    String errorstr=NULL;
    String closefail="Cannot close destination file %s\n";
    String openfail=GV_ERROR_OPEN_FAIL;
#   ifdef VMS
       char *pos;
#   endif

    BEGINMESSAGE(save_file)
    if (*name == '\0') {
       INFMESSAGE(no filename) ENDMESSAGE(save_file) return(NULL);
    }
#   ifdef VMS
       pos=strrchr(name,';');
       if ((pswrite = fopen(name, "w")) == NULL) {
          if (pos) {
             /* strip the version number and try again */
             char *tmp=XtNewString(name);
             pos=strrchr(tmp,';');
             *pos='\0';
             if ((pswrite = fopen(tmp, "w")) == NULL)
                errorstr = open_fail_error(errno,openfail,tmp,False);
             XtFree(tmp);
          } else {
             errorstr = open_fail_error(errno,openfail,name,False);
          }
       }
#   else
       if ((pswrite = fopen(name, "w")) == NULL)
          errorstr = open_fail_error(errno,openfail,name,False);
#   endif
    if (!errorstr) {
       if (pagelist) {
          pscopydoc(pswrite,pagelist);
       } else {
          FILE *psfile;
          if ((psfile=fopen(filename,"r"))==NULL) {
             errorstr = open_fail_error(errno,openfail,filename,False);
             if (fclose(pswrite)!=0) fprintf(stderr,closefail,name);
          }
          while ((bytes = read(fileno(psfile),buf,BUFSIZ))) bytes = write(fileno(pswrite), buf, bytes);
       }
       if (fclose(pswrite)!=0) { sprintf(buf,closefail,name); errorstr=XtNewString(buf); }
    }

    ENDMESSAGE(save_file)
    return(errorstr);
}

/*############################################################*/
/* print_file */
/* Attempt to print file.  Return error string on failure */
/*############################################################*/

String
print_file(print_command, pagelist)
    String print_command;
    Boolean *pagelist;
{
    FILE *printer;
    int bytes;
    char buf[BUFSIZ];
    int failed = 0;
    String ret_val=NULL;
    String closefail="Cannot close temporary file %s\n";
    String openfail=GV_ERROR_OPEN_FAIL;
    String printfail=GV_ERROR_PRINT_FAIL;
#   ifdef VMS
       char fnam[256];
#   else
       char *fnam;
#   endif

    BEGINMESSAGE(print_file)

#   ifdef VMS
       sprintf(fnam, "%sGV_%s.tmp",app_res.scratch_dir,tmpnam(NULL));
#   else
       fnam = tmpnam(NULL);
#   endif
    printer = fopen(fnam, "w");
    if (!printer) { ret_val=open_fail_error(errno,openfail,fnam,False); }
    else {
       if (pagelist) {
          pscopydoc(printer,pagelist);
       } else {
          FILE *psfile;
          if ((psfile=fopen(filename,"r"))==NULL) {
             ret_val = open_fail_error(errno,openfail,filename,False);
             if (fclose(printer)!=0) fprintf(stderr,closefail,fnam);
          }
          while ((bytes = read(fileno(psfile), buf, BUFSIZ))) bytes = write(fileno(printer), buf, bytes);
       }
       if (!ret_val) {
          sprintf(buf, "%s %s",print_command, fnam);
          INFSMESSAGE(printing:,buf)
          failed = (int) fclose(printer);
#         ifdef VMS
             if (!failed && ((system(buf)&7) != 1)) failed = 1;
#         else
             if (!failed && (system(buf) != 0))     failed = 1;
#         endif
          if (failed) {
             sprintf(buf,printfail,print_command); 
             ret_val = XtNewString(buf);
          }
       }
    }
    ENDMESSAGE(print_file)
    return(ret_val);
}

/*############################################################*/
/* pscopydoc */
/* Copy the headers, marked pages, and trailer to fp */
/*############################################################*/

/* length calculates string length at compile time */
/* can only be used with character constants */
#define length(a) (sizeof(a)-1)

void
pscopydoc(fp,pagelist)
    FILE *fp;
    Boolean *pagelist;
{
    FILE *psfile;
    char text[PSLINELENGTH];
    char *comment;
    Boolean pages_written = False;
    Boolean pages_atend = False;
    int pages = 0;
    int page = 1;
    int i, j;
    long here;

    BEGINMESSAGE(pscopydoc)
    psfile = fopen(filename, "r");

    here = doc->beginheader;
    while ((comment = pscopyuntil(psfile, fp, here,
				 doc->endheader, "%%Pages:"))) {
	here = ftell(psfile);
	if (pages_written || pages_atend) {
	    free(comment);
	    continue;
	}
	sscanf(comment+length("%%Pages:"), "%s", text);
	if (strcmp(text, "(atend)") == 0) {
	    fputs(comment, fp);
	    pages_atend = True;
	} else {
	    switch (sscanf(comment+length("%%Pages:"), "%*d %d", &i)) {
		case 1:
		    fprintf(fp, "%%%%Pages: %d %d\n", pages, i);
		    break;
		default:
		    fprintf(fp, "%%%%Pages: %d\n", pages);
		    break;
	    }
	    pages_written = True;
	}
	free(comment);
    }
    pscopy(psfile, fp, doc->beginpreview, doc->endpreview);
    pscopy(psfile, fp, doc->begindefaults, doc->enddefaults);
    pscopy(psfile, fp, doc->beginprolog, doc->endprolog);
    pscopy(psfile, fp, doc->beginsetup, doc->endsetup);

    for (i = 0; i < doc->numpages; i++) {
	if (doc->pageorder == DESCEND) 
	    j = (doc->numpages - 1) - i;
	else
	    j = i;
	if (pagelist[j]==True) {
	    comment = pscopyuntil(psfile, fp, doc->pages[i].begin,
				  doc->pages[i].end, "%%Page:");
	    fprintf(fp, "%%%%Page: %s %d\n",
		    doc->pages[i].label, page++);
	    free(comment);
	    pscopy(psfile, fp, -1, doc->pages[i].end);
	}
    }

    here = doc->begintrailer;
    while ((comment = pscopyuntil(psfile, fp, here,
				 doc->endtrailer, "%%Pages:"))) {
	here = ftell(psfile);
	if (pages_written) {
	    free(comment);
	    continue;
	}
	switch (sscanf(comment+length("%%Pages:"), "%*d %d", &i)) {
	    case 1:
		fprintf(fp, "%%%%Pages: %d %d\n", pages, i);
		break;
	    default:
		fprintf(fp, "%%%%Pages: %d\n", pages);
		break;
	}
	pages_written = True;
	free(comment);
    }
    fclose(psfile);

    ENDMESSAGE(pscopydoc)
}
#undef length

/*############################################################*/
/* set_new_magstep */
/*############################################################*/

Boolean
set_new_magstep()
{
    int new_magstep;
    Boolean changed = False;
    Arg args[20];
    Cardinal num_args;
    float xdpi, ydpi;

    BEGINMESSAGE(set_new_magstep)
    new_magstep = app_res.magstep;
    /* If magstep changed, stop interpreter and setup for new dpi. */
    if (new_magstep != current_magstep) {
	GhostviewDisableInterpreter(page);
	changed = True;
	xdpi = default_xdpi;
	ydpi = default_ydpi;
	magnify(&xdpi, new_magstep);
	magnify(&ydpi, new_magstep);
							num_args = 0;
	XtSetFloatArg(args[num_args], XtNxdpi, xdpi);	num_args++;
	XtSetFloatArg(args[num_args], XtNydpi, ydpi);	num_args++;
	XtSetValues(page, args, num_args);

	XtSetArg(args[0], XtNleftBitmap, None);
	XtSetValues(magstepEntry[current_magstep - app_res.minimum_magstep],
		    args, ONE);
	current_magstep = new_magstep;
    }
    XtSetArg(args[0], XtNleftBitmap, app_res.selected_bitmap);
    XtSetValues(magstepEntry[current_magstep - app_res.minimum_magstep],
		args, ONE);

    ENDMESSAGE(set_new_magstep)
    return changed;
}

/*############################################################*/
/* set_new_orientation */
/*############################################################*/

Boolean
set_new_orientation(number)
    int number;
{
    Boolean changed = False;
    Boolean from_doc = False;
    Arg args[1];
    XtPageOrientation new_orientation;

    BEGINMESSAGE(set_new_orientation)
    if (gv_forced_orientation != XtPageOrientationUnspecified) {
        new_orientation = gv_forced_orientation;
    } else {
	if (doc) {
	    if (toc_text && doc->pages[number].orientation != NONE) {
		new_orientation = xorient(doc->pages[number].orientation);
		from_doc = True;
	    } else if (doc->default_page_orientation != NONE) {
		new_orientation = xorient(doc->default_page_orientation);
		from_doc = True;
	    } else if (doc->orientation != NONE) {
		new_orientation = xorient(doc->orientation);
		from_doc = True;
	    } else {
		new_orientation        = gv_orientation;
		gv_default_orientation = gv_orientation;
	    }
	} else {
	    new_orientation        = gv_orientation;
            gv_default_orientation = gv_orientation;
	}
    }
    gv_orientation = gv_default_orientation;

    /* If orientation changed,
     * stop interpreter and setup for new orientation. */
    if (new_orientation != current_orientation) {
	GhostviewDisableInterpreter(page);
	changed = True;
	XtSetArg(args[0], XtNorientation, new_orientation);
	XtSetValues(page, args, ONE);
	XtSetArg(args[0], XtNleftBitmap, None);
	if (current_orientation == XtPageOrientationPortrait) 
	    XtSetValues(portraitEntry, args, ONE);
	else if (current_orientation == XtPageOrientationLandscape)
            XtSetValues(landscapeEntry, args, ONE);
        else if (current_orientation == XtPageOrientationUpsideDown)
            XtSetValues(upsidedownEntry, args, ONE);
        else if (current_orientation == XtPageOrientationSeascape)
            XtSetValues(seascapeEntry, args, ONE);
	current_orientation = new_orientation;
    }

    if (gv_forced_orientation!=XtPageOrientationUnspecified) {
	XtSetArg(args[0], XtNleftBitmap, app_res.forced_bitmap);
    } else if (from_doc) {
	XtSetArg(args[0], XtNleftBitmap, app_res.document_bitmap);
    } else {
	XtSetArg(args[0], XtNleftBitmap, app_res.selected_bitmap);
    }
    if (current_orientation == XtPageOrientationPortrait) 
	XtSetValues(portraitEntry, args, ONE);
    else if (current_orientation == XtPageOrientationLandscape)
	XtSetValues(landscapeEntry, args, ONE);
    else if (current_orientation == XtPageOrientationUpsideDown)
	XtSetValues(upsidedownEntry, args, ONE);
    else if (current_orientation == XtPageOrientationSeascape)
	XtSetValues(seascapeEntry, args, ONE);
    
    ENDMESSAGE(set_new_orientation)
    return changed;
}

/*############################################################*/
/* set_new_pagemedia */
/*############################################################*/

Boolean
set_new_pagemedia(number)
    int number;
{
    int new_pagemedia;
    int new_llx;
    int new_lly;
    int new_urx;
    int new_ury;
    Boolean changed = False;
    Boolean from_doc = False;
    Arg args[4];

    BEGINMESSAGE(set_new_pagemedia)
    if (gv_forced_pagemedia>=0) {
        new_pagemedia = gv_forced_pagemedia;
    } else {
	if (doc) {
	    if (toc_text && doc->pages[number].media != NULL) {
		new_pagemedia = doc->pages[number].media - doc->media;
		from_doc = True;
	    } else if (doc->default_page_media != NULL) {
		new_pagemedia = doc->default_page_media - doc->media;
		from_doc = True;
	    } else {
		new_pagemedia = gv_pagemedia;
                if (gv_pagemedia>=base_papersize) gv_default_pagemedia = gv_pagemedia;
	    }
	} else {
	    new_pagemedia = gv_pagemedia;
            if (gv_pagemedia>=base_papersize) gv_default_pagemedia = gv_pagemedia;
	}
    }
    gv_pagemedia = gv_default_pagemedia;

    /* If pagemedia changed, remove the old marker. */
    if (new_pagemedia != current_pagemedia) {
	XtSetArg(args[0], XtNleftBitmap, None);
	if (pagemediaEntry[current_pagemedia])
	    XtSetValues(pagemediaEntry[current_pagemedia], args, ONE);
	else
	    XtSetValues(pagemediaEntry[current_pagemedia-1], args, ONE);
	current_pagemedia = new_pagemedia;
    }

    if (gv_forced_pagemedia>=0) {
	XtSetArg(args[0], XtNleftBitmap, app_res.forced_bitmap);
    } else if (from_doc) {
	XtSetArg(args[0], XtNleftBitmap, app_res.document_bitmap);
    } else {
	XtSetArg(args[0], XtNleftBitmap, app_res.selected_bitmap);
    }
    if (pagemediaEntry[current_pagemedia])
	XtSetValues(pagemediaEntry[current_pagemedia], args, ONE);
    else
	XtSetValues(pagemediaEntry[current_pagemedia-1], args, ONE);

    /* Compute bounding box */
    if (gv_forced_pagemedia<0 &&
	doc && doc->epsf &&
	/* Ignore malformed bounding boxes */
	(doc->boundingbox[URX] > doc->boundingbox[LLX]) &&
	(doc->boundingbox[URY] > doc->boundingbox[LLY])) {
	new_llx = doc->boundingbox[LLX];
	new_lly = doc->boundingbox[LLY];
	new_urx = doc->boundingbox[URX];
	new_ury = doc->boundingbox[URY];
    } else {
	new_llx = new_lly = 0;
	if (new_pagemedia < base_papersize) {
	    new_urx = doc->media[new_pagemedia].width-1; /*#test#*/
	    new_ury = doc->media[new_pagemedia].height-1;/*#test#*/
	} else {
	    new_urx = papersizes[new_pagemedia-base_papersize].width-1;/*#test#*/
	    new_ury = papersizes[new_pagemedia-base_papersize].height-1;/*#test#*/
	}
    }

    /* If bounding box changed, setup for new size. */
    if ((new_llx != current_llx) || (new_lly != current_lly) ||
	(new_urx != current_urx) || (new_ury != current_ury)) {
        INFMESSAGE(bounding box changed)
        INFIIMESSAGE(lower left:,new_llx,new_lly)
        INFIIMESSAGE(upper right:,new_urx,new_ury)
	GhostviewDisableInterpreter(page);
	changed = True;
	current_llx = new_llx;
	current_lly = new_lly;
	current_urx = new_urx;
	current_ury = new_ury;
	XtSetArg(args[0], XtNllx, current_llx);
	XtSetArg(args[1], XtNlly, current_lly);
	XtSetArg(args[2], XtNurx, current_urx);
	XtSetArg(args[3], XtNury, current_ury);
	XtSetValues(page, args, FOUR);
    }

    ENDMESSAGE(set_new_pagemedia)
    return changed;
}

/*############################################################*/
/* same_document_media */
/*############################################################*/

static Boolean
same_document_media()
{
    int i;

    BEGINMESSAGE(same_document_media)
    if (olddoc == NULL && doc == NULL)     {ENDMESSAGE(same_document_media)return True;}
    if (olddoc == NULL || doc == NULL)     {ENDMESSAGE(same_document_media)return False;}
    if (olddoc->nummedia != doc->nummedia) {ENDMESSAGE(same_document_media)return False;}
    for (i = 0; i < doc->nummedia; i++) {
	if (strcmp(olddoc->media[i].name, doc->media[i].name)) {
           ENDMESSAGE(same_document_media)
           return False;
        }
    }
    ENDMESSAGE(same_document_media)
    return True;
}

/*############################################################*/
/* build_pagemedia_menu */
/*############################################################*/

void
build_pagemedia_menu()
{
    Arg args[20];
    Cardinal num_args;
    Widget w;
    int i;

    BEGINMESSAGE(build_pagemedia_menu)
    if (pagemediaMenu && same_document_media()) {
       ENDMESSAGE(build_pagemedia_menu)
       return;
    }
    if (pagemediaMenu) XtDestroyWidget(pagemediaMenu);

    pagemediaMenu = XtCreatePopupShell("documentMenu", simpleMenuWidgetClass,
				       pagemediaButton, NULL, ZERO);

    /* Build the Page Media menu */
    /* the Page media menu has two parts.
     *  - the document defined page medias
     *  - the standard page media defined from Adobe's PPD
     */
    base_papersize = 0;
    if (doc) base_papersize = doc->nummedia;
    for (i = 0; papersizes[i].name; i++) {}	/* Count the standard entries */
    i += base_papersize;
    pagemediaEntry = (Widget *) XtMalloc(i * sizeof(Widget));

    if (doc && doc->nummedia) {
	for (i = 0; i < doc->nummedia; i++) {
								num_args = 0;
	    XtSetArg(args[num_args], XtNleftMargin, 20);	num_args++;
	    pagemediaEntry[i] = XtCreateManagedWidget(doc->media[i].name,
				smeBSBObjectClass, pagemediaMenu,
				args, num_args);
	    XtAddCallback(pagemediaEntry[i], XtNcallback,
			  cb_setPagemedia, (XtPointer)i);
	}

							num_args = 0;
	w = XtCreateManagedWidget("line", smeLineObjectClass, pagemediaMenu,
				  args, num_args);
    }

    for (i = 0; papersizes[i].name; i++) {
	pagemediaEntry[i+base_papersize] = NULL;
	if (i > 0) {
	    /* Skip over same paper size with small imageable area */
	    if ((papersizes[i].width == papersizes[i-1].width) &&
		(papersizes[i].height == papersizes[i-1].height)) {
		continue;
	    }
	}
							num_args = 0;
	XtSetArg(args[num_args], XtNleftMargin, 20);	num_args++;
	pagemediaEntry[i+base_papersize] = XtCreateManagedWidget(
					    papersizes[i].name,
					    smeBSBObjectClass, pagemediaMenu,
					    args, num_args);
	XtAddCallback(pagemediaEntry[i+base_papersize], XtNcallback,
		      cb_setPagemedia, (XtPointer)(i+base_papersize));
    }
    ENDMESSAGE(build_pagemedia_menu)
}

/*############################################################*/
/* build_label_menu */
/*############################################################*/

Widget
build_label_menu(parent, name, label, bitmap)
    Widget parent;
    String name, label;
    Pixmap bitmap;
{
    Arg args[10];
    Cardinal num_args;
    Widget menu, entry;

    BEGINMESSAGE(build_label_menu)
								num_args = 0;
    menu = XtCreatePopupShell("labelMenu", simpleMenuWidgetClass,parent, args, num_args);
								num_args = 0;
    XtSetArg(args[num_args], XtNlabel, label);			num_args++;
    if (bitmap) {
       XtSetArg(args[num_args], XtNleftMargin, 20);		num_args++;
       XtSetArg(args[num_args], XtNleftBitmap, bitmap);		num_args++;
    }
    XtSetArg(args[num_args], XtNjustify, XtJustifyCenter);	num_args++;
    entry = XtCreateManagedWidget(name, smeBSBObjectClass,menu, args, num_args);
    ENDMESSAGE(build_label_menu)
    return menu;
}

/*############################################################*/
/* catch_Xerror */
/* Catch X errors die gracefully if one occurs */
/*############################################################*/

int
catch_Xerror(dpy, err)
    Display *dpy;
    XErrorEvent *err;
{
    BEGINMESSAGE(catch_Xerror)
    if (err->error_code == BadImplementation) {
	old_Xerror(dpy, err);
	return 0;
    }
    if (dying) return 0;
    dying = True;
    bomb = *err;
    XtDestroyWidget(toplevel);
    ENDMESSAGE(catch_Xerror)
    return 0;
}
